/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.rest.tools;

import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;
import org.apache.commons.lang3.StringUtils;
import com.openexchange.ajax.requesthandler.AJAXRequestData;
import com.openexchange.documentconverter.IDocumentConverter;
import com.openexchange.documentconverter.Properties;
import com.openexchange.exception.OXException;
import com.openexchange.java.Streams;
import com.openexchange.office.document.DocFileHelper;
import com.openexchange.office.tools.doc.StreamInfo;
import com.openexchange.office.tools.error.ErrorCode;
import com.openexchange.office.tools.error.ExceptionToErrorCode;
import com.openexchange.office.tools.files.StorageHelper;
import com.openexchange.office.tools.logging.v2.ELogLevel;
import com.openexchange.office.tools.logging.v2.impl.Logger;
import com.openexchange.office.tools.logging.v2.impl.Slf4JLogger;
import com.openexchange.office.tools.osgi.ServiceLookupRegistry;
import com.openexchange.tools.session.ServerSession;
import com.openexchange.user.User;

public class ConversionHelper {

    // ---------------------------------------------------------------
    private static final Logger LOG = Slf4JLogger.create(ConversionHelper.class);

    private ConversionHelper()
    {
        // nothing to do
    }

    // ---------------------------------------------------------------
    public static class ConversionData
    {
        private InputStream convertedInputStream;
        private String      convertedMimeType;
        private String      convertedExtension;
        private ErrorCode   conversionError;

        public ConversionData(InputStream inputStream, String mimeType, String extension, ErrorCode errorCode) {
            convertedInputStream = inputStream;
            convertedMimeType    = mimeType;
            convertedExtension   = extension;
            conversionError      = errorCode;
        }

        public InputStream getInputStream() { return convertedInputStream; }
        public String      getMimeType() { return convertedMimeType; }
        public String      getExtension() { return convertedExtension; }
        public ErrorCode   getErrorCode( ) { return conversionError; }
    }

    // ---------------------------------------------------------------
    public static ConversionData convertDocument(final ServerSession session, String fileId, String folderId, String version, String fileName, final Map<String, String> conversionFormat, String encryptionInfo) throws Exception {
        // conversion already sets the correct target document format
        final IDocumentConverter documentConverter = ServiceLookupRegistry.get().getService(IDocumentConverter.class);
        ErrorCode errorCode = ErrorCode.NO_ERROR;

        if (null != documentConverter) {
            InputStream documentInputStm = null;

            // OX Drive file must be loaded using Files API
            AJAXRequestData loadRequest = new AJAXRequestData();
            loadRequest.putParameter("id", fileId);
            loadRequest.putParameter("folder_id", folderId);
            loadRequest.putParameter("version", version);
            loadRequest.putParameter("fileName", fileName);

            final StorageHelper storageHelper = new StorageHelper(session, folderId);
            final StreamInfo streamInfo = DocFileHelper.getDocumentStream(null, session, loadRequest, storageHelper, encryptionInfo);
            documentInputStm = streamInfo.getDocumentStream();
            errorCode = streamInfo.getErrorCode();

            if (null != documentInputStm) {
                try {
                    return convertDocument(session, documentInputStm, fileName, conversionFormat, streamInfo.isDecrypted());
                } finally {
                    Streams.close(documentInputStm);
                    streamInfo.close();
                }
            }
        }

        return new ConversionData(null, null, null, errorCode);
    }

    // ---------------------------------------------------------------
    public static ConversionData convertDocument(
        final ServerSession session,
        final InputStream documentInputStm,
        final String fileName,
        final Map<String, String> conversionFormat,
        final boolean decrypted) {

        ErrorCode errorCode = ErrorCode.GENERAL_UNKNOWN_ERROR;

        if (null == documentInputStm)
            return new ConversionData(null, null, null, errorCode);

        final IDocumentConverter documentConverter = ServiceLookupRegistry.get().getService(IDocumentConverter.class);

        if (null != documentConverter) {
            try {
                final HashMap<String, Object> jobProperties = new HashMap<>(8);
                final HashMap<String, Object> resultProperties = new HashMap<>(8);
                final String filterShortName = conversionFormat.get(Properties.PROP_FILTER_SHORT_NAME);

                jobProperties.put(Properties.PROP_INPUT_STREAM, documentInputStm);
                jobProperties.put(Properties.PROP_FILTER_SHORT_NAME, filterShortName);
                jobProperties.put(Properties.PROP_LOCALE, getSessionLanguage(session));

                // this is a user request in every case
                jobProperties.put(Properties.PROP_USER_REQUEST, Boolean.TRUE);

                if (null != fileName) {
                    jobProperties.put(Properties.PROP_INFO_FILENAME, fileName);
                }

                if (decrypted) {
                    // set 'NO_CACHE=true' property for decrypted documents since we don't
                    // want to cache decrypted document conversion results for security reasons
                    jobProperties.put(Properties.PROP_NO_CACHE, Boolean.TRUE);
                }

                final InputStream inputStm = documentConverter.convert(filterShortName, jobProperties, resultProperties);

                if (null != inputStm) {
                    // set new mime type and extensionType
                    final String mimeType = conversionFormat.get(Properties.PROP_MIME_TYPE);
                    final String extensionType = conversionFormat.get(Properties.PROP_INPUT_TYPE);

                    return new ConversionData(inputStm, mimeType, extensionType, ErrorCode.NO_ERROR);
                }

                errorCode = ErrorCode.GENERAL_UNKNOWN_ERROR;
            } catch (final Exception e) {
                errorCode = ErrorCode.GENERAL_UNKNOWN_ERROR;
                if (e instanceof OXException)
                    errorCode = ExceptionToErrorCode.map((OXException)e, errorCode, false);
            }
        } else {
            LOG.forLevel(ELogLevel.E_WARNING).withMessage("RT connection: cannot retrieve document converter service!").log();
            errorCode = ErrorCode.GENERAL_SERVER_COMPONENT_NOT_WORKING_ERROR;
        }

        return new ConversionData(null, null, null, errorCode);
    }

    // ---------------------------------------------------------------
    public static String getSessionLanguage(ServerSession session) {
        final User sessionUser = session.getUser();
        String prefLanguage = null;

        // user language
        if (null != sessionUser) {
            prefLanguage = sessionUser.getPreferredLanguage();
        }

        return StringUtils.isBlank(prefLanguage) ? "en_US" : prefLanguage;
    }

}
