/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.rest.operations;

import java.io.InputStream;
import java.nio.charset.Charset;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.openexchange.filemanagement.ManagedFile;
import com.openexchange.filemanagement.ManagedFileManagement;
import com.openexchange.office.tools.osgi.ServiceLookupRegistry;
import com.openexchange.timer.TimerService;

public class OperationDataCache implements Runnable {

    // ---------------------------------------------------------------
	private static final Logger log = LoggerFactory.getLogger(OperationDataCache.class);

    //-------------------------------------------------------------------------
    private final static int  MAX_OPERATIONSCACHE_ENTRIES = 32;

    //-------------------------------------------------------------------------
    private final static long GC_FREQUENCY                = 60000;

    //-------------------------------------------------------------------------
    private final static long MAX_IDLE_TIMESPAN           = 300000;

    //-------------------------------------------------------------------------
    private static class Holder {
        static final OperationDataCache INSTANCE = new OperationDataCache(MAX_OPERATIONSCACHE_ENTRIES);
    }

    //-------------------------------------------------------------------------
    private static class OperationDataHolder {
        public String      key;
        public String      hash;
        public long        lastAccessTime;
        public ManagedFile htmlDocManagedFile;
        public ManagedFile opsDocManagedFile;
        public ManagedFile previewOpsManagedFile;

        // ---------------------------------------------------------------
        public OperationDataHolder(String key, String hash, ManagedFile htmlDocManagedFile, ManagedFile previewOpsManagedFile, ManagedFile opsDocManagedFile) {
            this.key = key;
            this.hash = hash;
            this.htmlDocManagedFile = htmlDocManagedFile;
            this.opsDocManagedFile = opsDocManagedFile;
            this.previewOpsManagedFile = previewOpsManagedFile;
            this.lastAccessTime = System.currentTimeMillis();
        }

        // ---------------------------------------------------------------
        public OperationDataHolder(final OperationDataHolder aHolder) {
            key = aHolder.key;
            hash = aHolder.hash;
            htmlDocManagedFile = aHolder.htmlDocManagedFile;
            opsDocManagedFile = aHolder.opsDocManagedFile;
            previewOpsManagedFile = aHolder.previewOpsManagedFile;
            lastAccessTime = aHolder.lastAccessTime;
        }

        // ---------------------------------------------------------------
        public void touch() {
            lastAccessTime = System.currentTimeMillis();

            if (null != opsDocManagedFile)
                opsDocManagedFile.touch();
            if (null != previewOpsManagedFile)
                previewOpsManagedFile.touch();
            if (null != htmlDocManagedFile)
                htmlDocManagedFile.touch();
        }
    }

    // ---------------------------------------------------------------
    private final Map<String, OperationDataHolder> cacheOpsData = new HashMap<>();

    // ---------------------------------------------------------------
    private final ManagedFileManagement managedFileService;

    // ---------------------------------------------------------------
    private int maxEntries;

    // ---------------------------------------------------------------
    private OperationDataCache(int maxEntries) {
        this.maxEntries = maxEntries;

        final TimerService timerService = ServiceLookupRegistry.get().getService(TimerService.class);
        if (null != timerService) {
            timerService.scheduleAtFixedRate(this, GC_FREQUENCY, GC_FREQUENCY);
        } else {
            log.error("OperationDataCache timer service not available - gc function won't work");
        }

        managedFileService = ServiceLookupRegistry.get().getService(ManagedFileManagement.class);
        if (null == managedFileService) {
            log.error("OperationDataCache cannot access mandatory ManagedFileManagement service - cache won't work!");
        }
    }

    //-------------------------------------------------------------------------
    public static OperationDataCache getInstance()
    {
        return Holder.INSTANCE;
    }

    // ---------------------------------------------------------------
    public boolean isEmpty() {
        synchronized (cacheOpsData) {
            return cacheOpsData.isEmpty();
        }
    }

    // ---------------------------------------------------------------
    public boolean hasEntryAndTouch(String fileId, String version, int context) {
        final String key = getKeyFromProps(fileId, version, context);

        OperationDataHolder cacheEntry = null;
        synchronized (cacheOpsData) {
            cacheEntry = cacheOpsData.get(key);

            if (null != cacheEntry)
                cacheEntry.touch();
        }

        return (null != cacheEntry);
    }

    // ---------------------------------------------------------------
    public OperationData get(String fileId, String version, int context) {
        final String key = getKeyFromProps(fileId, version, context);

        OperationDataHolder aHolder = null;
        synchronized (cacheOpsData) {
            final OperationDataHolder cacheEntry = cacheOpsData.get(key);

            if (null != cacheEntry) {
                cacheEntry.touch();
                aHolder = new OperationDataHolder(cacheEntry);
            }
        }

        if (null != aHolder) {
            try {
                return this.createOperationData(aHolder);
            } catch (final Exception e) {
                removeEntryIfHashIdentical(key, aHolder.hash);

                log.debug("OperationDataCache caught exception trying to convert managed file data to cache entry - managed file expired?", e);
            }
        }

        return null;
    }

    // ---------------------------------------------------------------
    public void put(String key, final OperationData aOpsData) {
        OperationDataHolder holder = null;

        try {
            holder = this.createOperationDataHolder(aOpsData);
        } catch (Exception e) {
            log.warn("OperationDataCache caught exception trying to create entry with managed files", e);
        }

        OperationDataHolder oldEntry = null;

        if (null != holder) {
            synchronized (cacheOpsData) {
                if (cacheOpsData.size() >= maxEntries)
                    removeOldestEntry();
                oldEntry = cacheOpsData.put(key, holder);
            }
        }

        if (null != oldEntry) {
            destroyHolder(oldEntry);
        }
    }

    // ---------------------------------------------------------------
    public static String getKeyFromProps(String fileId, String version, int context) {
        final StringBuilder tmp = new StringBuilder(fileId);
        tmp.append("-").append(version).append("-").append(context);
        return tmp.toString();
    }

    // ---------------------------------------------------------------
    @Override
    public void run() {
        removeExpiredEntries();
    }

    // ---------------------------------------------------------------
    private void removeEntryIfHashIdentical(String key, String hash) {
        synchronized (cacheOpsData) {
            final OperationDataHolder aHolder = cacheOpsData.get(key);
            // check for possible null values
            if ((aHolder != null) && StringUtils.equals(aHolder.hash, hash)) {
                removeEntry(key);
            }
        }
    }

    // ---------------------------------------------------------------
    private void removeOldestEntry() {
        long   oldestAccessTime = System.currentTimeMillis();
        String oldestEntryKey   = null;

        synchronized (cacheOpsData) {
            final Collection<OperationDataHolder> entries = cacheOpsData.values();
            for (final OperationDataHolder data : entries) {
                if (data.lastAccessTime < oldestAccessTime) {
                    oldestAccessTime = data.lastAccessTime;
                    oldestEntryKey   = data.key;
                }
            }

            if (StringUtils.isNotEmpty(oldestEntryKey)) {
                removeEntry(oldestEntryKey);
            }
        }
    }

    // ---------------------------------------------------------------
    private void removeExpiredEntries() {
        long currentTime = System.currentTimeMillis();

        synchronized (cacheOpsData) {
            final Collection<OperationDataHolder> entries = cacheOpsData.values();

            if (!entries.isEmpty()) {
                final Set<String> expiredEntries = entries.stream()
                                                          .filter(o -> expiredEntry(o, currentTime))
                                                          .map(o -> o.key)
                                                          .collect(Collectors.toSet());
                removeEntries(expiredEntries);
            }
        }
    }

    // ---------------------------------------------------------------
    private boolean expiredEntry(final OperationDataHolder data, long currentTime) {
        return (Math.abs(currentTime - data.lastAccessTime) > MAX_IDLE_TIMESPAN);
    }

    // ---------------------------------------------------------------
    private void removeEntries(final Set<String> keys) {
        log.debug("OperationDataCache removes expired entries " + keys.toString());

        for (final String key : keys) {
            removeEntry(key);
        }
    }

    // ---------------------------------------------------------------
    private void removeEntry(final String key) {
        final OperationDataHolder aHolder = cacheOpsData.remove(key);

        if (null != aHolder)
            destroyHolder(aHolder);
    }

    // ---------------------------------------------------------------
    private void destroyHolder(final OperationDataHolder dataHolder) {
        try {
            if (null != dataHolder.opsDocManagedFile)
                dataHolder.opsDocManagedFile.delete();
            if (null != dataHolder.htmlDocManagedFile)
                dataHolder.htmlDocManagedFile.delete();
            if (null != dataHolder.previewOpsManagedFile)
                dataHolder.previewOpsManagedFile.delete();
        } catch (Exception e) {
            // nothing to do - managed files are deleted automatically on timeout
        }
    }

    // ---------------------------------------------------------------
    private OperationData createOperationData(final OperationDataHolder dataHolder) throws Exception {
        JSONArray  jsonOpsArray    = null;
        JSONObject jsonPreviewData = null;
        String     htmlString      = null;

        try (final InputStream opsDataStream = dataHolder.opsDocManagedFile.getInputStream()) {
            byte[] aJsonBytes = IOUtils.toByteArray(opsDataStream);
            final String jsonArrayOpsString = new String(aJsonBytes, "UTF-8");
            aJsonBytes = null;
            jsonOpsArray = new JSONArray(jsonArrayOpsString);
        } finally {
            // nothing to do
        }

        try (final InputStream previewDataStream = dataHolder.previewOpsManagedFile.getInputStream()) {
            byte[] aJsonBytes = IOUtils.toByteArray(previewDataStream);
            final String jsonPreviewDataString = new String(aJsonBytes, "UTF-8");
            aJsonBytes = null;
            jsonPreviewData = new JSONObject(jsonPreviewDataString);
        } finally {
            // nothing to do
        }

        if (null != dataHolder.htmlDocManagedFile) {
            try (final InputStream htmlStream = dataHolder.htmlDocManagedFile.getInputStream()) {
                byte[] aJsonBytes = IOUtils.toByteArray(htmlStream);
                htmlString = new String(aJsonBytes, "UTF-8");
            } finally {
                // nothing to do
            }
        }

        return new OperationData(dataHolder.key, dataHolder.hash, htmlString, jsonPreviewData, jsonOpsArray);
    }

    // ---------------------------------------------------------------
    private OperationDataHolder createOperationDataHolder(final OperationData aOpsData) throws Exception {
        byte[] aOpsDocData = aOpsData.getOperations().toString().getBytes(Charset.forName("UTF-8"));
        final ManagedFile aOpsDataManagedFile = managedFileService.createManagedFile(aOpsDocData);
        aOpsDocData = null;

        ManagedFile aPreviewOpsManagedFile = null;
        if (null != aOpsData.getPreviewData()) {
            final byte[] aPreviewDocBytes = aOpsData.getPreviewData().toString().getBytes(Charset.forName("UTF-8"));
            aPreviewOpsManagedFile = managedFileService.createManagedFile(aPreviewDocBytes);
        }

        ManagedFile aHtmlDocManagedFile = null;
        if (null != aOpsData.getHtmlDoc()) {
            final byte[] aHtmlDocBytes = aOpsData.getHtmlDoc().getBytes(Charset.forName("UTF-8"));
            aHtmlDocManagedFile = managedFileService.createManagedFile(aHtmlDocBytes);
        }

        return new OperationDataHolder(aOpsData.getKey(), aOpsData.getHash(), aHtmlDocManagedFile, aPreviewOpsManagedFile, aOpsDataManagedFile);
    }

}
