/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.rest;

import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.commons.lang3.BooleanUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.ajax.requesthandler.AJAXRequestData;
import com.openexchange.ajax.requesthandler.AJAXRequestResult;
import com.openexchange.exception.OXException;
import com.openexchange.office.tools.config.ConfigurationHelper;
import com.openexchange.tools.session.ServerSession;


public class LogPerformanceDataAction extends DocumentRESTAction {
    @SuppressWarnings("deprecation")
    private static final Log LOG = com.openexchange.log.Log.loggerFor(LogPerformanceDataAction.class);

    private static AtomicReference<Boolean> logPerformanceDataEnabled = new AtomicReference<>(null);
    private static AtomicReference<String> logPerformanceDataPath = new AtomicReference<>(null);

    private LogPerformanceDataWriter aPerformanceDataWriter;
    private AtomicBoolean aPerformanceDataWriterInit = new AtomicBoolean(false);

    public LogPerformanceDataAction(final LogPerformanceDataWriter aDataWriter) {
        super();

        aPerformanceDataWriter = aDataWriter;
    }

    @Override
    public AJAXRequestResult perform(AJAXRequestData requestData, ServerSession session) throws OXException {
        if ((null == session) || (session.isAnonymous())) {
            return getAJAXRequestResultWithStatusCode(401);
        }

        final String performanceData = requestData.getParameter("performanceData");
        if (StringUtils.isEmpty(performanceData)) {
            return getAJAXRequestResultWithStatusCode(400);
        }

        Boolean enabled = logPerformanceDataEnabled.get();
        String  loggingFilePath = logPerformanceDataPath.get();

        if (null == enabled) {
            try {
                enabled = ConfigurationHelper.getBooleanOfficeConfigurationValue(null, session, "module/logPerformanceData", false);
                if (enabled) {
                    loggingFilePath = ConfigurationHelper.getStringOfficeConfigurationValue(
                                          null, session, "//module/performanceDataLogPath", null);

                    if (StringUtils.isNotEmpty(loggingFilePath)) {
                        logPerformanceDataPath.compareAndSet(null, loggingFilePath);
                        logPerformanceDataEnabled.compareAndSet(null, enabled);
                    }
                }
            } catch (Exception e) {
                // this is an optional feature therefore we just log the problem
                LOG.warn("Exception caught while tyring to determine performance logging properties", e);

                logPerformanceDataEnabled.compareAndSet(null, false);
                logPerformanceDataPath.compareAndSet(null, "");
            }
        }

        if (BooleanUtils.isTrue(enabled) && StringUtils.isNotEmpty(loggingFilePath)) {
            try {
                if (aPerformanceDataWriterInit.compareAndSet(false, true)) {
                    aPerformanceDataWriter.init(loggingFilePath);
                }

                // don't accept data which is too long
                if (performanceData.length() >= (LogPerformanceDataWriter.MAX_PERFORMANCE_LOG_FILESIZE / 10)) {
                    return getAJAXRequestResultWithStatusCode(400);
                }

                final JSONObject aJSONPerformanceData = new JSONObject(performanceData);
                writePerformanceData(aJSONPerformanceData);
                return getAJAXRequestResultWithStatusCode(200);
            }
            catch (final JSONException e) {
                return getAJAXRequestResultWithStatusCode(400);
            }
            catch (final Exception e) {
                LOG.warn("Exception caught while performance logging", e);
                return getAJAXRequestResultWithStatusCode(500);
            }
        }

        return getAJAXRequestResultWithStatusCode(410);
    }

    private void writePerformanceData(final JSONObject jsonPerformanceData) {
        aPerformanceDataWriter.add(jsonPerformanceData);
    }

    /**
     * Provide a AJAXRequestResult with a set status code.
     *
     * @param statusCode the status code to be set at the resulting AJAXRequestResult instance
     * @return the AJAXRequestResult with the provided status code set
     */
    private AJAXRequestResult getAJAXRequestResultWithStatusCode(int statusCode) {
        final AJAXRequestResult requestResult = new AJAXRequestResult();
        requestResult.setHttpStatusCode(statusCode);
        return requestResult;
    }

}
