/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.rest;

import static org.apache.commons.lang3.StringUtils.isEmpty;
import static org.apache.commons.lang3.StringUtils.isNotEmpty;
import java.awt.Dimension;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.security.SecureRandom;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import javax.crypto.Cipher;
import javax.crypto.CipherInputStream;
import javax.crypto.CipherOutputStream;
import javax.crypto.KeyGenerator;
import javax.crypto.SecretKey;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.common.base.Throwables;
import com.openexchange.ajax.requesthandler.AJAXRequestData;
import com.openexchange.ajax.requesthandler.AJAXRequestResult;
import com.openexchange.ajax.requesthandler.DispatcherNotes;
import com.openexchange.annotation.NonNull;
import com.openexchange.config.ConfigurationService;
import com.openexchange.documentconverter.IDocumentConverter;
import com.openexchange.documentconverter.Properties;
import com.openexchange.exception.ExceptionUtils;
import com.openexchange.exception.OXException;
import com.openexchange.imageconverter.api.ElementLock.LockMode;
import com.openexchange.imageconverter.api.ElementLocker;
import com.openexchange.office.document.DocFileHelper;
import com.openexchange.office.imagemgr.IResourceManager;
import com.openexchange.office.imagemgr.Resource;
import com.openexchange.office.rest.tools.EncryptionInfo;
import com.openexchange.office.session.SessionService;
import com.openexchange.office.tools.doc.StreamInfo;
import com.openexchange.office.tools.files.FileHelper;
import com.openexchange.office.tools.files.StorageHelper;
import com.openexchange.office.tools.osgi.ServiceLookupRegistry;
import com.openexchange.office.tools.user.AuthorizationCache;
import com.openexchange.tools.session.ServerSession;

/**
 * {@link GetFileAction}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 */
/*
 * MH/KA compile fix for buildsystem
 *
 * @Action(method = RequestMethod.GET, name = "getfile", description =
 * "Get the whole document as PDF file or get the substream of an infostore file in its naive format.", parameters = {
 *
 * @Parameter(name = "session", description = "A session ID previously obtained from the login module."),
 *
 * @Parameter(name = "id", description = "Object ID of the requested infoitem."),
 *
 * @Parameter(name = "folder_id", description = "Folder ID of the requested infoitem."),
 *
 * @Parameter(name = "uid", description = "The unique id of the client application."),
 *
 * @Parameter(name = "version", optional=true, description =
 * "If present, the infoitem data descresourceManagerribes the given version. Otherwise the current version is returned."),
 *
 * @Parameter(name = "filter_format", optional=true, description =
 * "If this value is set to 'pdf', the whole document is as converted PDF file, in all other cases, the content of the document's substream, described by the parameter 'fragment, is returned as unmodified file."
 * ),
 *
 * @Parameter(name = "fragment", optional=true, description =
 * "If this value is set and the filter_format is not set or not set to 'pdf', this parameter describes the substream name of the document internal file to be returned."
 * ),
 *
 * @Parameter(name = "filename", optional=true, description =
 * "If present, this parameter contains the name of the infostore item to be used as initial part of the filename for the returned file, other the filename is set to 'file'."
 * ), }, responseDescription = "Response with timestamp: The content of the documents substream in its native format, or converted to PNG in
 * the case of EMF, WMF and SVM graphic file formats)
 */
@DispatcherNotes(defaultFormat = "file", allowPublicSession = true)
public class GetFileAction extends DocumentRESTAction {
    /**
     * {@link Cache}
     *
     * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
     * @since v7.10.5
     */
    private class Cache {

        /**
         * CACHE_ROOT_DIRECTORY_NAME (subdirectory of given tmp directory)
         */
        final private static String CACHE_ROOT_DIRECTORY_NAME = ".OGFC";

        /**
         * CLEAR_ALL_ENTRIES
         */
        final private static long CLEAR_ALL_ENTRIES = 0L;

        /**
         * use AES encyption
         */
        final private static String ENCRYPTION_ALGORITHM = "AES";

        /**
         * {@link Entry}
         *
         * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
         * @since v7.10.5
         */
        private class Entry {

            /**
             * Initializes a new {@link Entry}.
             * @param rootDir
             * @param documentStm
             */
            public Entry(@NonNull final File rootDir, @NonNull InputStream documentStm) throws Exception {
                final long startTimeMillis = System.currentTimeMillis();

                FileUtils.forceMkdir(m_entryDir = new File(rootDir, UUID.randomUUID().toString()));

                if (m_entryDir.canWrite()) {
                    // create and init en/decryption KeyGenerator with current system time
                    final KeyGenerator encryptionKeyGenerator = KeyGenerator.getInstance(ENCRYPTION_ALGORITHM);

                    encryptionKeyGenerator.init(new SecureRandom(Long.toString(startTimeMillis).getBytes()));

                    // create en/decrypt ciphers
                    final SecretKey secretKey = encryptionKeyGenerator.generateKey();

                    (m_encryptCipher = Cipher.getInstance(ENCRYPTION_ALGORITHM)).init(Cipher.ENCRYPT_MODE, secretKey);
                    (m_decryptCipher = Cipher.getInstance(ENCRYPTION_ALGORITHM)).init(Cipher.DECRYPT_MODE, secretKey);

                    // extract resources and write to tmp folder using the encryption cipher
                    implExtractResources(documentStm);

                    // update timestamp after extraction of all resources
                    touch();

                    if (log.isTraceEnabled()) {
                        log.trace("GetFile cache creating new entry {} took {}ms to cache {} resources",
                            m_entryDir.getAbsolutePath(),
                            System.currentTimeMillis() - startTimeMillis,
                            m_entryNames.size());
                    }
                } else {
                    throw new IllegalArgumentException("GetFile cache entry is not writable: " + m_entryDir.getAbsolutePath());
                }
            }

            /**
             *
             */
            public void touch() {
                m_timestampMillis = System.currentTimeMillis();
            }

            /**
             * @return
             */
            public long getTimestampMillis() {
                return m_timestampMillis;
            }

            /**
             *
             */
            public synchronized void clear() {
                FileUtils.deleteQuietly(m_entryDir);
                m_timestampMillis = 0;
            }

            /**
             * @param resourceName
             * @return
             */
            public synchronized byte[] getResourceBuffer(final String resourceName, final Dimension imageExtents, final String[] mimeType) {
                if (null != resourceName) {
                    String entryName = resourceName;

                    // some resources might be specified by an uid fragment only =>
                    // try to find a possible match for this within all entries
                    if (!m_entryNames.contains(entryName)) {
                        final String uidResourceName = FileHelper.getBaseName(resourceName);

                        if ((null != uidResourceName) && uidResourceName.startsWith(Resource.RESOURCE_UID_PREFIX) &&
                            (uidResourceName.length() > Resource.RESOURCE_UID_PREFIX.length())) {

                            for (final String curEntryName : m_entryNames) {
                                if (curEntryName.contains(uidResourceName)) {
                                    // use first entry whose uid fragment is contained in entry name
                                    entryName = curEntryName;
                                    break;
                                }
                            }
                        }
                    }

                    try {
                        final File resourceFile = new File(m_entryDir, entryName);

                        if (resourceFile.canRead()) {
                            final String fileExtension = FilenameUtils.getExtension(entryName).trim().toLowerCase();

                            if (ArrayUtils.isNotEmpty(mimeType) && IMAGE_EXTENSION_TO_MIMETYPE_MAP.containsKey(fileExtension)) {
                                mimeType[0] = IMAGE_EXTENSION_TO_MIMETYPE_MAP.get(fileExtension);
                            }

                            // read from encrypted file with defined decryption cipher
                            try (final InputStream inputStm = new CipherInputStream(FileUtils.openInputStream(resourceFile), m_decryptCipher)) {
                                if (null != inputStm) {
                                    if (needsConversion(entryName, fileExtension)) {
                                        return convert(inputStm, entryName, fileExtension, imageExtents, mimeType);
                                    }

                                    return IOUtils.toByteArray(inputStm);
                                }
                            }
                        }
                    } catch (Exception e) {
                        log.error("GetFile cache caught exception in getResourceBuffer for " + resourceName, e);
                    }
                }

                return null;
            }

            // - Implementation ----------------------------------------------------

            /**
             * @param documentStm
             * @return The number of all extracted resources
             */
            void implExtractResources(@NonNull InputStream documentStm) throws Exception {
                try (final ZipInputStream zipInputStm = new ZipInputStream(documentStm)) {
                    for (ZipEntry zipEntry = null; null != (zipEntry = zipInputStm.getNextEntry());) {
                        try {
                            final String curEntryName = zipEntry.getName();

                            if (curEntryName.equals(DocFileHelper.OX_RESCUEDOCUMENT_DOCUMENTSTREAM_NAME)) {
                                // check for resources contained within the original rescue document
                                byte[] entryBuffer = IOUtils.toByteArray(zipInputStm);

                                if (null != entryBuffer) {
                                    try (final InputStream entryInputStream = new ByteArrayInputStream(entryBuffer)) {
                                        implExtractResources(entryInputStream);
                                    }
                                }
                            } else if (!zipEntry.isDirectory()) {
                                try {
                                    final String fileExtension = FilenameUtils.getExtension(curEntryName).trim().toLowerCase();

                                    // extract image to cache entry dir as encrypted file
                                    if (IMAGE_EXTENSION_TO_MIMETYPE_MAP.containsKey(fileExtension)) {
                                        // write to encrypted file with defined encryption cipher
                                        try (OutputStream outputStm = new CipherOutputStream(
                                            FileUtils.openOutputStream(
                                                new File(m_entryDir, curEntryName)), m_encryptCipher)) {

                                            // !!! use IOUtils.copy in order to leave the zip input stream open !!!
                                            IOUtils.copy(zipInputStm, outputStm);
                                        }

                                        // store entry names to be able to search for resources,
                                        // specified by uid fragment only when retrieving files
                                        m_entryNames.add(curEntryName);
                                    }
                                } catch (IOException e) {
                                    log.error("GetFile cache exception caught while extracting image entry from document stream: " + curEntryName, e);
                                }
                            }
                        } finally {
                            zipInputStm.closeEntry();
                        }
                    }
                }
            }

            // - Members -----------------------------------------------------------

            final private File m_entryDir;

            final private Set<String> m_entryNames = new LinkedHashSet<>();

            final private Cipher m_encryptCipher;

            final private Cipher m_decryptCipher;

            private volatile long m_timestampMillis = 0;
        }

        /**
         * Initializes a new {@link Cache}.
         * Each call to this class is thread safe.
         * If subsequent calls to this class are made with the same document id,
         * the call sequence needs to be made between calls to the lock(documentId)/
         * unlock(documentId) method calls (blocking of subsequent calls with same documentId).
         * E.g. <code> lock(documentId) =>
         *  try { hasEntry(documentId) => has and/or add entry followed by getEntry }
         *  finally { unlock(documentId) } </code>
         */
        public Cache(final File rootDir, final long maxTimeToLiveMillis) throws IllegalArgumentException {
            super();

            m_maxTimeToLiveMillis = maxTimeToLiveMillis;
            m_cacheDir = new File(
                (null != rootDir) ? rootDir : FileUtils.getTempDirectory(),
                CACHE_ROOT_DIRECTORY_NAME);

            try {
                // clean all possibly left over dir entries from last session and create new working directory
                FileUtils.deleteQuietly(m_cacheDir);
                FileUtils.forceMkdir(m_cacheDir);
            } catch (IOException e) {
                log.error("GetFile cache Ctor could not create working directory {}: {}",
                    m_cacheDir.getAbsolutePath(), Throwables.getRootCause(e));
            }

            // we don't have a file directory to work on => disable cache
            if (!m_cacheDir.canWrite()) {
                throw new IllegalArgumentException("GetFile cache is not able to create cache working directory: " + m_cacheDir.getAbsolutePath());
            }
        }

        /**
         *
         */
        public void shutdown() {
            if (m_isRunning.compareAndSet(true, false)) {
                final boolean trace = log.isTraceEnabled();
                long startTimeMillis = 0;

                if (trace) {
                    startTimeMillis = System.currentTimeMillis();
                    log.trace("GetFile cache starting shutdown");
                }

                m_timer.cancel();
                m_timer.purge();

                // clear all entries
                implCleanupEntries(CLEAR_ALL_ENTRIES);

                FileUtils.deleteQuietly(m_cacheDir);

                if (trace) {
                    log.trace("GetFile cache finished shutdown in {}ms", System.currentTimeMillis() - startTimeMillis);
                }
            }
        }

        /**
         * @return
         */
        public boolean isRunning() {
            return m_isRunning.get();
        }

        /**
         * @param documentId
         */
        public boolean lock(final String documentId) {
            if (null != documentId) {
                return ElementLocker.lock(documentId);
            }

            return false;
        }

        /**
         * @param documentId
         */
        public void unlock(final String documentId) {
            if (null != documentId) {
                ElementLocker.unlock(documentId);
            }
        }

        /**
         * @param documentId
         * @return
         */
        public boolean hasEntry(String documentId) {
            if (isRunning() && lock(documentId)) {
                try {
                    return m_entries.containsKey(documentId);
                } finally {
                    unlock(documentId);
                }
            }

            return false;
        }

        /**
         * @param documentId
         * @param documentStm
         */
        public void addEntry(String documentId, InputStream documentStm) {
            boolean restartTimer = false;

            if (isRunning() && (null != documentStm) && lock(documentId)) {
                try {
                    m_entries.put(documentId, new Entry(m_cacheDir, documentStm));
                    restartTimer = true;
                } catch (Exception e ) {
                    log.error("GetFile cache not able to create new entry: {}", Throwables.getRootCause(e));
                } finally {
                    ElementLocker.unlock(documentId);
                }
            }

            if (restartTimer) {
                implRestartTimer(m_maxTimeToLiveMillis);
            }
        }

        /**
         * @param documentId
         * @param resourceName
         * @param mimeType
         * @param imageExtents
         * @return
         */
        public byte[] getEntryResource(String documentId, String resourceName, Dimension imageExtents, String[] mimeType) {
            if (isRunning() && (null != resourceName) && lock(documentId)) {
                try {
                    Entry entryToUse = null;

                    synchronized (m_entries) {
                        // reorder touched entry to end of list
                        entryToUse = m_entries.remove(documentId);

                        if (null != entryToUse) {
                            entryToUse.touch();
                            m_entries.put(documentId, entryToUse);
                        }
                    }

                    // return content of touched entry as return value
                    if (null != entryToUse) {
                        return entryToUse.getResourceBuffer(resourceName, imageExtents, mimeType);
                    }
                } finally {
                    unlock(documentId);
                }
            }

            return null;
        }

        // - Implementation --------------------------------------------------------

        /**
         * @param timeoutMillis
         */
        private void implRestartTimer(final long timeoutMillis) {
            if (isRunning() && m_timerRunning.compareAndSet(false, true)) {
                m_timer.schedule(new TimerTask() {
                    @Override
                    public void run() {
                        m_timerRunning.set(false);
                        implCleanupEntries(System.currentTimeMillis());
                    }
                }, Math.max(0L, timeoutMillis));
            }
        }

        /**
         * @param curTimeMillis
         */
        private void implCleanupEntries(final long curTimeMillis) {
            long nextTimestampMillis = 0;

            // clear all entries from map that are currently accessible via tryLock
            synchronized (m_entries) {
                final Iterator<String> iter = m_entries.keySet().iterator();
                final boolean clearAll = (CLEAR_ALL_ENTRIES == curTimeMillis);
                final int startCount = m_entries.size();
                int removedCount = 0;

                while (iter.hasNext()) {
                    final String curDocumentId = iter.next();

                    if (ElementLocker.lock(curDocumentId, LockMode.TRY_LOCK)) {
                        try {
                            // get entry to be cleared and remove from map
                            final Entry curEntry = m_entries.get(curDocumentId);
                            final boolean timeoutReached = (curTimeMillis - curEntry.getTimestampMillis()) >= m_maxTimeToLiveMillis;

                            if (timeoutReached || clearAll) {
                                iter.remove();
                                curEntry.clear();
                                ++removedCount;
                            }

                            // finish cleanup in case of timeout based cleanup and all
                            // ordered, next entries did not reach their timeout yet
                            if (!timeoutReached && !clearAll) {
                                break;
                            }
                        } finally {
                            unlock(curDocumentId);
                        }
                    }
                }

                final int sizeLeft = m_entries.size();

                if (log.isTraceEnabled() && (removedCount > 0)) {
                    log.trace("GetFile cache automatically removed {}/{} entries ({} entries available after removal)",
                        removedCount, startCount, sizeLeft);
                }

                // restart timer based on next timeout entry in case there are entries left
                if (sizeLeft > 0) {
                    final Entry nextTimeoutEntry = m_entries.get(m_entries.keySet().iterator().next());

                    if (null != nextTimeoutEntry) {
                        nextTimestampMillis = nextTimeoutEntry.getTimestampMillis();
                    }
                }
            }

            // restart timer if there are entries left
            if (0 != nextTimestampMillis) {
                implRestartTimer(nextTimestampMillis + m_maxTimeToLiveMillis - System.currentTimeMillis());
            }
        }

        // - Members ---------------------------------------------------------------

        final private Map<String, Entry> m_entries = Collections.synchronizedMap(new LinkedHashMap<>());

        final private AtomicBoolean m_isRunning = new AtomicBoolean(true);

        final private Timer m_timer = new Timer();

        final private AtomicBoolean m_timerRunning = new AtomicBoolean(false);

        final private File m_cacheDir;

        final private long m_maxTimeToLiveMillis;
    }

    /**
     * Initializes a new {@link GetFileAction}.
     * @param resMgr
     * @param sessionService
     */
    public GetFileAction(final IResourceManager resMgr, final SessionService sessionService) {
        super();

        this.resourceManager = resMgr;
        this.documentConverter = ServiceLookupRegistry.get().getService(IDocumentConverter.class);
        this.configurationService = ServiceLookupRegistry.get().getService(ConfigurationService.class);

		try {
		    // create Cache instance to speedup GetFile requests for a document
		    final long cacheTimeoutMillis = (null != this.configurationService) ?
		        this.configurationService.getIntProperty("com.openexchange.office.cache.getFile.timeout", 120000) :
		            120000;

		    if (cacheTimeoutMillis > 0) {
		        this.cache = new Cache(new File(this.configurationService.getProperty("UPLOAD_DIRECTORY", "/tmp/")), cacheTimeoutMillis);
		        log.info("AjaxRequest GetFileAction cache is enabled with a timeout of {}ms", cacheTimeoutMillis);
		    } else {
                log.info("AjaxRequest GetFileAction cache is disabled");
		    }
		} catch (Exception e) {
            log.error("AjaxRequest GetFileAction exception caught while creating cache => disabling cache)", e);
		}
    }


    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.ajax.requesthandler.AJAXActionService#perform(com.openexchange.ajax.requesthandler.AJAXRequestData,
     * com.openexchange.tools.session.ServerSession)
     */
    @Override
    public AJAXRequestResult perform(AJAXRequestData request, ServerSession session) {
        AJAXRequestResult requestResult = null;
        String resourceName = request.getParameter("get_filename");
        String getFormat = request.getParameter("get_format");
        final String fileId = request.getParameter("id");
        final String folderId = request.getParameter("folder_id");
        final String versionOrAttachment = request.getParameter("version");
        final String source = request.getParameter("source");

        // check fileId/folderId
        if (isEmpty(fileId) || isEmpty(folderId)) {
            requestResult = new AJAXRequestResult();
            requestResult.setHttpStatusCode(400);

            log.error("AjaxRequest GetFileAction: for source {} and id {} / folderId {} / version {} not valid: ", source, fileId, folderId, versionOrAttachment);
            return requestResult;
        }

        int userId = Integer.MIN_VALUE;
        // retrieve user id and create data source access
        try {
            userId = session.getUserId();
        } catch (Exception e) {
            log.error("AjaxRequest GetFileAction exception caught while trying to get user id", e);

            final AJAXRequestResult result = new AJAXRequestResult();
            result.setHttpStatusCode(403);
            return result;
        }

        // check read access rights
        if (Integer.MIN_VALUE == userId) {
            final AJAXRequestResult result = new AJAXRequestResult();
            result.setHttpStatusCode(403);

            log.error("AjaxRequest GetFileAction error while checking file access rights");
            return result;
        }

        // check and sanitize resourceName
        if (isEmpty(resourceName)) {
            final AJAXRequestResult result = new AJAXRequestResult();
            result.setHttpStatusCode(400);

            log.error("AjaxRequest GetFileAction error while checking resource name");

            return result;
        } else  if (resourceName.startsWith("./")) {
            resourceName = resourceName.substring(2);
        }

        String fileName = FileHelper.getBaseName(resourceName);

        log.debug("AjaxRequest GetFileAction request for " + fileName);

        if (!"json".equals(getFormat)) {
            getFormat = "file";
        }

        boolean fileNameStartWithXID = fileName.startsWith(Resource.RESOURCE_UID_PREFIX);

        // try to get the graphic from the resource manager(s) first
        if (fileNameStartWithXID) {
            // 1. try: resource is already contained within this nodes resource manager
            final String uid = Resource.getUIDFromResourceName(fileName);
            final Resource nodeResource = resourceManager.getResource(uid);
            byte[] resdata = ((nodeResource != null) ? nodeResource.getBuffer() : null);

            log.trace("AjaxRequest GetFileAction " + fileName + " not found in resourceManager");

            // 2. try: create a managed resource with the given uid
            if (null == resdata) {
                try {
                    final String managedId = Resource.getManagedIdFromUID(uid);

                    if ((null != managedId) && (managedId.length() > 0)) {
                        final Resource managedResource = Resource.createFrom(null, managedId);

                        // add a new managed resource to this nodes resource manager for later speedup
                        // if we have data with content, otherwise we fall-back to read the doc stream
                        resdata = (null != managedResource) ? managedResource.getBuffer() : null;
                        if ((null != resdata) && (resdata.length > 0)) {
                            resourceManager.addResource(managedResource);
                        } else {
                            resdata = null;
                        }
                    }
                } catch (Exception e) {
                    // This is just an information: a cleanly loaded document contains the images
                    // and does not provide it via managed file. The 3rd access should provide the
                    // image. Therefore change this log into info.
                    log.debug("AjaxRequest GetFileAction exception caught while trying to create managed resource",  e);
                }
            }

            // if we've got the resource data, set it as request result
            if (null != resdata) {
                requestResult = createFileRequestResult(request, resdata, getFormat, DocFileHelper.getMimeType(resourceName), resourceName);
            }
        }

        // 3. try getting the graphic from the Cache (preferred) or the document's zip archive itself
        if (null == requestResult) {
            final String authCode = AuthorizationCache.getKey("" + session.getContextId(), "" + session.getUserId(), fileId);
            final String encryptionInfo = (null != authCode) ? EncryptionInfo.createEncryptionInfo(session, authCode) : "";
            final String widthParam = request.getParameter("width");
            final String heightParam = request.getParameter("height");
            final Dimension imageExtents = getImageExtents(widthParam, heightParam);
            final StorageHelper storageHelper = new StorageHelper(session, folderId);
            final String[] mimeType = { "" };
            final String documentId = new StringBuilder(256).
                append("MetaData [folderId=").
                append(folderId).
                append(", id=").
                append(fileId).
                append(", versionOrAttachment=").
                append(versionOrAttachment).
                append(", fileName=").
                append(fileName).
                append(", source=").
                append(source).
                append("]").toString();

            // always use the current version of the document
            request.putParameter("version", "0");

            // 3.1 try to get result from cache or add to/retrieve from cache afterwards
            if ((null != cache) && (null != documentId) && cache.lock(documentId)) {
                try {
                    if (!cache.hasEntry(documentId)) {
                        try (final StreamInfo streamInfo = DocFileHelper.getDocumentStream(null, session, request, storageHelper, encryptionInfo);
                            final InputStream documentStm = streamInfo.getDocumentStream()) {

                            if (null != documentStm) {
                                cache.addEntry(documentId, documentStm);
                            }
                        }
                    }

                    final byte[] resourceBuffer = cache.getEntryResource(documentId, resourceName, imageExtents, mimeType);

                    if (null != resourceBuffer) {
                        if (fileNameStartWithXID) {
                            // add resource to global resource manager for a faster second access
                            resourceManager.addResource(resourceBuffer);
                        }

                        requestResult = createFileRequestResult(request, resourceBuffer, getFormat, mimeType[0], fileName);

                        log.trace("AjaxRequest GetFileAction " + fileName + " successfully retrieved from cache");
                    } else {
                        log.warn("AjaxRequest GetFileAction " + fileName + " not able to be retrieved from cache");
                    }
                } catch (IOException e) {
                    log.trace("AjaxRequest GetFileAction could not access file via GetFile cache " + fileName, e);
                } finally {
                    cache.unlock(documentId);
                }
            }

            // 3.2 we got no result from cache => image must be retrieved from document stream
            if (null == requestResult) {
                log.trace("AjaxRequest GetFileAction image must be retrieved from document stream");

                try (final StreamInfo streamInfo = DocFileHelper.getDocumentStream(null, session, request, storageHelper, encryptionInfo);
                    final InputStream documentStm = streamInfo.getDocumentStream()) {

                    if (null != documentStm) {
                        log.trace("AjaxRequest GetFileAction document stream successfully retrieved - try to get substream for " + fileName);

                        try (final InputStream resultStm = getZipEntryStream(documentStm, session, resourceName, imageExtents, mimeType)) {
                            if (null != resultStm) {
                                byte[] buffer = IOUtils.toByteArray(resultStm);

                                if (null != buffer) {
                                    if (fileNameStartWithXID) {
                                        // add resource to global resource manager for a faster second access
                                        resourceManager.addResource(buffer);
                                    }

                                    requestResult = createFileRequestResult(request, buffer, getFormat, mimeType[0], fileName);

                                    log.trace("AjaxRequest GetFileAction " + fileName + " successfully retrieved");
                                } else {
                                    log.warn("AjaxRequest GetFileAction " + fileName + " not found");
                                }
                            }
                        }
                    }
                } catch (IOException e) {
                    log.error("AjaxRequest GetFileAction I/O exception caught while reading data from document stream", e);
                }
            }
        }

        if (null == requestResult) {
            log.warn("AjaxRequest: GetFileAction request not successfully processed for resource: " + ((null == resourceName) ? "unknown resource name" : resourceName) + " in: " + fileName);

            request.setFormat("json");
            requestResult = new AJAXRequestResult();
        }

        return requestResult;
    }

    // - Public API ------------------------------------------------------------

    /**
     * @throws OXException
     */
    public void shutdown() throws OXException {
        if (null != cache) {
            cache.shutdown();
        }
    }

    /**
     * Determines if a content stream format must be converted to be ensure that
     * a browser is able to display the content.
     *
     * @param fileExtension the file extension of the stream
     * @param resourceName the resource name of the stream
     * @return TRUE if the format must be converted or otherwise FALSE.
     */
    public boolean needsConversion(final String resourceName, final String fileExtension) {
        return "emf".equals(fileExtension) ||
               "wmf".equals(fileExtension) ||
               "svm".equals(fileExtension) ||
               (StringUtils.isNotEmpty(resourceName) && (resourceName.indexOf("ObjectReplacements/") > -1));
    }

    /**
     * @param inputStm Image input stream to be converted.
     * @param resourceName The name of the current resource
     * @param fileExtension The extension of the input image resource
     * @param extents The target image extents, migh be <code>null</code>
     * @param mimeType contains the mime type of the converted result stream (out param)
     * @return The converted result image buffer or a copy of the given input stream
     *  as result buffer, if no DocumentConverter is available
     */
    public byte[] convert(final InputStream inputStm,
        final String resourceName,
        final String fileExtension,
        final Dimension extents,
        final String[] mimeType) throws IOException {

        if ((null != documentConverter) && (null != fileExtension) && (null != resourceName)) {
            final HashMap<String, Object> jobProperties = new HashMap<>(8);
            final HashMap<String, Object> resultProperties = new HashMap<>(8);

            jobProperties.put(Properties.PROP_INPUT_STREAM, inputStm);
            jobProperties.put(Properties.PROP_INPUT_TYPE, fileExtension);
            jobProperties.put(Properties.PROP_MIME_TYPE, "image/png");
            jobProperties.put(Properties.PROP_INFO_FILENAME, resourceName);

            if (null != extents) {
                jobProperties.put(Properties.PROP_PIXEL_WIDTH, Integer.valueOf(extents.width));
                jobProperties.put(Properties.PROP_PIXEL_HEIGHT, Integer.valueOf(extents.height));
            }

            // this is a user request in every case
            jobProperties.put(Properties.PROP_USER_REQUEST, Boolean.TRUE);

            try (final InputStream resultStm = documentConverter.convert("graphic", jobProperties, resultProperties)) {
                if (null != resultStm) {
                    if (ArrayUtils.isNotEmpty(mimeType)) {
                        mimeType[0] = (String) resultProperties.get(Properties.PROP_RESULT_MIME_TYPE);
                    }

                    return IOUtils.toByteArray(resultStm);
                }
            }
        }

        // return copy of given input stream, if conversion is not possible
        return IOUtils.toByteArray(inputStm);
    }

    // - Implementation --------------------------------------------

    /**
     * @param documentStm
     * @param resourceName
     * @param mimeType
     * @param closeDocumentStream
     * @return
     */
    final private InputStream getZipEntryStream(InputStream documentStm, ServerSession session, String resourceName, Dimension extents, String[] mimeType) {
        InputStream resultStm = null;

        if (null != documentStm) {
            final String filename = FileHelper.getBaseName(resourceName);
            String uidResourceName = null;

            // check for uid Resources within all entries
            if (filename.startsWith(Resource.RESOURCE_UID_PREFIX) && (filename.length() > Resource.RESOURCE_UID_PREFIX.length())) {
                uidResourceName = filename;
            }

            try (final ZipInputStream zipInputStm = new ZipInputStream(documentStm)) {

                for (ZipEntry zipEntry = null; (null == resultStm) && ((zipEntry = zipInputStm.getNextEntry()) != null);) {
                    try {
                        final String curEntryName = zipEntry.getName();

                        if (curEntryName.equals(DocFileHelper.OX_RESCUEDOCUMENT_DOCUMENTSTREAM_NAME)) {
                            // check for resources contained within the original rescue document
                            byte[] entryBuffer = IOUtils.toByteArray(zipInputStm);

                            if (null != entryBuffer) {
                                try (final InputStream entryInputStream = new ByteArrayInputStream(entryBuffer)) {
                                    resultStm = getZipEntryStream(entryInputStream, session, resourceName, extents, mimeType);
                                }
                            }
                        } else if (curEntryName.equals(resourceName) || ((null != uidResourceName && curEntryName.contains(uidResourceName)))) {
                            // check for resources equal the the given name and for resources matching the uid resource manager naming schema
                            byte[] entryBuffer = null;

                            try {
                                // use an entry buffer as we don't want the zipInputStm
                                // to be closed by possibly called method implementations
                                entryBuffer = IOUtils.toByteArray(zipInputStm);
                            } catch (IOException e) {
                                log.error("AjaxRequest GetFileAction.getZipEntryStream exception caught while trying to read data from document stream", e);
                            }

                            if (null != entryBuffer) {
                                final String fileExtension = FilenameUtils.getExtension(resourceName).trim().toLowerCase();

                                // check if we create a converted png file that can be displayed within the browser,
                                // we will do this for the formats "wmf and emf... TODO: the resulting graphic should
                                // somehow be part of the resourcemanager to prevent unnecessary graphic conversions
                                if (needsConversion(resourceName, fileExtension)) {
                                    try (final InputStream inputStm = new ByteArrayInputStream(entryBuffer)) {
                                        final byte[] convertBuffer = convert(inputStm,resourceName, fileExtension, extents, mimeType);

                                        if (null != convertBuffer) {
                                            resultStm = new ByteArrayInputStream(convertBuffer);
                                        }
                                    }
                                }

                                if (null == resultStm) {
                                    resultStm = new ByteArrayInputStream(entryBuffer);
                                    mimeType[0] = DocFileHelper.getMimeType(resourceName);
                                }
                            }
                        }
                    } finally {
                        zipInputStm.closeEntry();
                    }
                }
            } catch (Throwable e) {
	        	ExceptionUtils.handleThrowable(e);
                log.error("AjaxRequest GetFileAction.getZipEntryStream exception caught while trying to read data from document stream", e);
            }
        }

        // !!! Possible resource leak warning has been checked:
        // returned resultStm is closed by caller of this method
        return resultStm;
    }

    /**
     * @param request
     * @return
     */
    private Dimension getImageExtents(String widthParam, String heightParam) {
        // get optional extents (width/height)
        try {
            int width = -1;
            int height = -1;

            if (isNotEmpty(widthParam)) {
                width = Integer.parseInt(widthParam);
            }

            if (isNotEmpty(heightParam)) {
                height = Integer.parseInt(heightParam);
            }

            if ((width != 0) || (height != 0)) {
                return new Dimension((width > 0) ? width : -1, (height > 0) ? height : -1);
            }
        } catch (NumberFormatException e) {
            log.error("AjaxRequest GetFileAction exception caught while parsing width/height parameters", e);
        }

        return null;

    }

    // - Members ---------------------------------------------------------------

    final private IResourceManager resourceManager;

    final private ConfigurationService configurationService;

    final private IDocumentConverter documentConverter;

    private Cache cache = null;

    // - Static members --------------------------------------------------------

    protected final static Logger log = LoggerFactory.getLogger(GetFileAction.class);

    /**
     * IMAGE_EXTENSION_TO_MIMETYPE_MAP
     */
    final private static Map<String, String> IMAGE_EXTENSION_TO_MIMETYPE_MAP = new HashMap<>(24);

    /**
     * initialization of static class member
     */
    {
        IMAGE_EXTENSION_TO_MIMETYPE_MAP.put("bmp", " image/bmp");
        IMAGE_EXTENSION_TO_MIMETYPE_MAP.put("emf", "image/x-emf");
        IMAGE_EXTENSION_TO_MIMETYPE_MAP.put("gif", "image/gif");
        IMAGE_EXTENSION_TO_MIMETYPE_MAP.put("heic", "image/heic");
        IMAGE_EXTENSION_TO_MIMETYPE_MAP.put("heif", "image/heif");
        IMAGE_EXTENSION_TO_MIMETYPE_MAP.put("ico", "image/x-icon");
        IMAGE_EXTENSION_TO_MIMETYPE_MAP.put("jpg", "image/jpeg");
        IMAGE_EXTENSION_TO_MIMETYPE_MAP.put("jpe", "image/jpeg");
        IMAGE_EXTENSION_TO_MIMETYPE_MAP.put("jpeg", "image/jpeg");
        IMAGE_EXTENSION_TO_MIMETYPE_MAP.put("pcx", "image/x-pcx");
        IMAGE_EXTENSION_TO_MIMETYPE_MAP.put("pct", "image/x-pict");
        IMAGE_EXTENSION_TO_MIMETYPE_MAP.put("pict", "image/x-pcx");
        IMAGE_EXTENSION_TO_MIMETYPE_MAP.put("pbm", "image/x-portable-bitmap");
        IMAGE_EXTENSION_TO_MIMETYPE_MAP.put("pnm", "image/x-portable-anymap");
        IMAGE_EXTENSION_TO_MIMETYPE_MAP.put("png", "image/png");
        IMAGE_EXTENSION_TO_MIMETYPE_MAP.put("psd", "image/vnd.adobe.photoshop");
        IMAGE_EXTENSION_TO_MIMETYPE_MAP.put("svm", "image/svm");
        IMAGE_EXTENSION_TO_MIMETYPE_MAP.put("tif", "image/tiff" );
        IMAGE_EXTENSION_TO_MIMETYPE_MAP.put("tiff", "image/tiff");
        IMAGE_EXTENSION_TO_MIMETYPE_MAP.put("tga", "image/x-tga");
        IMAGE_EXTENSION_TO_MIMETYPE_MAP.put("wmf", "windows/metafile");
    }

}
