/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.rest;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import com.openexchange.ajax.requesthandler.AJAXActionService;
import com.openexchange.ajax.requesthandler.AJAXActionServiceFactory;
import com.openexchange.folderstorage.FolderService;
import com.openexchange.office.document.LocalFileMappingManager;
import com.openexchange.office.document.TemplateFilesScanner;
import com.openexchange.office.imagemgr.IResourceManager;
import com.openexchange.office.imagemgr.IResourceManagerFactory;
import com.openexchange.office.session.SessionService;
import com.openexchange.office.tools.osgi.ServiceLookupRegistryService;

/**
 * {@link DocumentRESTActionFactory}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 */
public class DocumentRESTActionFactory implements AJAXActionServiceFactory {

    private final Map<String, AJAXActionService> actions = new HashMap<>();
    private final IResourceManager resourceManager;
    private final SessionService sessionService;
    private final FolderService folderService;
    private TemplateFilesScanner templateFilesScanner = null;
    private LocalFileMappingManager localFileMapper = null;
    private LogPerformanceDataWriter logPerformanceDataWriter = null;

    /**
     * Initializes a new {@link DocumentRESTActionFactory}.
     */
    public DocumentRESTActionFactory() throws Exception {
        super();

        resourceManager = ServiceLookupRegistryService.getInstance().getService(IResourceManagerFactory.class).createInstance();
        sessionService = ServiceLookupRegistryService.getInstance().getService(SessionService.class);
        folderService = ServiceLookupRegistryService.getInstance().getService(FolderService.class);

        localFileMapper = new LocalFileMappingManager();
        templateFilesScanner = new TemplateFilesScanner(null, localFileMapper, TemplateFilesScanner.TemplateFilterType.ALL_DOCUMENTS);
        logPerformanceDataWriter = new LogPerformanceDataWriter();

        actions.put("addfile", new AddFileAction(resourceManager));
        actions.put("createdefaultdocument", new CreateDefaultDocumentAction());
        actions.put("getfeatures", new GetFeaturesAction());
        actions.put("getfile", new GetFileAction(resourceManager, sessionService));
        actions.put("gettemplatelist", new GetTemplateListAction(templateFilesScanner));
        actions.put("createfromtemplate", new CreateFromTemplateAction(localFileMapper));
        actions.put("templatepreview", new TemplatePreviewAction(localFileMapper));
        actions.put("gettemplateandrecentlist", new GetTemplateAndRecentListAction(templateFilesScanner));
        actions.put("deletefilefromrecentlist", new DeleteFileFromRecentListAction());
        actions.put("clearrecentlist", new ClearRecentListAction());
        actions.put("getuserinfo", new GetUserInfoAction(sessionService));
        actions.put("getcontexttemplatefolders", new GetContextTemplateFoldersAction());
        actions.put("saveaswithbackup", new SaveAsWithBackupAction());
        actions.put("logperformancedata", new LogPerformanceDataAction(logPerformanceDataWriter));
        actions.put("getoperations", new GetOperationsAction());
        actions.put("getdocument", new GetDocumentAction());
        actions.put("logclientdata", new LogClientDataAction(folderService));
    }

    @Override
    public AJAXActionService createActionService(final String action) {
        return (actions.get(action));
    }

    public Collection<? extends AJAXActionService> getSupportedServices() {
        return java.util.Collections.unmodifiableCollection(actions.values());
    }

    public void shutdown() {
        final LogPerformanceDataWriter aDataWriter = logPerformanceDataWriter;
        if (null != aDataWriter) {
            aDataWriter.shutdown();
        }

        final GetOperationsAction getOperationsAction = (GetOperationsAction)actions.get("getoperations");
        if (null != getOperationsAction) {
            getOperationsAction.shutdown();
        }
    }
}
