/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.ot.tools;

import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.office.filter.api.OCKey;
import com.openexchange.office.filter.api.OCValue;
import com.openexchange.office.ot.TransformHandlerBasic;
import com.openexchange.office.tools.doc.DocumentType;
import com.openexchange.office.tools.monitoring.PresenterEvent;
import com.openexchange.office.tools.monitoring.PresenterEventType;
import com.openexchange.office.tools.monitoring.Statistics;

public class OTUtils {
    // -------------------------------------------------------------------------
    // Helper functions

    /**
     * Transforming an array of external operations by using all pending operations
     * that need to be applied to the external operations.
     *
     * @param {JSONArray} pendingOps
     *  The collection of operations that are already applied to the document on
     *  server side, but were missing, when the client generated the new operations
     *  before sending them to the server.
     *
     * @param {JSONArray} newOps
     *  The array that contains all operations sent from the client. After this OT
     *  process these operations are applied to the document and sent to all other
     *  clients.
     *
     * @returns {Object[]}
     *  The transformed external operations. These operations are applied to the
     *  document and sent to all other clients.
     */
    public static JSONArray transformOperations(final DocumentType documentType, final Map<OpPair, ITransformHandler> transformHandlerMap, final JSONArray pendingOps, final JSONArray newOps) throws JSONException {

        final JSONArray transformedOperations = new JSONArray();
        for (int i = 0; i < newOps.length(); i++) {

            final JSONObject oneExternOp = newOps.getJSONObject(i);
            JSONArray allExtOps = new JSONArray(); // the collector for all external operations
            allExtOps.put(oneExternOp);
            JSONObject result = null;

            for (int j = 0; j < pendingOps.length(); j++) {

                JSONArray newLclOpsBefore = null; // collect new local operations to be added before the current local operation
                JSONArray newLclOpsAfter = null; // collect new local operations to be added after the current local operation

                final JSONObject localOp = pendingOps.getJSONObject(j);

                for (int k = 0; k < allExtOps.length(); k++) {
                    final JSONObject externOp = allExtOps.getJSONObject(k);

                    result = applyLocalOpToExternalOp(documentType, transformHandlerMap, localOp, externOp);

                    if (result != null) {

                        int afterLength = 0;
                        int beforeLength = 0;

                        final JSONArray externalOpsAfter = result.optJSONArray("externalOpsAfter");
                        if (externalOpsAfter!=null) {
                            afterLength = externalOpsAfter.length();
                            for (int m = 0; m < afterLength; m++) {
                                allExtOps.add(k + m + 1, externalOpsAfter.get(m));
                            }
                        }
                        final JSONArray externalOpsBefore = result.optJSONArray("externalOpsBefore");
                        if (externalOpsBefore!=null) {
                            beforeLength = externalOpsBefore.length();
                            for (int m = 0; m < beforeLength; m++) {
                                allExtOps.add(k + m, externalOpsBefore.get(m));
                            }
                        }
                        k = k + afterLength + beforeLength; // increasing the counter for the array allExtOps

                        // also collecting the new local operations
                        final JSONArray localOpsBefore = result.optJSONArray("localOpsBefore");
                        if (localOpsBefore!=null) {
                            if (newLclOpsBefore == null) {
                                newLclOpsBefore = new JSONArray();
                            }
                            for (int m = 0; m < localOpsBefore.length(); m++) {
                                newLclOpsBefore.put(localOpsBefore.get(m));
                            }
                        }
                        final JSONArray localOpsAfter = result.optJSONArray("localOpsAfter");
                        if (localOpsAfter!=null) {
                            if (newLclOpsAfter == null) {
                                newLclOpsAfter = new JSONArray();
                            }
                            for (int m = 0; m < localOpsAfter.length(); m++) {
                                newLclOpsAfter.put(localOpsAfter.get(m));
                            }
                        }
                    }
                };

                int afterLen = 0;
                int beforeLen = 0;

                if (newLclOpsAfter != null) {
                    afterLen = newLclOpsAfter.length();
                    for (int m = 0; m < afterLen; m++) {
                        pendingOps.add(j + m + 1, newLclOpsAfter.get(m));
                    }
                }
                if (newLclOpsBefore != null) {
                    beforeLen = newLclOpsBefore.length();
                    for (int m = 0; m < beforeLen; m++) {
                        pendingOps.add(j + m, newLclOpsBefore.get(m));
                    }
                }
                j = j + afterLen + beforeLen; // increasing the counter for the pending OPs
            };

            if (allExtOps.length() > 0) {
                for (int m = 0; m < allExtOps.length() ; m++) {
                    final JSONObject oneExtOp = allExtOps.getJSONObject(m);
                    if (!OTUtils.isOperationRemoved(oneExtOp)) {
                        transformedOperations.put(oneExtOp);
                    }
                }
            }
        };
        return transformedOperations;
    }


    // -------------------------------------------------------------------------
    /**
     * The operation dispatcher for the Operational Transformations.
     *
     * @param {JSONObject} localOp
     *  The local operation.
     *
     * @param {JSONObject} extOp
     *  The external operation.
     *
     * @param {JSONArray} allExtOps
     *  The container for the external operation(s).
     *
     * @param {int} extIdx
     *  The index of the external operation inside its operation container.
     *
     * @param {JSONArray} localOps
     *  The container with the local operations.
     *
     * @param {int} localIdx
     *  The index of the local operation inside its operation container of the action.
     *
     * @throws JSONException
     */
    public static JSONObject applyLocalOpToExternalOp(final DocumentType documentType, Map<OpPair, ITransformHandler> transformHandlerMap, final JSONObject localOp, final JSONObject externOp) throws JSONException {

        // check, if the local operation is removed in the meantime
        if (OTUtils.isOperationRemoved(localOp) || OTUtils.isOperationRemoved(externOp)) { return null; }

        // check the target of the operations
        if (OTUtils.hasTarget(localOp) || OTUtils.hasTarget(externOp)) {
            boolean isTargetRemove = OTUtils.isDeleteTargetOperation(localOp) || OTUtils.isDeleteTargetOperation(externOp);
            if (!isTargetRemove) {
                if ((OTUtils.hasTarget(localOp) && !OTUtils.hasTarget(externOp)) || (!OTUtils.hasTarget(localOp) && OTUtils.hasTarget(externOp))) { return null; }
                if (!OTUtils.getTarget(localOp).equals(OTUtils.getTarget(externOp))) { return null; }
            }
        }
        final OCValue opA = OCValue.fromValue(localOp.getString(OCKey.NAME.value()));
        final OCValue opB = OCValue.fromValue(externOp.getString(OCKey.NAME.value()));
        final OpAlias aliasA = OpAlias.fromValue(opA);
        final OpAlias aliasB = OpAlias.fromValue(opB);
        final OpPair k = new OpPair(aliasA!=null?aliasA:opA, aliasB!=null?aliasB:opB);

        ITransformHandler handler = transformHandlerMap!=null ? transformHandlerMap.get(k) : null;

        if(handler==null) {
            handler = TransformHandlerBasic.getTransformHandlerMap().get(k);
        }

        if (null != documentType) {
            Statistics.handleTransformation(documentType);
        }

        return handler.handle(localOp, externOp);
    }

    public static JSONObject getOperation(OCValue op, JSONObject op1, JSONObject op2) throws JSONException {
        return op1.getString(OCKey.NAME.value()).equals(op.value()) ? op1 : op2;
    }

    public static JSONArray getStartPosition(final JSONObject operation) throws JSONException {
        return operation.getJSONArray(OCKey.START.value());
    }

    public static void setStartPosition(final JSONObject operation, JSONArray pos) throws JSONException {
        operation.put(OCKey.START.value(), pos);
    }

    public static JSONArray optEndPosition(final JSONObject operation) {
        return operation.optJSONArray(OCKey.END.value());
    }

    public static void setEndPosition(final JSONObject operation, JSONArray pos) throws JSONException {
        operation.put(OCKey.END.value(), pos);
    }

    public static void removeEndPosition(final JSONObject operation) {
        operation.remove(OCKey.END.value());
    }

    public static void removeType(final JSONObject operation) {
        operation.remove(OCKey.TYPE.value());
    }

    public static JSONArray getToPosition(final JSONObject operation) throws JSONException {
        return operation.getJSONArray(OCKey.TO.value());
    }

    public static String getName(final JSONObject operation) throws JSONException {
        return operation.getString(OCKey.NAME.value());
    }

    public static String getTarget(final JSONObject operation) {
        return operation.optString(OCKey.TARGET.value());
    }

    public static boolean hasTarget(final JSONObject operation) {
        return operation.has(OCKey.TARGET.value());
    }

    public static JSONObject getAttributes(final JSONObject operation) throws JSONException {
        return operation.getJSONObject(OCKey.ATTRS.value());
    }

    public static void setAttributes(final JSONObject operation, JSONObject attrs) throws JSONException {
        operation.put(OCKey.ATTRS.value(), attrs);
    }

    public static void setOperationName(final JSONObject operation, String value) throws JSONException {
        operation.put(OCKey.NAME.value(), value);
    }

    public static void setOperationNameDelete(final JSONObject operation) throws JSONException {
        operation.put(OCKey.NAME.value(), OCValue.DELETE.value());
    }

    public static void setOperationNameDeleteHeaderFooter(final JSONObject operation) throws JSONException {
        operation.put(OCKey.NAME.value(), OCValue.DELETE_HEADER_FOOTER.value());
    }

    public static void setOperationNameInsertDrawing(final JSONObject operation) throws JSONException {
        operation.put(OCKey.NAME.value(), OCValue.INSERT_DRAWING.value());
    }

    public static String getText(final JSONObject operation) {
        return operation.optString(OCKey.TEXT.value());
    }

    public static boolean isInsertTextOperation(final JSONObject operation) throws JSONException {
        return operation.getString(OCKey.NAME.value()).equals(OCValue.INSERT_TEXT.value());
    }

    public static boolean isDeleteOperation(final JSONObject operation) throws JSONException {
        return operation.getString(OCKey.NAME.value()).equals(OCValue.DELETE.value());
    }

    public static boolean isSetAttributesOperation(final JSONObject operation) throws JSONException {
        return operation.getString(OCKey.NAME.value()).equals(OCValue.SET_ATTRIBUTES.value());
    }

    public static boolean isUpdateComplexFieldOperation(final JSONObject operation) throws JSONException {
        return operation.getString(OCKey.NAME.value()).equals(OCValue.UPDATE_COMPLEX_FIELD.value());
    }

    public static boolean isUpdateFieldOperation(final JSONObject operation) throws JSONException {
        return operation.getString(OCKey.NAME.value()).equals(OCValue.UPDATE_FIELD.value());
    }

    public static boolean isAnyUpdateFieldOperation(final JSONObject operation) throws JSONException {
        return isUpdateFieldOperation(operation) || isUpdateComplexFieldOperation(operation);
    }

    public static boolean isInsertParagraphOperation(final JSONObject operation) throws JSONException {
        return operation.getString(OCKey.NAME.value()).equals(OCValue.INSERT_PARAGRAPH.value());
    }

    public static boolean isInsertTableOperation(final JSONObject operation) throws JSONException {
        return operation.getString(OCKey.NAME.value()).equals(OCValue.INSERT_TABLE.value());
    }

    public static boolean isSplitParagraphOperation(final JSONObject operation) throws JSONException {
        return operation.getString(OCKey.NAME.value()).equals(OCValue.SPLIT_PARAGRAPH.value());
    }

    public static boolean isMergeParagraphOperation(final JSONObject operation) throws JSONException {
        return operation.getString(OCKey.NAME.value()).equals(OCValue.MERGE_PARAGRAPH.value());
    }

    public static boolean isMergeTableOperation(final JSONObject operation) throws JSONException {
        return operation.getString(OCKey.NAME.value()).equals(OCValue.MERGE_TABLE.value());
    }

    public static boolean isSplitTableOperation(final JSONObject operation) throws JSONException {
        return operation.getString(OCKey.NAME.value()).equals(OCValue.SPLIT_TABLE.value());
    }

    public static boolean isInsertRowsOperation(final JSONObject operation) throws JSONException {
        return operation.getString(OCKey.NAME.value()).equals(OCValue.INSERT_ROWS.value());
    }

    public static boolean isInsertCellsOperation(final JSONObject operation) throws JSONException {
        return operation.getString(OCKey.NAME.value()).equals(OCValue.INSERT_CELLS.value());
    }

    public static boolean isInsertColumnOperation(final JSONObject operation) throws JSONException {
        return operation.getString(OCKey.NAME.value()).equals(OCValue.INSERT_COLUMN.value());
    }

    public static boolean isDeleteColumnsOperation(final JSONObject operation) throws JSONException {
        return operation.getString(OCKey.NAME.value()).equals(OCValue.DELETE_COLUMNS.value());
    }

    public static boolean isDeleteListStyleOperation(final JSONObject operation) throws JSONException {
        return operation.getString(OCKey.NAME.value()).equals(OCValue.DELETE_LIST_STYLE.value());
    }

    public static boolean isMoveOperation(final JSONObject operation) throws JSONException {
        return operation.getString(OCKey.NAME.value()).equals(OCValue.MOVE.value());
    }

    public static boolean isDeleteCommentOperation(final JSONObject operation) throws JSONException {
        return operation.getString(OCKey.NAME.value()).equals(OCValue.DELETE_COMMENT.value());
    }

    public static boolean isDeleteHeaderFooterOperation(final JSONObject operation) throws JSONException {
        return operation.getString(OCKey.NAME.value()).equals(OCValue.DELETE_HEADER_FOOTER.value());
    }

    public static boolean isDeleteStyleSheetOperation(final JSONObject operation) throws JSONException {
        return operation.getString(OCKey.NAME.value()).equals(OCValue.DELETE_STYLE_SHEET.value());
    }

    public static boolean isDeleteTargetOperation(final JSONObject operation) throws JSONException {
        return isDeleteCommentOperation(operation) || isDeleteHeaderFooterOperation(operation);
    }

    public static String getIdProperty(final JSONObject operation) {
        return operation.optString(OCKey.ID.value());
    }

    public static String getStyleIdProperty(final JSONObject operation) {
        return operation.optString(OCKey.STYLE_ID.value());
    }

    public static int getCountProperty(final JSONObject operation, int def) {
        return operation.optInt(OCKey.COUNT.value(), def);
    }

    public static int getStartGridProperty(final JSONObject operation) throws JSONException {
        return operation.getInt(OCKey.START_GRID.value());
    }

    public static void setStartGridProperty(final JSONObject operation, int value) throws JSONException {
        operation.put(OCKey.START_GRID.value(), value);
    }

    public static void removeStartGridProperty(final JSONObject operation) {
        operation.remove(OCKey.START_GRID.value());
    }

    public static int getEndGridProperty(final JSONObject operation) throws JSONException {
        return operation.getInt(OCKey.END_GRID.value());
    }

    public static void setEndGridProperty(final JSONObject operation, int value) throws JSONException {
        operation.put(OCKey.END_GRID.value(), value);
    }

    public static void removeEndGridProperty(final JSONObject operation) {
        operation.remove(OCKey.END_GRID.value());
    }

    public static int getGridPositionProperty(final JSONObject operation) throws JSONException {
        return operation.getInt(OCKey.GRID_POSITION.value());
    }

    public static void setGridPositionProperty(final JSONObject operation, int value) throws JSONException {
        operation.put(OCKey.GRID_POSITION.value(), value);
    }

    public static String getInsertModeProperty(final JSONObject operation) {
        return operation.optString(OCKey.INSERT_MODE.value(), "behind");
    }

    public static boolean isInsertModeBehind(final String mode) {
        return mode.equals("behind");
    }

    public static boolean isInsertModeBefore(final String mode) {
        return mode.equals("before");
    }

    public static String getMergeLengthProperty(final JSONObject operation) throws JSONException {
        return isMergeTableOperation(operation) ? "rowcount" : "paralength";
    }

    public static String getMergeNamePropertyAfterSplit(final JSONObject operation) throws JSONException {
        return isSplitTableOperation(operation) ? OCValue.MERGE_TABLE.value() : OCValue.MERGE_PARAGRAPH.value();
    }

    public static String getSplitNamePropertyAfterMerge(final JSONObject operation) throws JSONException {
        return isMergeTableOperation(operation) ? OCValue.SPLIT_TABLE.value() : OCValue.SPLIT_PARAGRAPH.value();
    }

    public static String getMergeLengthPropertyAfterSplit(final JSONObject operation) throws JSONException {
        return isSplitTableOperation(operation) ? "rowcount" : "paralength";
    }

    public static int getParaLengthProperty(final JSONObject operation) throws JSONException {
        String key = isMergeTableOperation(operation) ? "rowcount" : "paralength";
        return operation.optInt(key, 0);
    }

    public static void setParaLengthProperty(final JSONObject operation, int value) throws JSONException {
        String key = isMergeTableOperation(operation) ? "rowcount" : "paralength";
        operation.put(key, value);
    }

    public static JSONArray getTableGridProperty(final JSONObject operation) throws JSONException {
        return operation.getJSONArray(OCKey.TABLE_GRID.value());
    }

    public static void setTableGridProperty(final JSONObject operation, JSONArray value) throws JSONException {
        operation.put(OCKey.TABLE_GRID.value(), value);
    }

    public static int getReferenceRowProperty(final JSONObject operation) throws JSONException {
        return operation.getInt(OCKey.REFERENCE_ROW.value());
    }

    public static boolean hasReferenceRowProperty(final JSONObject operation) {
        return operation.has(OCKey.REFERENCE_ROW.value());
    }

    public static void setReferenceRowProperty(final JSONObject operation, int value) throws JSONException {
        operation.put(OCKey.REFERENCE_ROW.value(), value);
    }

    public static void setOperationRemoved(final JSONObject operation) throws JSONException {
        operation.put("_REMOVED_OPERATION_", true);
    }

    public static boolean isOperationRemoved(final JSONObject operation) {
        return operation.has("_REMOVED_OPERATION_");
    }

    public static String getInstructionProperty(final JSONObject operation) {
        return operation.optString(OCKey.INSTRUCTION.value(), "");
    }

    public static String getTypeProperty(final JSONObject operation) {
        return operation.optString(OCKey.TYPE.value(), "");
    }

    public static String getRepresentationProperty(final JSONObject operation) {
        return operation.optString(OCKey.REPRESENTATION.value(), "");
    }

    public static String getListStyleIdProperty(final JSONObject operation) {
        return operation.optString(OCKey.LIST_STYLE_ID.value(), "");
    }

    public static Boolean isSameStyleSheet(final JSONObject op1, final JSONObject op2) {
        return OTUtils.getStyleIdProperty(op1).equals(OTUtils.getStyleIdProperty(op2)) && OTUtils.getTypeProperty(op1).equals(OTUtils.getTypeProperty(op2));
    }

    // -------------------------------------------------------------------------

    public static JSONArray cloneJSONArray(JSONArray source) throws JSONException {
        final JSONArray dest = new JSONArray(source.length());
        for(int i=0; i < source.length(); i++) {
            final Object s = source.get(i);
            Object d;
            if(s instanceof JSONObject) {
                d = cloneJSONObject((JSONObject)s);
            }
            else if(s instanceof JSONArray) {
                d = cloneJSONArray((JSONArray)s);
            }
            else {
                // should be java primitive now
                d = s;
            }
            dest.put(i, d);
        }
        return dest;
    }

    public static JSONObject cloneJSONObject(JSONObject source) throws JSONException {
        final JSONObject dest = new JSONObject(source.length());
        final Iterator<Entry<String, Object>> sourceIter = source.entrySet().iterator();
        while(sourceIter.hasNext()) {
            final Entry<String, Object> sourceEntry = sourceIter.next();
            final Object s = sourceEntry.getValue();
            Object d;
            if(s instanceof JSONObject) {
                d = cloneJSONObject((JSONObject)s);
            }
            else if(s instanceof JSONArray) {
                d = cloneJSONArray((JSONArray)s);
            }
            else {
                // should be java primitive now
                d = s;
            }
            dest.put(sourceEntry.getKey(), d);
        }
        return dest;
    }
}
