/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.imagemgr;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.exception.OXException;
import com.openexchange.filemanagement.ManagedFile;
import com.openexchange.filemanagement.ManagedFileManagement;
import com.openexchange.office.tools.osgi.ServiceLookupRegistry;
import com.openexchange.server.ServiceLookup;
import com.openexchange.tools.encoding.Hex;

/**
 * {@link Resource}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 */
public class Resource {

    public static final String RESOURCE_UID_PREFIX = "xid";

    private static final Logger LOG = LoggerFactory.getLogger(Resource.class);
    private static final String idPrefix = "office.resource0x";

    private static class ResourceInfo {

        public ManagedFile managedFile;
        public String uid;

        public ResourceInfo(final ManagedFile managedFile, String uid) {
            this.managedFile = managedFile;
            this.uid = uid;
        }
    }

    /**
     * Initializes a new {@link Resource}.
     *
     * @param services
     * @param buffer
     */
    public Resource(ServiceLookup servicesDEPRECATED, byte[] buffer) {
        this(null, getSHA256(buffer), buffer);
    }

    /**
     * Initializes a new {@link Resource}.
     *
     * @param services If set, a <link>ManagedFile</Link> will be created to hold the content
     * @param _buffer The content of this resource as a byte Array
     */
    public Resource(ServiceLookup servicesDEPRECATED, String sha256, byte[] buffer) {
        if (null != buffer) {
            m_uid = Resource.getUIDFromData(sha256, buffer);

            final ManagedFileManagement fileMgr = ServiceLookupRegistry.get().getService(ManagedFileManagement.class);

            if (null != fileMgr) {
                // first try to find out, if we already have the managed file
                try {
                    final String managedId = getManagedIdFromUID(m_uid);
                    if (fileMgr.contains(managedId))
                        m_managedFile = fileMgr.getByID(managedId);
                } catch (OXException e) {
                    // nothing to do - exception means managed file is not known
                }

                if (null == m_managedFile) {
                    // create a new managed file
                    try {
                        final InputStream inputStream = new ByteArrayInputStream(buffer);
                        m_managedFile = fileMgr.createManagedFile(getManagedIdFromUID(m_uid), inputStream, true);
                        IOUtils.closeQuietly(inputStream);
                    } catch (OXException e) {
                        LOG.error(e.getMessage(), e);
                    }
                }
            }
        }
    }

    /**
     * Initializes a new {@link Resource}.
     *
     * @param services
     * @param id The id of the distributed managed file
     */
    public Resource(ServiceLookup servicesDEPRECATED, String id) {
        try {
            final ResourceInfo aResInfo = Resource.createResourceInfoIfExists(id);
            if (null != aResInfo) {
                m_managedFile = aResInfo.managedFile;
                m_uid = aResInfo.uid;
            }
        } catch (OXException e) {
            LOG.error(e.getMessage(), e);
        }
    }

    /**
     * Initializes a new {@link Resource using an
     * initialized ResourceInfo object.
     *
     * @param aResourceInfo a valid ResourceInfo object to initialize this Resource instance
     */
    private Resource(ResourceInfo aResourceInfo) {
        m_managedFile = aResourceInfo.managedFile;
        m_uid = aResourceInfo.uid;
    }

    /**
     * Creates a ResourceInfo object which contains the mandatory
     * data to describe a unique managed file to be used by the
     * ResourceManager.
     *
     * @param id The unique resource identifier used by the ManagedFileManagement.
     * @return A ResourceInfo object or null if no object could be created
     * @throws OXException A ManagedFileExceptionErrorMessage.NOT_FOUND is thrown if
     *             the object couldn't found in the ManagedFileManagement.
     */
    static private ResourceInfo createResourceInfoIfExists(String id) throws OXException {
        ResourceInfo result = null;
        String uid = null;

        final ManagedFileManagement fileMgr = ServiceLookupRegistry.get().getService(ManagedFileManagement.class);
        uid = Resource.getUIDFromManagedId(id);

        if ((null != fileMgr) && (null != uid) && (fileMgr.contains(id))) {
            ManagedFile aManagedFile = null;

            if (null != (aManagedFile = fileMgr.getByID(id))) {
                result = new ResourceInfo(aManagedFile, uid);
            }
        }

        return result;
    }

    /**
     * @return The content of this resource as a byte array
     */
    public byte[] getBuffer() {
        byte[] ret = null;

        if (null == m_managedFile) {
            ret = m_buffer;
        } else {
            try {
                final InputStream inputStream = m_managedFile.getInputStream();

                if (null != inputStream) {
                    try {
                        ret = IOUtils.toByteArray(inputStream);
                    } catch (IOException e) {
                        LOG.error(e.getMessage(), e);
                    } finally {
                        IOUtils.closeQuietly(inputStream);
                    }
                }
            } catch (OXException e) {
                LOG.error(e.getMessage());
            }
        }

        return ret;
    }

    /**
     * @param data
     * @return
     */
    public static String getSHA256(byte[] data) {
        String result = null;
        try {
            result = Hex.toHexString(MessageDigest.getInstance("SHA-256").digest(data));
        } catch (NoSuchAlgorithmException e) {
            LOG.error("Not possible to determine SHA-256 hash from data", e);
        }

        return result;
    }

    /**
     * @return The UID of the contained data
     */
    public String getUID() {
        return m_uid;
    }

    /**
     * @return The id of the distributed managed file or null if the id is not valid
     */
    public String getManagedId() {
        return (null != m_uid) ? getManagedIdFromUID(getUID()) : null;
    }

    /**
     * @return The resource identifier of this resource as a string according to the following schema: "xid" + HexValueString
     */
    public String getResourceName() {
        return getResourceNameFromUID(m_uid);
    }

    /**
     *
     */
    public void touch() {
        if (null != m_managedFile) {
            m_managedFile.touch();
        }
    }

    /**
     *
     */
    public void delete() {
        if (null != m_managedFile) {
            m_managedFile.delete();
        }
    }

    /**
     * @param crc32Key
     * @return
     */
    static public String getManagedIdFromUID(String hash) {
        return (idPrefix + hash);
    }

    /**
     * @param id
     * @return
     */
    static public String getUIDFromManagedId(String managedFileId) {
        String uid = null;

        if (StringUtils.isNotEmpty(managedFileId) && managedFileId.startsWith(idPrefix)) {
            uid = managedFileId.substring(idPrefix.length());
        }

        return uid;
    }

    /**
     * @param services
     * @param id
     * @return The distributed managed file resource or null if no resource with this id exists
     */
    static public Resource createFrom(ServiceLookup servicesDEPRECATED, String id) {
        Resource result = null;

        try {
            final ResourceInfo aResInfo = Resource.createResourceInfoIfExists(id);

            if (null != aResInfo) {
                result = new Resource(aResInfo);
            }
        } catch (final OXException e) {
            // nothing to do - resource could not be created
        }

        return result;
    }

    /**
     * @param uid The uid of a resource
     * @return The resource identifier for the given uid as a string according to the
     *         following schema: "xid" + HexValueString, or null in case of an invalid uid
     */
    static public String getResourceNameFromUID(String hash) {
        return ((null != hash) ? (RESOURCE_UID_PREFIX + hash) : null);
    }

    /**
     * @param resourceName A resource name with the folllowing schema: "xid" + HexValueString
     * @return The uid of a resource name, or 0 in case of an invalid resource name
     */
    static public String getUIDFromResourceName(String resourceName) {
        return ((null != resourceName) ? resourceName.substring(RESOURCE_UID_PREFIX.length()) : null);
    }

    /**
     * @param sha256
     * @param resdata
     * @return
     */
    public static String getUIDFromData(String sha256, byte[] resdata) {
        return sha256;
    }

    // - Members ---------------------------------------------------------------

    private ManagedFile m_managedFile = null;

    private byte[] m_buffer = null;

    private String m_uid = null;
}
