/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.imagemgr.impl;

import java.util.Hashtable;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import com.openexchange.office.imagemgr.IResourceManager;
import com.openexchange.office.imagemgr.Resource;
import com.openexchange.office.tools.logging.v2.ELogLevel;
import com.openexchange.office.tools.logging.v2.impl.Logger;
import com.openexchange.office.tools.logging.v2.impl.Slf4JLogger;
import com.openexchange.office.tools.osgi.ServiceLookupRegistryService;
import com.openexchange.tools.encoding.Base64;


/**
 * {@link ResourceManagerImpl}
 *
 * @author <a href="mailto:sven.jacobi@open-xchange.com">Sven Jacobi</a>
 */
public class ResourceManagerImpl implements IResourceManager {

    static protected Logger LOG = Slf4JLogger.create(ResourceManagerImpl.class);
    private final Map<String, Resource> m_resources = new ConcurrentHashMap<>();

    /**
     * Initializes a new {@link ResourceManagerImpl}.
     *
     * @param services
     */
    private ResourceManagerImpl() {
    }

    /**
     * Creates a new instance of a ResourceManager
     * @return
     */
    public static ResourceManagerImpl createInstance() {
        return new ResourceManagerImpl();
    }

    /**
     *
     */
    void touch() {
        synchronized (m_resources) {
            for (final Iterator<Resource> resourceIter = m_resources.values().iterator(); resourceIter.hasNext();) {
                resourceIter.next().touch();
            }
        }
    }

    /**
     * @param lockResources
     */
    @Override
    public void lockResources(boolean lock) {
        final ResourceLocker aResourceLocker = ServiceLookupRegistryService.getInstance().getService(ResourceLocker.class);

        if (null != aResourceLocker) {
        	LOG.forLevel(ELogLevel.E_TRACE).withMessage("lockResources with lock=" + lock + " for instance=" + this).log();
            aResourceLocker.lockResourceManager(this, lock);
        }
    }

    /**
     * @param base64data
     * @return
     */
    @Override
    public String addBase64(String base64Data) {
        String uid = null;

        final String basePattern = "base64,";
        int pos = base64Data.indexOf(basePattern);
        if ((pos != -1) && ((pos += basePattern.length()) < (base64Data.length() - 1))) {
            final byte[] byteData = Base64.decode(base64Data.substring(pos));
            if (byteData != null && byteData.length > 0) {
                uid = addResource(byteData);
            }
        }
        return uid;
    }

    /**
     * @param resdata
     * @return
     */
    @Override
    public String addResource(byte[] resourceData) {
        return addResource(new Resource(null, resourceData));
    }

    /**
     * @param crc32 crc32 of resdata
     * @param resdata
     * @return the unique key: crc32 | (vector index << 32)
     */
    @Override
    public String addResource(String sha256, byte[] resourceData) {
        return addResource(new Resource(null, sha256, resourceData));
    }

    /**
     * @param resource
     * @return the unique key: crc32 | (vector index << 32)
     */
    @Override
    public String addResource(Resource resource) {
        final String uidObject = (null != resource) ? resource.getUID() : null;

        if (null != uidObject) {
            synchronized (m_resources) {
                if (!m_resources.containsKey(uidObject)) {
                    m_resources.put(uidObject, resource);
                }
            }
        }

        return ((null != uidObject) ? uidObject : null);
    }

    /**
    *
    */
    @Override
    public void removeResource(String uid) {
        synchronized (m_resources) {
            m_resources.remove(uid);
        }
    }

    @Override
    public boolean containsResource(String uid) {
        synchronized (m_resources) {
            return m_resources.containsKey(uid);
        }
    }

    @Override
    public Resource getResource(String uid) {
        synchronized (m_resources) {
            return m_resources.get(uid);
        }
    }

    @Override
    public byte[] getResourceBuffer(String uid) {
        Resource resource = null;

        synchronized (m_resources) {
            resource = m_resources.get(uid);
        }

        return (null != resource) ? resource.getBuffer() : null;
    }

    /**
     * @param id
     * @return the resource for the given ManagedFile id or null if the resource couldn't be found
     */
    @Override
    public Resource createManagedResource(String managedResourceId) {
        return Resource.createFrom(null, managedResourceId);
    }

    /**
     * @return
     */
    @Override
    public Hashtable<String, byte[]> getByteArrayTable() {
        Hashtable<String, byte[]> ret = null;

        synchronized (m_resources) {
            ret = new Hashtable<>(m_resources.size());

            for (final String curKey : m_resources.keySet()) {
                final Resource curResource = m_resources.get(curKey);

                if (null != curResource) {
                    ret.put(curKey, curResource.getBuffer());
                }
            }
        }

        return ret;
    }
}
