/*
*
*    OPEN-XCHANGE legal information
*
*    All intellectual property rights in the Software are protected by
*    international copyright laws.
*
*
*    In some countries OX, OX Open-Xchange, open xchange and OXtender
*    as well as the corresponding Logos OX Open-Xchange and OX are registered
*    trademarks.
*    The use of the Logos is not covered by the GNU General Public License.
*    Instead, you are allowed to use these Logos according to the terms and
*    conditions of the Creative Commons License, Version 2.5, Attribution,
*    Non-commercial, ShareAlike, and the interpretation of the term
*    Non-commercial applicable to the aforementioned license is published
*    on the web site http://www.open-xchange.com/EN/legal/index.html.
*
*    Please make sure that third-party modules and libraries are used
*    according to their respective licenses.
*
*    Any modifications to this package must retain all copyright notices
*    of the original copyright holder(s) for the original code used.
*
*    After any such modifications, the original and derivative code shall remain
*    under the copyright of the copyright holder(s) and/or original author(s)per
*    the Attribution and Assignment Agreement that can be located at
*    http://www.open-xchange.com/EN/developer/. The contributing author shall be
*    given Attribution for the derivative code and a license granting use.
*
*     Copyright (C) 2016-2020 OX Software GmbH
*     Mail: info@open-xchange.com
*
*
*     This program is free software; you can redistribute it and/or modify it
*     under the terms of the GNU General Public License, Version 2 as published
*     by the Free Software Foundation.
*
*     This program is distributed in the hope that it will be useful, but
*     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
*     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
*     for more details.
*
*     You should have received a copy of the GNU General Public License along
*     with this program; if not, write to the Free Software Foundation, Inc., 59
*     Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
*/
package com.openexchange.office.hazelcast.serialization.document;

import java.io.IOException;
import java.util.Date;
import java.util.HashSet;
import java.util.Set;

import org.apache.commons.lang.Validate;

import com.hazelcast.nio.serialization.ClassDefinition;
import com.hazelcast.nio.serialization.ClassDefinitionBuilder;
import com.hazelcast.nio.serialization.Portable;
import com.hazelcast.nio.serialization.PortableReader;
import com.hazelcast.nio.serialization.PortableWriter;
import com.openexchange.hazelcast.serialization.CustomPortable;
import com.openexchange.office.tools.directory.DocRestoreID;
import com.openexchange.office.tools.directory.DocumentResources;

/**
* {@link PortableDocumentResources} - Efficient serialization of default {@link Presence} fields via {@link Portable} mechanism. Use
* {@link PortableDocumentResources#getPresence()} to reconstruct the serialized Presence via the {@link Presence.Builder}.
*
* @author <a href="mailto:marc.arens@open-xchange.com">Carsten Driesner</a>
* @since 7.8.1
*/
public class PortableDocumentResources implements CustomPortable {

    public static final int CLASS_ID = 204;
    public static ClassDefinition CLASS_DEFINITION = null;

    private PortableRestoreID restoreId;
    private static final String FIELD_DOCRESTOREID = "docRestoreId";

    private long timeStamp;
    private static final String FIELD_TIMESTAMP = "timeStamp";

    private String uniqueInstanceId;
    private static final String FIELD_UNIQUEINSTANCEID = "uniqueInstanceId";

    private Set<String> resourceList;
    private static final String FIELD_RESOURCELIST = "resourceList";

    static {
        CLASS_DEFINITION = new ClassDefinitionBuilder(FACTORY_ID, CLASS_ID)
        .addPortableField(FIELD_DOCRESTOREID, PortableRestoreID.CLASS_DEFINITION)
        .addLongField(FIELD_TIMESTAMP)
        .addUTFField(FIELD_UNIQUEINSTANCEID)
        .addPortableArrayField(FIELD_RESOURCELIST, PortableResourceID.CLASS_DEFINITION)
        .build();
    }

    public PortableDocumentResources() {
        super();
    }

    public PortableDocumentResources(final DocumentResources doc) {
        Validate.notNull(doc, "Mandatory argument missing: doc");
        this.restoreId = new PortableRestoreID(doc.getDocRestoreId());
        this.resourceList = doc.getResources();
        this.uniqueInstanceId = doc.getUniqueInstanceId();
    }

    @Override
    public void writePortable(PortableWriter writer) throws IOException {
        writer.writePortable(FIELD_DOCRESTOREID, this.restoreId);
        writer.writeLong(FIELD_TIMESTAMP, this.timeStamp);
        writer.writeUTF(FIELD_UNIQUEINSTANCEID, this.uniqueInstanceId);
        if (null != resourceList) {
            Set<PortableResourceID> portableResIds = new HashSet<PortableResourceID>(resourceList.size());
            for (String s : resourceList) {
                portableResIds.add(new PortableResourceID(s));
            }
            writer.writePortableArray(FIELD_RESOURCELIST, portableResIds.toArray(new Portable[portableResIds.size()]));
        } else {
            writer.writePortableArray(FIELD_RESOURCELIST, null);
        }
    }

    @Override
    public void readPortable(PortableReader reader) throws IOException {
        restoreId = reader.readPortable(FIELD_DOCRESTOREID);
        timeStamp = reader.readLong(FIELD_TIMESTAMP);
        uniqueInstanceId = reader.readUTF(FIELD_UNIQUEINSTANCEID);
        Portable[] portables = reader.readPortableArray(FIELD_RESOURCELIST);
        if (null != portables) {
            if (null == resourceList) { resourceList = new HashSet<String>(); }
            for (Portable portable : portables) {
                if (portable instanceof PortableResourceID) {
                    PortableResourceID portResId = (PortableResourceID)portable;
                    resourceList.add(portResId.toString());
                }
            }
        }
    }

    @Override
    public int getFactoryId() {
        return FACTORY_ID;
    }

    @Override
    public int getClassId() {
        return CLASS_ID;
    }

    public DocRestoreID getRestoreId() {
        return DocRestoreID.create(this.restoreId.getResourceID(), this.restoreId.getCreated());
    }

    public void setRestoreId(DocRestoreID restoreId) {
        this.restoreId = new PortableRestoreID(restoreId);
    }

    public long getTimeStamp() {
        return this.timeStamp;
    }

    public String getUniqueInstanceId() {
        return this.uniqueInstanceId;
    }

    public void touch() {
        this.timeStamp = new Date().getTime();
    }

    public Set<String> getResourceList() {
        return this.resourceList;
    }

    @Override
    public String toString() {
        final StringBuffer strBuf = new StringBuffer();

        boolean first = true;
        if (null != resourceList) {
            for (String s : resourceList) {
                if (!first) { strBuf.append(","); }
                strBuf.append(s);
                first = false;
            }
        }

        return "PortableDocumentResources [restoreId=" + this.restoreId + ", timestamp=" + this.timeStamp + ", uniqueID=" + this.uniqueInstanceId + ", resources=[" + strBuf.toString() + "]" + "]";
    }

    public static DocumentResources createFrom(final PortableDocumentResources portableDocResources) {
        DocumentResources docResources = null;

        if (null != portableDocResources) {
            docResources = new DocumentResources(
                    portableDocResources.getRestoreId(),
                    portableDocResources.getTimeStamp(),
                    portableDocResources.getUniqueInstanceId(),
                    portableDocResources.getResourceList());
        }
        return docResources;
    }
}
