/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.filter.ooxml.components;

import java.util.HashMap;
import java.util.Map;
import org.docx4j.vml.CTArc;
import org.docx4j.vml.CTCurve;
import org.docx4j.vml.CTGroup;
import org.docx4j.vml.CTImage;
import org.docx4j.vml.CTLine;
import org.docx4j.vml.CTOval;
import org.docx4j.vml.CTPath;
import org.docx4j.vml.CTPolyLine;
import org.docx4j.vml.CTRect;
import org.docx4j.vml.CTRoundRect;
import org.docx4j.vml.CTShape;
import org.docx4j.vml.CTShapetype;
import org.docx4j.vml.CTTextbox;
import org.docx4j.vml.STStrokeJoinStyle;
import org.docx4j.vml.STTrueFalse;
import org.docx4j.vml.officedrawing.STConnectType;
import org.docx4j.vml.officedrawing.STInsetMode;
import org.docx4j.vml.spreadsheetDrawing.CTClientData;
import org.docx4j.vml.spreadsheetDrawing.STObjectType;
import org.json.JSONObject;
import com.openexchange.office.filter.core.DLNode;
import com.openexchange.office.filter.core.component.Component;
import com.openexchange.office.filter.core.component.ComponentContext;
import com.openexchange.office.filter.core.IContentAccessor;
import com.openexchange.office.filter.core.DLList;
import com.openexchange.office.filter.ooxml.OfficeOpenXMLOperationDocument;

public class VMLRootComponent extends VMLBase {

     public VMLRootComponent(String documentType, OfficeOpenXMLOperationDocument operationDocument, IContentAccessor contentAccessor) {
        super(documentType, operationDocument, contentAccessor);
        operationDocument.setVMLShapeTypes(new HashMap<String, CTShapetype>());
     }

     @Override
     public Component<OfficeOpenXMLOperationDocument> getNextChildComponent(ComponentContext<OfficeOpenXMLOperationDocument> previousChildContext, Component<OfficeOpenXMLOperationDocument> previousChildComponent) {
         final DLNode<Object> paragraphNode = getNode();
         final DLList<Object> nodeList = (DLList<Object>)((IContentAccessor)paragraphNode.getData()).getContent();
         final int nextComponentNumber = previousChildComponent!=null?previousChildComponent.getNextComponentNumber():0;
         DLNode<Object> childNode = previousChildContext!=null ? previousChildContext.getNode().getNext() : nodeList.getFirstNode();

         OfficeOpenXMLComponent nextComponent = null;
         for(; nextComponent==null&&childNode!=null; childNode = childNode.getNext()) {
             final Object o = getContentModel(childNode, paragraphNode.getData());
             if(o instanceof CTShapetype) {
                 final String id = ((CTShapetype)o).getVmlId();
                 if(id!=null&&!id.isEmpty()) {
                     getOperationDocument().getVMLShapeTypes().put(id, (CTShapetype)o);
                 }
             }
             else if(o instanceof CTGroup) {
                 nextComponent = new VMLGroup(documentType, this, childNode, nextComponentNumber);
             }
             else if(o instanceof CTShape) {
                 nextComponent = new VMLShape(documentType, this, childNode, nextComponentNumber);
             }
             else if(o instanceof CTArc) {
                 nextComponent = new VMLShape(documentType, this, childNode, nextComponentNumber);
             }
             else if(o instanceof CTCurve) {
                 nextComponent = new VMLShape(documentType, this, childNode, nextComponentNumber);
             }
             else if(o instanceof CTImage) {
                 nextComponent = new VMLShape(documentType, this, childNode, nextComponentNumber);
             }
             else if(o instanceof CTLine) {
                 nextComponent = new VMLShape(documentType, this, childNode, nextComponentNumber);
             }
             else if(o instanceof CTOval) {
                 nextComponent = new VMLShape(documentType, this, childNode, nextComponentNumber);
             }
             else if(o instanceof CTPolyLine) {
                 nextComponent = new VMLShape(documentType, this, childNode, nextComponentNumber);
             }
             else if(o instanceof CTRect) {
                 nextComponent = new VMLShape(documentType, this, childNode, nextComponentNumber);
             }
             else if(o instanceof CTRoundRect) {
                 nextComponent = new VMLShape(documentType, this, childNode, nextComponentNumber);
             }
         }
         return nextComponent;
     }

     @Override
     public Object getChild() {
         return getNextChildComponent(null, null);
     }

     @Override
     public Object insertChild(int number, Type type) {
         try {
             return insertChildComponent(number, null, type);
         }
         catch(Exception e) {
             return null;
         }
     }

     @Override
     public Component<OfficeOpenXMLOperationDocument> insertChildComponent(ComponentContext<OfficeOpenXMLOperationDocument> parentContext, DLNode<Object> contextNode, int number, Component<OfficeOpenXMLOperationDocument> child, Component.Type type, JSONObject attrs) {

         final DLList<Object> DLList = (DLList<Object>)((IContentAccessor)contextNode.getData()).getContent();
         final DLNode<Object> referenceNode = child!=null ? child.getNode() : null;

         switch(type) {
             case AC_COMMENT : {
                 final Map<String, CTShapetype> shapeTypes = getOperationDocument().getVMLShapeTypes();
                 if(!shapeTypes.containsKey("_x0000_t202")) {
                     final CTShapetype shapeType = new CTShapetype();
                     shapeType.setCoordsize("21600,21600");
                     shapeType.setSpt(202);
                     shapeType.setPath("m,l,21600r21600,l21600,xe");
                     shapeType.setVmlId("_x0000_t202");
                     shapeType.getStroke(true).setJoinstyle(STStrokeJoinStyle.MITER);
                     final CTPath path = shapeType.getPathElement(true);
                     path.setGradientshapeok(STTrueFalse.TRUE);
                     path.setConnecttype(STConnectType.RECT);
                     DLList.addNode(referenceNode, new DLNode<Object>(shapeType), true);
                     shapeTypes.put(shapeType.getVmlId(), shapeType);
                 }
                 final CTShape newShape = new CTShape();
                 newShape.setParent(getObject());
                 newShape.setType("#_x0000_t202");
                 newShape.setStyle("position:absolute;margin-left:251.25pt;margin-top:52.5pt;width:96pt;height:64.5pt;visibility:visible");
                 newShape.setFillcolor("infoBackground [80]");
                 newShape.setStrokecolor("none [81]");
                 newShape.setInsetmode(STInsetMode.AUTO);
                 newShape.getFill(true).setColor2("infoBackground [80]");
                 newShape.getPathElement(true).setConnecttype(STConnectType.NONE);
                 final CTTextbox textBox = newShape.getTextBox(true);
                 textBox.setStyle("mso-direction-alt:auto");
                 final CTClientData clientData = newShape.getClientData(true);
                 clientData.setObjectType(STObjectType.NOTE);
                 clientData.setAutoFill("False");
                 clientData.setVisible(new String());
                 final DLNode<Object> newChildNode = new DLNode<Object>(newShape);
                 DLList.addNode(referenceNode, newChildNode, true);
                 return new VMLShape(documentType, parentContext, newChildNode, number);
             }
             case AC_GROUP: {
                 final CTGroup newGroup = new CTGroup();
                 newGroup.setParent(getObject());
                 newGroup.setVmlId("Group 1");
                 newGroup.setStyle("position:absolute;width:100;height:100");
                 newGroup.setSpid(operationDocument.getPackage().getNextMarkupId().toString());
                 final DLNode<Object> newChildNode = new DLNode<Object>(newGroup);
                 DLList.addNode(referenceNode, newChildNode, true);
                 return new VMLGroup(documentType, parentContext, newChildNode, number);
             }
             case AC_CONNECTOR:
                 // PASSTHROUGH INTENDED
             case AC_SHAPE: {
                 final CTRect newShape = new CTRect();
                 newShape.setParent(getObject());
                 newShape.setVmlId("Rectangle 1");
                 newShape.setStyle("position:absolute;width:100;height:100;visibility:visible;mso-wrap-style:square;v-text-anchor:middle");
                 newShape.setFilled(org.docx4j.vml.STTrueFalse.F);
                 newShape.setStroked(org.docx4j.vml.STTrueFalse.F);
                 newShape.setStrokeweight("1pt");
                 newShape.setSpid(operationDocument.getPackage().getNextMarkupId().toString());
                 final DLNode<Object> newChildNode = new DLNode<Object>(newShape);
                 DLList.addNode(referenceNode, newChildNode, true);
                 return new VMLShape(documentType, parentContext, newChildNode, number);
             }
             case AC_IMAGE: {
                 final CTImage newImage = new CTImage();
                 newImage.setParent(getObject());
                 newImage.setVmlId("Image 1");
                 newImage.setStyle("position:absolute;width:100;height:100;visibility:visible;mso-wrap-style:square");
                 newImage.setSpid(operationDocument.getPackage().getNextMarkupId().toString());
                 final DLNode<Object> newChildNode = new DLNode<Object>(newImage);
                 DLList.addNode(referenceNode, newChildNode, true);
                 return new VMLShape(documentType, parentContext, newChildNode, number);
             }
             default : {
                 throw new UnsupportedOperationException();
             }
         }
     }

    @Override
    public void applyAttrsFromJSON(JSONObject attrs) {
        // the VMLRoot object is not having attributes, instead it is used only to access vml childs
    }

    @Override
    public JSONObject createJSONAttrs(JSONObject attrs) {
        // the VMLRoot object is not having attributes, instead it is used only to access vml childs
        return attrs;
    }
}
