/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.filter.ooxml.components;

import java.util.ArrayList;
import java.util.List;
import org.docx4j.wml.CTTxbxContent;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.office.filter.core.DLNode;
import com.openexchange.office.filter.core.SplitMode;
import com.openexchange.office.filter.core.component.Component;
import com.openexchange.office.filter.core.component.ComponentContext;
import com.openexchange.office.filter.ooxml.OfficeOpenXMLOperationDocument;

public class MultiComponent extends OfficeOpenXMLComponent implements IParagraph, ITable, IRow {

	final private List<OfficeOpenXMLComponent> componentList;

	public MultiComponent(OfficeOpenXMLOperationDocument operationDocument, List<OfficeOpenXMLComponent> componentList) {
		super(operationDocument, new DLNode<Object>(componentList), 0);

		this.componentList = componentList;
	}
	public List<OfficeOpenXMLComponent> getComponentList() {
		return componentList;
	}
	@Override
	public OfficeOpenXMLComponent getComponent(int number) {
		final List<OfficeOpenXMLComponent> nextComponents = new ArrayList<OfficeOpenXMLComponent>(componentList.size());
		for(OfficeOpenXMLComponent component:componentList) {
			final OfficeOpenXMLComponent nextComponent = (OfficeOpenXMLComponent)component.getComponent(number);
			if(nextComponent!=null) {
				nextComponents.add(nextComponent);
			}
		}
		return createComponent(operationDocument, nextComponents);
    }
	@Override
    public int getComponentNumber() {
        return componentList.get(0).getComponentNumber();
    }
    @Override
    public int getNextComponentNumber() {
		return componentList.get(0).getNextComponentNumber();
    }
	@Override
	public OfficeOpenXMLComponent getNextComponent() {
		final List<OfficeOpenXMLComponent> nextComponents = new ArrayList<OfficeOpenXMLComponent>(componentList.size());
		for(OfficeOpenXMLComponent component:componentList) {
			final OfficeOpenXMLComponent nextComponent = (OfficeOpenXMLComponent)component.getNextComponent();
			if(nextComponent!=null) {
				nextComponents.add(nextComponent);
			}
		}
		return createComponent(operationDocument, nextComponents);
	}
	@Override
	public Component<OfficeOpenXMLOperationDocument> getNextChildComponent(ComponentContext<OfficeOpenXMLOperationDocument> previousChildContext, Component<OfficeOpenXMLOperationDocument> previousChildComponent) {
		final List<OfficeOpenXMLComponent> childComponents = new ArrayList<OfficeOpenXMLComponent>(componentList.size());
		for(OfficeOpenXMLComponent component:componentList) {
			final OfficeOpenXMLComponent childComponent = (OfficeOpenXMLComponent)component.getNextChildComponent(previousChildContext, previousChildComponent);
			if(childComponent!=null) {
				childComponents.add(childComponent);
			}
		}
		return createComponent(operationDocument, childComponents);
	}
	@Override
	public Component<OfficeOpenXMLOperationDocument> insertChildComponent(ComponentContext<OfficeOpenXMLOperationDocument> parentContext, DLNode<Object> contextNode, int number, Component<OfficeOpenXMLOperationDocument> child, Component.Type type, JSONObject attrs) {
		return null;
	}
	@Override
    public Component<OfficeOpenXMLOperationDocument> insertChildComponent(int number, JSONObject attrs, Component.Type type) throws Exception {
	    final List<OfficeOpenXMLComponent> childComponents = new ArrayList<OfficeOpenXMLComponent>(componentList.size());
		for(OfficeOpenXMLComponent component:componentList) {
			childComponents.add((OfficeOpenXMLComponent)component.insertChildComponent(number, attrs, type));
		}
		return new MultiComponent(operationDocument, childComponents);
	}
	@Override
	public void applyAttrsFromJSON(JSONObject attrs) throws Exception {

		for(OfficeOpenXMLComponent component:componentList) {
			component.applyAttrsFromJSON(attrs);
		}
	}
	@Override
	public JSONObject createJSONAttrs(JSONObject attrs) throws Exception {
		return componentList.get(0).createJSONAttrs(attrs);
	}
	@Override
	public void delete(int count) {
		for(OfficeOpenXMLComponent component:componentList) {
	        component.delete(count);
	    }
	}
	@Override
    public void splitStart(int n, SplitMode splitMode) {
		for(OfficeOpenXMLComponent component:componentList) {
			component.splitStart(n, splitMode);
		}
    }
	@Override
    public void splitEnd(int n, SplitMode splitMode) {
		for(OfficeOpenXMLComponent component:componentList) {
			component.splitEnd(n, splitMode);
		}
    }
	// IParagraph interfaces
	@Override
	public void insertText(int textPosition, String text, JSONObject attrs) throws Exception {
		for(OfficeOpenXMLComponent component:componentList) {
			((IParagraph)component).insertText(textPosition, text, attrs);
		}
	}
	@Override
    public void splitParagraph(int textPosition) {
		for(OfficeOpenXMLComponent component:componentList) {
			((IParagraph)component).splitParagraph(textPosition);
		}
	}
	@Override
    public void mergeParagraph() {
		for(OfficeOpenXMLComponent component:componentList) {
			((IParagraph)component).mergeParagraph();
		}
	}
	// ITable interface
	@Override
	public void insertRows(int rowPosition, int count, boolean insertDefaultCells, int referenceRow, JSONObject attrs) throws Exception {
	    for(OfficeOpenXMLComponent component:componentList) {
			((ITable)component).insertRows(rowPosition, count, insertDefaultCells, referenceRow, attrs);
		}
	}
	@Override
	public void splitTable(int rowPosition) {
		for(OfficeOpenXMLComponent component:componentList) {
			((ITable)component).splitTable(rowPosition);
		}
	}
	@Override
	public void mergeTable() {
		for(OfficeOpenXMLComponent component:componentList) {
			((ITable)component).mergeTable();
		}
	}
	@Override
	public void insertColumn(JSONArray tableGrid, int gridPosition, String insertMode)
		throws JSONException {

		for(OfficeOpenXMLComponent component:componentList) {
			((ITable)component).insertColumn(tableGrid, gridPosition, insertMode);
		}
	}
	@Override
    public void deleteColumns(int gridStart, int gridEnd)
    	throws JSONException {

		for(OfficeOpenXMLComponent component:componentList) {
			((ITable)component).deleteColumns(gridStart, gridEnd);
		}
	}
	// IRow interface
	@Override
	public void insertCells(int cellPosition, int count, JSONObject attrs) throws Exception {
		for(OfficeOpenXMLComponent component:componentList) {
			((IRow)component).insertCells(cellPosition, count, attrs);
		}
	}

	public static OfficeOpenXMLComponent createComponent(OfficeOpenXMLOperationDocument operationDocument, List<OfficeOpenXMLComponent> componentList) {
	    if(componentList.isEmpty()) {
	        return null;
	    }
	    if(componentList.size()==1) {
	        return componentList.get(0);
	    }
	    final OfficeOpenXMLComponent a = componentList.get(0);
	    final OfficeOpenXMLComponent b = componentList.get(1);
	    if(a instanceof IShapeProvider && b instanceof IShapeProvider) {
	        final IShape shapea = ((IShapeProvider)a).getShape();
            final IShape shapeb = ((IShapeProvider)b).getShape();
            if(shapea instanceof IShapeTxBxContentAccessor && shapeb instanceof IShapeTxBxContentAccessor) {
    	        final CTTxbxContent txa = ((IShapeTxBxContentAccessor)shapea).getTextboxContent(false);
    	        final CTTxbxContent txb = ((IShapeTxBxContentAccessor)shapeb).getTextboxContent(false);
    	        if(txa!=null&&txb!=null) {
    	            // provide one instance for text only
    	            ((IShapeTxBxContentAccessor)shapeb).setTextboxContent(txa);
    	            return a;
    	        }
            }
	    }
	    return new MultiComponent(operationDocument, componentList);
	}
}
