/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.filter.ooxml.xlsx.components;

import java.util.HashMap;
import org.docx4j.dml.spreadsheetdrawing.AnchorBase;
import org.docx4j.dml.spreadsheetdrawing.CTGraphicalObjectFrame;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.office.filter.api.OCKey;
import com.openexchange.office.filter.core.DLNode;
import com.openexchange.office.filter.core.component.Component;
import com.openexchange.office.filter.core.component.ComponentContext;
import com.openexchange.office.filter.ooxml.OfficeOpenXMLOperationDocument;
import com.openexchange.office.filter.ooxml.components.IShapeType;
import com.openexchange.office.filter.ooxml.components.ShapeType;
import com.openexchange.office.filter.ooxml.drawingml.ChartWrapper;
import com.openexchange.office.filter.ooxml.drawingml.DMLGraphic;
import com.openexchange.office.filter.ooxml.drawingml.DMLHelper;
import com.openexchange.office.filter.ooxml.xlsx.tools.Drawings;

public class ShapeGraphicComponent extends XlsxComponent implements IShapeType {

	final private CTGraphicalObjectFrame graphicalObjectFrame;
	private AnchorBase cellAnchor;

	private ShapeType shapeType = ShapeType.UNDEFINED;

	final HashMap<String, Object> drawingProperties = new HashMap<String, Object>();

	public ShapeGraphicComponent(ComponentContext<OfficeOpenXMLOperationDocument> parentContext, DLNode<Object> _node, CTGraphicalObjectFrame graphicalObjectFrame, AnchorBase cellAnchor, int _componentNumber) {
        super(parentContext, _node, _componentNumber);

        this.graphicalObjectFrame = graphicalObjectFrame;
        this.cellAnchor = cellAnchor;

        DMLGraphic.getGraphicProperties(operationDocument, operationDocument.getContextPart(), graphicalObjectFrame.getGraphic(), drawingProperties);
        if(drawingProperties.containsKey(OCKey.TYPE.value())&&drawingProperties.get(OCKey.TYPE.value()).equals("chart")) {
            shapeType = ShapeType.CHART;
        }
	}

	public CTGraphicalObjectFrame getObjectFrame() {
	    return graphicalObjectFrame;
	}

	public HashMap<String, Object> getDrawingProperties() {
	    return drawingProperties;
	}

	@Override
	public Component<OfficeOpenXMLOperationDocument> getNextChildComponent(ComponentContext<OfficeOpenXMLOperationDocument> previousChildContext, Component<OfficeOpenXMLOperationDocument> previousChildComponent) {
		return null;
    }

    @Override
    public void applyAttrsFromJSON(JSONObject attrs) throws Exception {
        if(cellAnchor!=null) {
            final JSONObject drawingAttrs = attrs.optJSONObject(OCKey.DRAWING.value());
            if(drawingAttrs!=null) {
                getNode().setData(Drawings.setDrawingAnchor(drawingAttrs, cellAnchor));
            }
        }
        DMLHelper.applyTransform2DFromJson(graphicalObjectFrame, attrs, cellAnchor!=null);
        DMLHelper.applyNonVisualDrawingProperties(graphicalObjectFrame, attrs.optJSONObject(OCKey.DRAWING.value()));
        DMLHelper.applyNonVisualDrawingShapeProperties(graphicalObjectFrame, attrs.optJSONObject(OCKey.DRAWING.value()));
        if(shapeType==ShapeType.CHART) {
            final JSONObject chartProperties = attrs.optJSONObject(OCKey.CHART.value());
            if(chartProperties!=null) {
                try {
                    Drawings.getChart(drawingProperties).applyChartSpaceProperties(null, chartProperties, attrs.optJSONObject(OCKey.FILL.value()));
                } catch (Exception e) {
                    throw new RuntimeException(e);
                }
            }
        }
    }

    @Override
    public JSONObject createJSONAttrs(JSONObject attrs)
        throws JSONException {

        if(cellAnchor!=null) {
            Drawings.createAnchorOperations(cellAnchor, attrs);
        }
        DMLHelper.createJsonFromTransform2D(attrs, graphicalObjectFrame.getXfrm(false), cellAnchor!=null);
        if(shapeType==ShapeType.CHART) {
            final JSONObject chartProperties = new JSONObject();
            chartProperties.put(OCKey.TYPE.value(), drawingProperties.get(OCKey.CHART_TYPE.value()));
            attrs.put(OCKey.CHART.value(), chartProperties);

            ChartWrapper chart = Drawings.getChart(drawingProperties);
            chart.createJSONAttrs(chartProperties, drawingProperties, attrs);

        }
        return attrs;
    }

	@Override
	public ShapeType getType() {
		return shapeType;
	}
}
