/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.filter.core.component;

import java.util.Set;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.office.filter.core.DLNode;
import com.openexchange.office.filter.core.IContentAccessor;
import com.openexchange.office.filter.core.OperationDocument;
import com.openexchange.office.filter.core.DLList;
import com.openexchange.office.filter.core.component.Child;
import com.openexchange.office.filter.core.component.ComponentContext;
import com.openexchange.office.filter.core.component.IComponent;

public abstract class Component<OpDoc extends OperationDocument> extends ComponentContext<OpDoc> implements IComponent<OpDoc> {

    protected int componentNumber;

    public enum Type {
        PARAGRAPH,
        TABLE,
        TR,
        TC,
        TAB,
        BOOKMARK_START,
        BOOKMARK_END,
        COMMENT_REFERENCE,
        COMMENT_RANGE_START,
        COMMENT_RANGE_END,
        SIMPLE_FIELD,
        COMPLEX_FIELD_START,
        COMPLEX_FIELD_SEPARATE,
        COMPLEX_FIELD_END,
        HARDBREAK_DEFAULT,
        HARDBREAK_PAGE,
        HARDBREAK_COLUMN,
        AC_SHAPE,
        AC_GROUP,
        AC_CHART,
        AC_CONNECTOR,
        AC_IMAGE,
        AC_COMMENT,
        AC_FRAME,
        HARDBREAK,
        FIELD
    }

    public Component(OpDoc operationDocument, DLNode<Object> _node, int _componentNumber) {
        super(operationDocument, _node);
        componentNumber = _componentNumber;
    }

    public Component(ComponentContext<OpDoc> parentContext, DLNode<Object> _node, int _componentNumber) {
        super(parentContext, _node);
        componentNumber = _componentNumber;
    }

    @Override
    public Component<OpDoc> getNextComponent() {
    	if(getParentContext()==null) {
    		return null;
    	}
        ComponentContext<OpDoc> parentContext = this;
        Component<OpDoc> nextComponent = null;
        do {
            ComponentContext<OpDoc> previousContext = parentContext;
            parentContext = parentContext.getParentContext();
            nextComponent = parentContext.getNextChildComponent(previousContext, this);
        }
        while(nextComponent==null&&!(parentContext instanceof Component));
        return nextComponent;
    }

    public ComponentContext<OpDoc> getContext(Class<?> contextClass) {
        ComponentContext<OpDoc> parentContext = getParentContext();
    	while(!(parentContext instanceof Component)&&(parentContext!=null)) {
    		if(parentContext.getClass()==contextClass) {
    			return parentContext;
    		}
    		parentContext = parentContext.getParentContext();
    	}
    	return null;
    }

    // the ComponentContext c is inserted behind one of the classes in the
    // parentContextList, it is inserted at least behind the parent Component,
    // so the parentContextList is allowed to be null
    //
    // the parentContext of c is inserted, so c does not need to have a parentContext
    public void insertContext(ComponentContext<OpDoc> c, Set<Class<?>> parentContextList) {
        ComponentContext<OpDoc> childContext = this;
        ComponentContext<OpDoc> parentContext = getParentContext();
        while(!(parentContext instanceof Component)&&((parentContextList==null)||(!parentContextList.contains(parentContext.getClass())))) {
            childContext = parentContext;
            parentContext = childContext.getParentContext();
        }

        final DLList<Object> parentContent = (DLList<Object>)((IContentAccessor)parentContext.getNode().getData()).getContent();
        parentContent.setNode(childContext.getNode(), c.getNode());
        final DLList<Object> content = (DLList<Object>)((IContentAccessor)c.getNode().getData()).getContent();
        content.addNode(childContext.getNode());
        c.setParentContext(parentContext);
        childContext.setParentContext(c);
        ((Child)c.getNode().getData()).setParent(parentContext.getNode().getData());
        ((Child)childContext.getNode().getData()).setParent(c.getNode().getData());
    }

    public void removeContext(Class<?> contextClass) {
        ComponentContext<OpDoc> childContext = this;
        ComponentContext<OpDoc> parentContext = getParentContext();
        while(!(parentContext instanceof Component)) {
            if(parentContext.getClass()==contextClass) {
            	final ComponentContext<OpDoc> newParentContext = parentContext.getParentContext();
            	((DLList<Object>)((IContentAccessor)newParentContext.getNode().getData()).getContent()).setNodes(parentContext.getNode(), (DLList<Object>)((IContentAccessor)parentContext.getNode().getData()).getContent());
            	childContext.setParentContext(newParentContext);
            	((Child)childContext.getNode().getData()).setParent(newParentContext.getNode().getData());
                break;
            }
            childContext = parentContext;
            parentContext = childContext.getParentContext();
        }
    }

    public ComponentContext<OpDoc> getContextChild(Set<Class<?>> parentContextList) {
        ComponentContext<OpDoc> childContext = this;
        ComponentContext<OpDoc> parentContext = getParentContext();
        while(!(parentContext instanceof Component)&&((parentContextList==null)||(!parentContextList.contains(parentContext.getClass())))) {
            childContext = parentContext;
            parentContext = childContext.getParentContext();
        }
        return childContext;
    }

    @Override
    public Component<OpDoc> getParentComponent() {
    	ComponentContext<OpDoc> parentContext = getParentContext();
    	while(!(parentContext instanceof Component)) {
    		parentContext = parentContext.getParentContext();
    	}
    	return (Component<OpDoc>)parentContext;
    }

    @Override
    public abstract Component<OpDoc> getNextChildComponent(ComponentContext<OpDoc> previousChildContext, Component<OpDoc> previousChildComponent);

    @Override
    public Component<OpDoc> getChildComponent(int childComponentNumber) {
    	Component<OpDoc> c = getNextChildComponent(null, null);
    	if(c!=null) {
    		c = c.getComponent(childComponentNumber);
    	}
    	return c;
    }

    @Override
    public Component<OpDoc> getComponent(JSONArray position, int positionCount) {
        try {
            if(positionCount==0) {
                return this;
            }
            // a part has to implement the ContentAccessor interface, otherwise no component can be accessed
            Component<OpDoc> c = getNextChildComponent(null, null);
            for(int i=0; c!=null&&i<positionCount;) {
                c = c.getComponent(position.getInt(i++));
                if(i!=positionCount) {
                    c = c.getNextChildComponent(null, null);
                }
            }
            return c;
        }
        catch(JSONException e) {
            // ups
        }
        return null;
    }

    @Override
    public Component<OpDoc> getComponent(int number) {
    	Component<OpDoc> c = this;
    	while(c!=null&&c.getNextComponentNumber()<=number) {
    		c = c.getNextComponent();
    	}
    	return c;
    }

    @Override
    public int getComponentNumber() {
        return componentNumber;
    }

    @Override
    public int getNextComponentNumber() {
        return componentNumber + 1;
    }

    public abstract Component<OpDoc> insertChildComponent(ComponentContext<OpDoc> parentContext, DLNode<Object> contextNode, int number, Component<OpDoc> child, Type type, JSONObject attrs) throws Exception;

    public Component<OpDoc> insertChildComponent(int number, JSONObject attrs, Type type) throws Exception {
        final Component<OpDoc> c = insertChildComponent(this, getNode(), number, getChildComponent(number), type, attrs);
        if(attrs!=null) {
            c.applyAttrsFromJSON(attrs);
        }
        return c;
    }

    public abstract void applyAttrsFromJSON(JSONObject attrs) throws Exception;

    public String componentToString() {
        final StringBuffer stringBuffer = new StringBuffer();
        componentToString(stringBuffer);
        return stringBuffer.toString();
    }

    public abstract String simpleName();

    private void componentToString(StringBuffer stringBuffer) {
        stringBuffer.append(simpleName());
        Component<OpDoc> childComponent = getChildComponent(0);
        if(childComponent!=null) {
            stringBuffer.append('(');
            while(childComponent!=null) {
                childComponent.componentToString(stringBuffer);
                childComponent = childComponent.getNextComponent();
                if(childComponent!=null) {
                    stringBuffer.append(',');
                }
            }
            stringBuffer.append(')');
        }
    }
}
