/*
/*
*
*    OPEN-XCHANGE legal information
*
*    All intellectual property rights in the Software are protected by
*    international copyright laws.
*
*
*    In some countries OX, OX Open-Xchange, open xchange and OXtender
*    as well as the corresponding Logos OX Open-Xchange and OX are registered
*    trademarks.
*    The use of the Logos is not covered by the GNU General Public License.
*    Instead, you are allowed to use these Logos according to the terms and
*    conditions of the Creative Commons License, Version 2.5, Attribution,
*    Non-commercial, ShareAlike, and the interpretation of the term
*    Non-commercial applicable to the aforementioned license is published
*    on the web site http://www.open-xchange.com/EN/legal/index.html.
*
*    Please make sure that third-party modules and libraries are used
*    according to their respective licenses.
*
*    Any modifications to this package must retain all copyright notices
*    of the original copyright holder(s) for the original code used.
*
*    After any such modifications, the original and derivative code shall remain
*    under the copyright of the copyright holder(s) and/or original author(s)per
*    the Attribution and Assignment Agreement that can be located at
*    http://www.open-xchange.com/EN/developer/. The contributing author shall be
*    given Attribution for the derivative code and a license granting use.
*
*     Copyright (C) 2016-2020 OX Software GmbH
*     Mail: info@open-xchange.com
*
*
*     This program is free software; you can redistribute it and/or modify it
*     under the terms of the GNU General Public License, Version 2 as published
*     by the Free Software Foundation.
*
*     This program is distributed in the hope that it will be useful, but
*     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
*     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
*     for more details.
*
*     You should have received a copy of the GNU General Public License along
*     with this program; if not, write to the Free Software Foundation, Inc., 59
*     Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
*/

package com.openexchange.office.documents.hazelcast.serialization;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.concurrent.Callable;

import com.hazelcast.nio.serialization.PortableReader;
import com.hazelcast.nio.serialization.PortableWriter;
import com.openexchange.hazelcast.serialization.CustomPortable;
import com.openexchange.office.documents.DocRequest;
import com.openexchange.office.documents.DocumentRequestProcessor;
import com.openexchange.office.tools.osgi.ServiceLookupRegistry;

public class PortableDocRequest implements CustomPortable, Callable<String> {

    //-------------------------------------------------------------------------
    public static final int CLASS_ID = 213;

    //-------------------------------------------------------------------------
    private static String FIELD_DOCREQUEST = "docRequest";

    //-------------------------------------------------------------------------
    private DocRequest docRequest;

    //-------------------------------------------------------------------------
    /**
     * Initializes a new {@link PortableDocRequest}.
     */
    public PortableDocRequest() {
        super();
    }

    //-------------------------------------------------------------------------
    /**
     * Initializes a new {@link PortableDocRequest}.
     *
     * @param cleaningMember The member that started the cleanup.
     * @param memberToClean The member that left the cluster.
     */
    public PortableDocRequest(DocRequest docRequest) {
        super();
        this.docRequest = docRequest;
    }

    //-------------------------------------------------------------------------
    /**
     * Gets the cleaningMember
     *
     * @return The cleaningMember
     */
    public final DocRequest getDocRequest() {
        return docRequest;
    }

    //-------------------------------------------------------------------------
    /**
     * Sets the cleaningMember
     *
     * @param cleaningMember The cleaningMember to set
     */
    public void setDocRequest(final DocRequest docRequest) {
        this.docRequest = docRequest;
    }

    //-------------------------------------------------------------------------
    @Override
    public int hashCode() {
        return (null != docRequest) ? docRequest.hashCode() : 0;
    }

    //-------------------------------------------------------------------------
    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (getClass() != obj.getClass())
            return false;
        PortableDocRequest other = (PortableDocRequest) obj;
        if (docRequest != other.docRequest)
            return false;
        if (docRequest == null) {
            if (other.docRequest != null)
                return false;
        } else if (!docRequest.equals(other.docRequest))
            return false;
        return true;
    }

    //-------------------------------------------------------------------------
    @Override
    public String toString() {
        return "PortableDocRequest [ docRequest = " + docRequest + "]";
    }

    //-------------------------------------------------------------------------
    @Override
    public int getClassId() {
        return CLASS_ID;
    }

    //-------------------------------------------------------------------------
    @Override
    public void writePortable(PortableWriter writer) throws IOException {
        writer.writeByteArray(FIELD_DOCREQUEST, getBytes(docRequest));
    }

    //-------------------------------------------------------------------------
    @Override
    public void readPortable(PortableReader reader) throws IOException {
        byte[] docRequestBytes = reader.readByteArray(FIELD_DOCREQUEST);
        try {
            docRequest = getDocRequest(docRequestBytes);
        } catch (ClassNotFoundException e) {
            throw new IOException(e);
        }
    }

    //-------------------------------------------------------------------------
    @Override
    public int getFactoryId() {
        return FACTORY_ID;
    }

    //-------------------------------------------------------------------------
	@Override
	public String call() throws Exception {
        final DocumentRequestProcessor aDocProcAccess = ServiceLookupRegistry.get().getService(DocumentRequestProcessor.class);

        return aDocProcAccess.processDocRequest(this.docRequest);
	}

    //-------------------------------------------------------------------------
    /**
     * Serialize a {@link DocRequest} into a byte array
     * @param docRequest The {@link DocRequest} to be serialized
     * @return The serialized {@link DocRequest} as byte array
     * @throws IOException If the {@link Stanza} can't be serialized
     */
    private static byte[] getBytes(DocRequest docRequest) throws IOException {
        final ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
        final ObjectOutputStream objectOutputStream = new ObjectOutputStream(byteArrayOutputStream);
        objectOutputStream.writeObject(docRequest);
        return byteArrayOutputStream.toByteArray();
    }

    //-------------------------------------------------------------------------
    /**
     * Deserialize a {@link DocRequest} from a byte array representation. Needs access to all the classes that make up a DocRequest via proper OSGI
     * imports.
     *
     * @param stanzaBytes The byte array representation of the Stanza
     * @return The deserialzed {@link Stanza}
     * @throws IOException If reading the byte array fails
     * @throws ClassNotFoundException If the OSGI imports are too restrictive and not all classes that make up a {@link DocRequest} subclass are
     *             accessible
     */
    private static DocRequest getDocRequest(byte[] docRequestBytes) throws IOException, ClassNotFoundException {
        final ByteArrayInputStream byteArrayInputStream = new ByteArrayInputStream(docRequestBytes);
        final ObjectInputStream objectInputStream = new ObjectInputStream(byteArrayInputStream);
        return DocRequest.class.cast(objectInputStream.readObject());
    }

}
