/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.document;

import com.openexchange.config.ConfigurationService;
import com.openexchange.exception.OXException;
import com.openexchange.file.storage.File;
import com.openexchange.file.storage.FileStorageFileAccess;
import com.openexchange.office.document.DocFileHelper.LockingInfo;
import com.openexchange.office.filter.api.IImporter;
import com.openexchange.office.tools.doc.DocumentFormat;
import com.openexchange.office.tools.doc.DocumentFormatHelper;
import com.openexchange.office.tools.doc.DocumentType;
import com.openexchange.office.document.EmptyDocData;
import com.openexchange.office.tools.error.ErrorCode;
import com.openexchange.office.tools.error.ExceptionToErrorCode;
import com.openexchange.office.tools.files.FileHelper;
import com.openexchange.office.tools.files.StorageHelper;
import com.openexchange.office.tools.logging.v2.ELogLevel;
import com.openexchange.office.tools.logging.v2.impl.Logger;
import com.openexchange.office.tools.logging.v2.impl.Slf4JLogger;
import com.openexchange.office.tools.osgi.ServiceLookupRegistry;
import com.openexchange.office.tools.user.UserData;
import com.openexchange.session.Session;

public class OXDocumentInfo implements IDocumentInfo {

    //-------------------------------------------------------------------------
    private static final Logger LOG                   = Slf4JLogger.create(OXDocument.class);

    //-------------------------------------------------------------------------
    protected static final int MAX_DOCUMENT_FILE_SIZE = 100; // MBytes

    //-------------------------------------------------------------------------
    private boolean m_newDoc                    = false;

    //-------------------------------------------------------------------------
    private File m_metaData                     = null;

    //-------------------------------------------------------------------------
    private ErrorCode m_lastErrorCode           = ErrorCode.NO_ERROR;

    //-------------------------------------------------------------------------
    private StorageHelper m_storageHelper       = null;

    //-------------------------------------------------------------------------
    private DocumentMetaData m_documentMetaData = null;

    //-------------------------------------------------------------------------
    private Session m_session                   = null;

    //-------------------------------------------------------------------------
    private DocumentFormat m_documentFormat     = null;

    //-------------------------------------------------------------------------
    private boolean m_isTemplateDoc             = false;

    //-------------------------------------------------------------------------
    private boolean m_isRescueDoc               = false;

    //-------------------------------------------------------------------------
    /**
     * Initializes a new {@link OXDocument}.
     *
     * @param session
     *            The session of the client that requested a service which needs documents access.
     * @param services
     *            The service lookup instance to be used by this instance.
     * @param folderId
     *            The folder id of the file to be accessed.
     * @param fileId
     *            The file id of the file to be accessed.
     * @param fileHelper
     *            The FileHelper instance that is responsible for the file access regarding the document.
     * @param newDoc
     *            Specifies, if this instance references a new document or not.
     * @param resourceManager
     *            The ResourceManager instance which stores resources addressed by the document to be available via REST-API and
     *            other documents.
     * @param storageHelper
     *            A helper instance which provides data to the referenced document storage. The instance must be
     *            initialized with the same folderId, where the document itself is stored.
     */
    public OXDocumentInfo(final Session session, final String fileId, final DocFileHelper fileHelper, boolean newDoc, final StorageHelper storageHelper) {
        super();

        m_session       = session;
        m_newDoc        = newDoc;
        m_storageHelper = storageHelper;

        impl_Init(fileId, fileHelper, "");
    }

    //-------------------------------------------------------------------------
    /**
     * Initializes a new {@link OXDocument}.
     *
     * @param session
     *            The session of the client that requested a service which needs documents access.
     * @param services
     *            The service lookup instance to be used by this instance.
     * @param userData
     *            The UserData instance of the user associated with the session.
     * @param fileHelper
     *            The FileHelper instance that is responsible for the file access regarding the document.
     * @param newDoc
     *            Specifies, if this instance references a new document or not.
     * @param resourceManager
     *            The ResourceManager instance which stores resources addressed by the document to be available via REST-API and
     *            other documents.
     * @param storageHelper
     *            A helper instance which provides data to the referenced document storage. The instance must be
     *            initialized with the same folderId, where the document itself is stored.
     */
    public OXDocumentInfo(final Session session, final UserData userData, final DocFileHelper fileHelper, boolean newDoc, final StorageHelper storageHelper) {
        super();

        m_session       = session;
        m_newDoc        = newDoc;
        m_storageHelper = storageHelper;

        impl_Init(userData.getFileId(), fileHelper, userData.getEncryptionInfo());
    }

    //-------------------------------------------------------------------------
    /**
     * Initializes the OXDocument instance for source mode file.
     *
     * @param folderId
     * @param fileId
     * @param fileHelper
     * @param encryptionInfo
     */
    private void impl_Init(final String fileId, final DocFileHelper fileHelper, String encryptionInfo) {

        if (null != fileHelper) {
            try {
                // See #40279 - it's possible to get the document stream, but not the meta data
                // In this case prevent that we have a document buffer - which will result in a
                // general error.
                m_metaData = fileHelper.getMetaDataFromFile(m_session, fileId, FileStorageFileAccess.CURRENT_VERSION);
                if (null != m_metaData) {
                    final ConfigurationService configService = ServiceLookupRegistry.get().getService(ConfigurationService.class);
                    long nMaxDocumentFileSize = MAX_DOCUMENT_FILE_SIZE * 1024L * 1024L;

                    if (null != configService) {
                        nMaxDocumentFileSize = configService.getIntProperty("com.openexchange.office.maxDocumentFileSize", MAX_DOCUMENT_FILE_SIZE) * 1024L * 1024L;
                    }

                    final long nFileSize = m_metaData.getFileSize();

                    if (nFileSize <= nMaxDocumentFileSize) {
                        final boolean     bWriteProtected = !FileHelper.canWriteToFile(m_session, m_metaData, m_session.getUserId());
                        final LockingInfo aLockingInfo    = DocFileHelper.getLockingInfo(m_session, m_metaData);

                        m_documentFormat   = fileHelper.getDocumentFormat();
                        m_documentMetaData = new DocumentMetaData(m_metaData, aLockingInfo.getLockedByUser(), aLockingInfo.getLockedByUserId());
                        m_documentMetaData.setWriteProtectedState(bWriteProtected);
                        m_isTemplateDoc    = DocumentFormatHelper.isSupportedTemplateFormat(m_documentMetaData.getMimeType(), FileHelper.getExtension(m_documentMetaData.getFileName()));
                    } else {
                    	m_lastErrorCode = ErrorCode.LOADDOCUMENT_COMPLEXITY_TOO_HIGH_ERROR;
                        LOG.forLevel(ELogLevel.E_DEBUG).withMessage("RT connection: Document exceeds limit for file size " + nMaxDocumentFileSize + "MB, doc size: " + (nFileSize / (1024L * 1024L)) + "MB!").log();
                    } 
                } else {
                    LOG.forLevel(ELogLevel.E_WARNING).withMessage("RT connection: Unable to retrieve the document's meta data - must give up to open the file").log();
                }
            } catch (OXException e) {
                m_lastErrorCode = ExceptionToErrorCode.map(e, ErrorCode.LOADDOCUMENT_FAILED_ERROR, false);
                LOG.forLevel(ELogLevel.E_ERROR).withError(e).withMessage("RT connection: unable to retrieve meta data or permissions, exception caught").log();
	        } catch (Exception e) {
                m_lastErrorCode = ErrorCode.GENERAL_UNKNOWN_ERROR;
                LOG.forLevel(ELogLevel.E_ERROR).withError(e).withMessage("RT connection: general exception caught").log();
	        }
        }
    }

    //-------------------------------------------------------------------------
	@Override
	public DocumentMetaData getDocumentMetaData() {
		return m_documentMetaData;
	}

    //-------------------------------------------------------------------------
	@Override
	public ErrorCode getLastError() {
        return m_lastErrorCode;
	}

    //-------------------------------------------------------------------------
	@Override
	public Session getSession() {
		return m_session;
	}

    //-------------------------------------------------------------------------
	@Override
	public String getFolderId() {
        return (m_metaData != null) ? m_metaData.getFolderId() : null;
	}

    //-------------------------------------------------------------------------
	@Override
	public String getFileId() {
        return (m_metaData != null) ? m_metaData.getId() : null;
	}

    //-------------------------------------------------------------------------
	@Override
	public String getVersion() {
        return (m_metaData != null) ? m_metaData.getVersion() : null;
	}

    //-------------------------------------------------------------------------
	@Override
	public File getMetaData() {
		return m_metaData;
	}

    //-------------------------------------------------------------------------
	@Override
	public StorageHelper getStorageHelper() {
        return m_storageHelper;
	}

    //-------------------------------------------------------------------------
	@Override
	public DocumentFormat getDocumentFormat() {
        return m_documentFormat;
	}

    //-------------------------------------------------------------------------
	@Override
	public DocumentType getDocumentType() {
        DocumentType type;
        final DocumentFormat format = getDocumentFormat();
        switch (format) {
        case DOCX:
        case ODT : type = DocumentType.TEXT; break;

        case XLSX:
        case ODS : type = DocumentType.SPREADSHEET; break;

        case PPTX:
        case ODP : type = DocumentType.PRESENTATION; break;

        case NONE:
        default  : type = DocumentType.NONE; break;
        }

        return type;
	}

    //-------------------------------------------------------------------------
	@Override
	public boolean isRescueDocument() {
        return m_isRescueDoc;
	}

    //-------------------------------------------------------------------------
	@Override
	public boolean isTemplateDocument() {
		return m_isTemplateDoc;
	}

    //-------------------------------------------------------------------------
	@Override
	public boolean isNewDocument() {
		return m_newDoc;
	}

    //-------------------------------------------------------------------------
	@Override
	public int getUniqueDocumentId() {
		return OXDocument.DEFAULT_DOCUMENT_ID;
	}

    //-------------------------------------------------------------------------
	@Override
	public byte[] getDocumentBuffer() {
		return null;
	}

}
