package com.openexchange.office.document;

import org.apache.commons.lang3.Validate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.openexchange.file.storage.File;
import com.openexchange.file.storage.composition.IDBasedFileAccess;
import com.openexchange.file.storage.composition.IDBasedFolderAccess;
import com.openexchange.office.session.SessionService;
import com.openexchange.office.tools.files.FileHelper;
import com.openexchange.office.tools.files.FolderHelper;
import com.openexchange.office.tools.files.StorageHelper;
import com.openexchange.office.tools.osgi.ServiceLookupRegistryService;
import com.openexchange.server.ServiceLookup;
import com.openexchange.session.Session;


public class BackupFileHelper {

    private final static Logger log = LoggerFactory.getLogger(BackupFileHelper.class);

    static public final String BACKUP_FILE_POSTFIX = "-bak";

    private BackupFileHelper() {}

    /**
     *
     * @param services
     * @param session
     * @param storageHelper
     * @param folderId
     * @param metaData
     * @return
     */
    public static boolean canCreateOrWriteBackupFile(final ServiceLookup servicesDEPRECATED, final Session session, final StorageHelper storageHelper, final String folderId, final DocumentMetaData metaData, boolean encrypted) {
       Validate.notNull(session, "session must be set");
       Validate.notNull(storageHelper, "storageHelper must be set");
       Validate.notNull(folderId, "folderId must be set");
       Validate.notNull(metaData, "metaData must be set");

       boolean canCreateWriteBackupFile = false;

       if (!storageHelper.supportsFileVersions()) {
           // we must check further requirements to answer the question
           try {
               final IDBasedFolderAccess folderAccess = FolderHelper.getFolderAccess(null, session);
               final IDBasedFileAccess fileAccess = FileHelper.getFileAccess(null, session);
               int[] permissions = FolderHelper.getFolderPermissions(folderAccess, folderId);
               if (null != permissions) {
                   int write  = permissions[FolderHelper.FOLDER_PERMISSIONS_WRITE];
                   int folder = permissions[FolderHelper.FOLDER_PERMISSIONS_FOLDER];
                   if ((folder >= com.openexchange.file.storage.FileStoragePermission.CREATE_OBJECTS_IN_FOLDER) &&
                       (write >= com.openexchange.file.storage.FileStoragePermission.WRITE_ALL_OBJECTS)) {
                       // permissions are sufficient to create/write a bak file - no further tests necessary
                       canCreateWriteBackupFile = true;
                   } else {
                       // We have again dig deeper and now check a possible bak-file existence
                       // First check file existence using the current file name
                       final String fileName       = metaData.getFileName();
                       final String backupFileName = FileHelper.createFilenameWithPostfix(fileName, BackupFileHelper.BACKUP_FILE_POSTFIX, encrypted);
                       // BE CAREFULL - search must be done using the final file name including a possible .pgp extension in case of encryption!!
                       final String searchBakFileName = (encrypted ? backupFileName + "." + FileHelper.STR_EXT_PGP : backupFileName);
                       final File bakMetaData  = FileHelper.getMetaDataFromFileName(fileAccess, folderId, searchBakFileName);

                       if (null != bakMetaData) {
                           // BACKUP file exists - check write permissions
                           final int userId = ServiceLookupRegistryService.getInstance().getService(SessionService.class).getUserId(session);
                           canCreateWriteBackupFile = FileHelper.canWriteToFile(session, bakMetaData, userId);
                       } else {
                           // BACKUP file doesn't exists - check folder permissions
                           canCreateWriteBackupFile = (folder >= com.openexchange.file.storage.FileStoragePermission.CREATE_OBJECTS_IN_FOLDER);
                       }
                   }
               }
           } catch (Exception e) {
        	   log.error("RT connection: Exception caught while trying to determine backup file permissions", e);
           }
       } else {
           canCreateWriteBackupFile = true;
       }

       return canCreateWriteBackupFile;
   }

}
