/*
*
*    OPEN-XCHANGE legal information
*
*    All intellectual property rights in the Software are protected by
*    international copyright laws.
*
*
*    In some countries OX, OX Open-Xchange, open xchange and OXtender
*    as well as the corresponding Logos OX Open-Xchange and OX are registered
*    trademarks.
*    The use of the Logos is not covered by the GNU General Public License.
*    Instead, you are allowed to use these Logos according to the terms and
*    conditions of the Creative Commons License, Version 2.5, Attribution,
*    Non-commercial, ShareAlike, and the interpretation of the term
*    Non-commercial applicable to the aforementioned license is published
*    on the web site http://www.open-xchange.com/EN/legal/index.html.
*
*    Please make sure that third-party modules and libraries are used
*    according to their respective licenses.
*
*    Any modifications to this package must retain all copyright notices
*    of the original copyright holder(s) for the original code used.
*
*    After any such modifications, the original and derivative code shall remain
*    under the copyright of the copyright holder(s) and/or original author(s)per
*    the Attribution and Assignment Agreement that can be located at
*    http://www.open-xchange.com/EN/developer/. The contributing author shall be
*    given Attribution for the derivative code and a license granting use.
*
*     Copyright (C) 2016-2020 OX Software GmbH
*     Mail: info@open-xchange.com
*
*
*     This program is free software; you can redistribute it and/or modify it
*     under the terms of the GNU General Public License, Version 2 as published
*     by the Free Software Foundation.
*
*     This program is distributed in the hope that it will be useful, but
*     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
*     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
*     for more details.
*
*     You should have received a copy of the GNU General Public License along
*     with this program; if not, write to the Free Software Foundation, Inc., 59
*     Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
*/

package com.openexchange.office.backup.manager.impl;

import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.openexchange.filemanagement.ManagedFileManagement;
import com.openexchange.office.backup.distributed.GCManagedFiles;
import com.openexchange.office.tools.monitoring.RestoreDocEvent;
import com.openexchange.office.tools.monitoring.RestoreDocEventType;
import com.openexchange.office.tools.monitoring.Statistics;
import com.openexchange.office.tools.osgi.ServiceLookupRegistry;

public class GCObsoleteStreams implements GCManagedFiles {
    private static final long GC_TIMEOUT_FOR_ENTRY = 30000;
	private static final String THREAD_NAME = "com.openexchange.office.backup.GCObsoleteStreams";

    private static final Logger log = LoggerFactory.getLogger(AsyncBackupWorker.class);

    private final Queue<GCObsoleteStreamEntry> gcManagedFileIds = new ConcurrentLinkedQueue<>();
    private final ManagedFileManagement managedFileManagementService;
    private Thread gcTaskThread;

    /**
     * Initializes a new GCObsoleteStreams instance which uses a thread-based
     * background processing-to remove obsolete managed files.
     *
     */
	public GCObsoleteStreams() {
        managedFileManagementService = ServiceLookupRegistry.get().getService(ManagedFileManagement.class);

        gcTaskThread = new Thread() {

            @Override
            public void run() {
                boolean bInterrupted = false;

                while (!bInterrupted) {
                	try {
                        final long timeStart = System.currentTimeMillis();

                        log.trace("GCObsoleteStreams starts garbage collection round - {} entries in gc queue", gcManagedFileIds.size());

                    	boolean allTimeupRemoved = false;
                        int nCount = 0; 
                    	while (!allTimeupRemoved) {
    	                    try {
    	                        final GCObsoleteStreamEntry managedFileToBeGC = gcManagedFileIds.peek();

    	                        if (managedFileToBeGC != null) {
        	                        final long diff = Math.abs(System.currentTimeMillis() - managedFileToBeGC.timeAdded());
        	                        if (diff >= GC_TIMEOUT_FOR_ENTRY) {
        	                            final String managedFileId = managedFileToBeGC.managedFileId();
        	
        	                            log.trace("Managed file {} removed by GCObsoleteStreams ", managedFileId);
        	
        	                            nCount++;
        	                            managedFileManagementService.removeByID(managedFileId);
        	                            gcManagedFileIds.poll();
        	
        	                            Statistics.handleRestoreDocEvent(new RestoreDocEvent(RestoreDocEventType.MANAGED_FILES_CURRENT_DEC));
        	                        } else {
        	                            allTimeupRemoved = true;
        	                        }
    	                        } else {
    	                        	allTimeupRemoved = true;
    	                        }

    	                        // make sure that we bail out as soon as possible after being interrupted
    	                        if (Thread.interrupted()) {
    	                        	throw new InterruptedException();
    	                        }
    	                    } catch (InterruptedException e) {
    	                    	Thread.currentThread().interrupt();
    	                    	bInterrupted = true;
    	                    	throw new InterruptedException();
    	                    } catch (Exception e) {
    	                        log.info("RT2: Exception caught while trying to remove an obsolete managed file", e);
    	                    }
                    	}

                        final long nTimeUsed = System.currentTimeMillis() - timeStart;

                        Statistics.handleRestoreDocEvent(new RestoreDocEvent(RestoreDocEventType.MANAGED_FILES_LAST_CYCLE_TIME, nTimeUsed));
                        Statistics.handleRestoreDocEvent(new RestoreDocEvent(RestoreDocEventType.MANAGED_FILES_LAST_CYCLE_REMOVED, nCount));
                        Statistics.handleRestoreDocEvent(new RestoreDocEvent(RestoreDocEventType.MANAGED_FILES_TO_BE_REMOVED, gcManagedFileIds.size()));

                        log.trace("GCObsoleteStreams garbage collected {} managed files in {} milliseconds", nCount, (Math.abs(System.currentTimeMillis() - timeStart)));

                        Thread.sleep(2000);
                	} catch (InterruptedException e) {
                    	Thread.currentThread().interrupt();
                    	bInterrupted = true;
                    }
                }
            }
        };

        gcTaskThread.setName(THREAD_NAME);
        gcTaskThread.setDaemon(true);
        gcTaskThread.setPriority(Thread.NORM_PRIORITY - 1);
        gcTaskThread.start();

        log.trace("GCObsoleteStreams thread created");
    }

	/**
	 * Adds a managed file id to be removed by garbage collector.
	 *
	 * @param fileId a managed file id - empty strings or null will be ignored
	 */
	public void addObseleteManagedFileId(String fileId) {
		if (StringUtils.isNotEmpty(fileId)) {

			final GCObsoleteStreamEntry entry = new GCObsoleteStreamEntry(fileId);
			gcManagedFileIds.add(entry);
		}
	}

	/**
	 * Removes pending jobs to be processed and stop the background
	 * processing. 
	 */
	public void dispose() {
		log.trace("GCObsoleteStreams.dispose() called");

	    gcManagedFileIds.clear();

	    final Thread aThread = gcTaskThread;
	    if (aThread != null) {
            try
            {
            	aThread.interrupt();
            	aThread.join(1000);
            	gcTaskThread = null;
            }
            catch (final InterruptedException e) {
                log.debug("GCObsoleteStreams: InterruptedException preparing shutdown of the gc thread", e);
                Thread.currentThread().interrupt();
            }
	    }
	}

}
