/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.imageconverter.impl;

import static com.openexchange.imageconverter.impl.ImageConverterUtils.LOG;
import java.io.File;
import java.io.IOException;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.StringUtils;
import org.im4java.core.ConvertCmd;
import org.im4java.core.IM4JavaException;
import org.im4java.core.IMOperation;
import org.im4java.core.IMOps;
import org.im4java.process.ProcessExecutor;
import org.im4java.process.ProcessStarter;
import org.im4java.process.ProcessTask;
import com.google.common.base.Throwables;
import com.openexchange.annotation.NonNull;
import com.openexchange.annotation.Nullable;
import com.openexchange.imageconverter.api.ImageConverterException;
import com.openexchange.imageconverter.api.ImageFormat;
import com.openexchange.imagetransformation.ScaleType;

/**
 * {@link ImageProcessor}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.0
 */
public class ImageProcessor {

/* Debug class for simulating long running conversion tasks
 * To be used ionstead of ConvertCmd class, seetting an appropriate
 * sleepTimeoutSeconds and using a 'sleep_convert' script file
 * TODO (KA): check disabling code for for production release

    class SleepConvertCmd extends ImageCommand{
        public SleepConvertCmd() {
            this(Boolean.getBoolean("im4java.useGM"));
        }

        public SleepConvertCmd(int sleepTimeoutSeconds) {
            this(Boolean.getBoolean("im4java.useGM"), sleepTimeoutSeconds);
        }

        public SleepConvertCmd(boolean useGM) {
            this(useGM, 0);
        }

        public SleepConvertCmd(boolean useGM, int sleepTimeoutSeconds) {
            final String sleepTimeoutStr = Integer.toString(sleepTimeoutSeconds);

            if (useGM) {
                setCommand("sleep_convert", sleepTimeoutStr, "gm","convert");
            } else {
                setCommand("sleep_convert", sleepTimeoutStr, "convert");
            }
        }
    }
*/

    /**
     * {@link ImageProcessorStatus}
     *
     * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
     * @since v7.10.1
     */
    public static class ImageProcessorStatus {

        /**
         * Initializes a new {@link ImageProcessorStatus}.
         * @param isAvailable
         * @param executablePath
         * @param isGraphicsMagick
         */
        public ImageProcessorStatus(final String processorDescription, final boolean isAvailable) {
            m_processorDescription = processorDescription;
            m_isAvailable = isAvailable;
        }

        /**
         * @return
         */
        public String getProcessorDescription() {
            return m_processorDescription;
        }

        /**
         * @return
         */
        public boolean isAvailable() {
            return m_isAvailable;
        }

        // - Members -----------------------------------------------------------

        final private String m_processorDescription;

        final private boolean m_isAvailable;
    }

    /**
     * Initializes a new {@link ImageProcessor}.
     */
    public ImageProcessor(@NonNull final String searchPath, boolean useGraphicsMagick, @Nullable String userComment) {
        super();

        ProcessStarter.setGlobalSearchPath(searchPath);

        m_useGraphicsMagick = useGraphicsMagick;
        m_imageProcessorStatus = implGetImageProcessorStatus(searchPath);
        m_userComment = userComment;
    }

    /**
     *
     */
    public void shutdown() {
        m_processExecutor.shutdownNow();

        try {
            m_processExecutor.awaitTermination(ImageConverterUtils.AWAIT_TERMINATION_TIMEOUT_MILLIS, TimeUnit.MILLISECONDS);
        } catch (@SuppressWarnings("unused") InterruptedException e) {
            Thread.currentThread().interrupt();
        }
    }

    /**
     * @return
     */
    public ImageProcessorStatus getStatus() {
        return m_imageProcessorStatus;
    }

    /**
     * @param inputFile
     * @param outputFile
     * @param imageFormat
     * @return
     * @throws ImageConverterException
     */
    public boolean scale(
        final File inputFile,
        @NonNull final File outputFile,
        @NonNull final ImageFormat imageFormat) throws ImageConverterException {

        boolean ret = false;

        if (isAlive()) {
            final IMOperation imOp = new IMOperation();
            final Integer imWidth = ImageConverterUtils.getIM4JExtent(imageFormat.getWidth());
            final Integer imHeight = ImageConverterUtils.getIM4JExtent(imageFormat.getHeight());
            final ScaleType icScaleType = imageFormat.getScaleType();

            // write user comment into target image, if set
            if (null != m_userComment) {
                imOp.comment(m_userComment);
            }

            imOp.addImage();

            if (imageFormat.isAutoRotate()) {
                imOp.autoOrient();
            }

            switch (icScaleType) {
                case CONTAIN: {
                    imOp.resize(imWidth, imHeight, imageFormat.isShrinkOnly() ? Character.valueOf('>') : null);
                    break;
                }

                case CONTAIN_FORCE_DIMENSION: {
                    // http://www.imagemagick.org/Usage/resize/
                    // http://www.imagemagick.org/Usage/thumbnails/#fit_summery
                    imOp.resize(imWidth, imHeight, imageFormat.isShrinkOnly() ? Character.valueOf('>') : null);
                    imOp.gravity("center");
                    imOp.extent(imWidth, imHeight);
                    break;
                }

                case COVER: {
                    imOp.resize(imWidth, imHeight, Character.valueOf('^'));
                    break;
                }

                case COVER_AND_CROP: {
                    // Fill Area Flag ('^' flag)
                    imOp.resize(imWidth, imHeight, Character.valueOf('^'));
                    imOp.gravity("center");
                    imOp.extent(imWidth, imHeight);
                    break;
                }

                case AUTO:
                default:
                    imOp.scale(imWidth, imHeight);
                    break;
            }

            ret = submitTask(inputFile, outputFile, imageFormat, imOp);
        }

        return ret;
    }

    // - Implementation --------------------------------------------------------

    /**
     * @return
     */
    protected boolean isAlive() {
        return !m_processExecutor.isTerminated() && !m_processExecutor.isTerminating();
    }

    /**
     * @param inputFile
     * @param outputFile
     * @param imageFormat
     * @param imageOperation
     * @return
     * @throws ImageConverterException
     */
    protected boolean submitTask(
        @NonNull final File inputFile,
        @NonNull final File outputFile,
        @NonNull final ImageFormat imageFormat,
        @NonNull final IMOps imageOps) throws ImageConverterException {

        final ConvertCmd convertCmd = new ConvertCmd(m_useGraphicsMagick);
        ImageConverterException rethrowable = null;

        try {
            imageOps.quality(Double.valueOf(imageFormat.getQuality()));
            imageOps.addImage();

            final ProcessTask processTask = convertCmd.getProcessTask(imageOps, ImageConverterUtils.getIM4JInput(inputFile), ImageConverterUtils.getIM4JOutput(outputFile, imageFormat));

            if (null != processTask) {
                Future<?> future = m_processExecutor.submit(processTask);

                try {
                    if (null != future.get(ImageConverterConfig.IMAGECONVERTER_CONVERT_TIMEOUT_MILLIS, TimeUnit.MILLISECONDS)) {
                        rethrowable = new ImageConverterException("IC error while executing IM4Java ProcessTask");
                    }
                } catch (@SuppressWarnings("unused") TimeoutException e) {
                    processTask.cancel(true);

                    // copy input file to error directory, if configured
                    if (null != ImageConverterConfig.IMAGECONVERTER_ERROR_PATH) {
                        try {
                            FileUtils.copyFile(inputFile, new File(ImageConverterConfig.IMAGECONVERTER_ERROR_PATH, inputFile.getName()));
                        } catch (Exception e1) {
                            LOG.error("IC received Exception: {}", Throwables.getRootCause(e1));
                        }
                    }

                    LOG.error(ImageConverterUtils.STR_BUILDER().
                        append("IC conversion timeout error while executing IM4Java ProcessTask after ").
                        append(ImageConverterConfig.IMAGECONVERTER_CONVERT_TIMEOUT_MILLIS).append("ms").toString());

                    return false;
                }
            }
        } catch (IM4JavaException e) {
            rethrowable = new ImageConverterException("IC error while creating IM4Java ProcessTask", e);
        } catch (IOException e) {
            rethrowable = new ImageConverterException("IC error while creating IM4Java ProcessTask", e);
        } catch (ExecutionException e) {
            rethrowable = new ImageConverterException("IC error while running IM4Java ProcessTask", e);
        } catch (@SuppressWarnings("unused") InterruptedException e) {
            LOG.trace("IC interrupted while running IM4Java ProcessTask");
        } catch (@SuppressWarnings("unused") RejectedExecutionException e) {
            LOG.trace("IC rejected execution while running IM4Java ProcessTask");
        }

        if (null != rethrowable) {
            throw rethrowable;
        }

        return true;
    }

    // - Implementation --------------------------------------------------------

    /**
     * @param searchPath
     * @return
     */
    ImageProcessorStatus implGetImageProcessorStatus(@NonNull final String searchPath) {
        File convertExecutable = new File("n/a");
        boolean isAvailable = false;

        if (StringUtils.isNotBlank(searchPath)) {
            isAvailable = (convertExecutable = new File(searchPath, "convert")).canExecute() &&
                (!m_useGraphicsMagick || (convertExecutable = new File(searchPath, "gm")).canExecute());
        }

        if (!isAvailable) {
            if (LOG.isErrorEnabled()) {
                LOG.error(ImageConverterUtils.STR_BUILDER().
                    append("IC is not able to detect a valid ").
                    append(m_useGraphicsMagick ? "GraphicsMagick 'gm'" : "ImageMagick 'convert' ").
                    append("executable at configured search directory: ").append(searchPath).toString());
            }
        } else if (LOG.isInfoEnabled()) {
            LOG.info(ImageConverterUtils.STR_BUILDER().
                append("IC is using ").
                append(m_useGraphicsMagick ? "GraphicsMagick 'gm'" : "ImageMagick 'convert' ").
                append("executable located at: ").append(convertExecutable).toString());
        }

        return new ImageProcessorStatus(isAvailable ? convertExecutable.getPath() : "n/a", isAvailable);
    }

    // - Members ---------------------------------------------------------------

    final private ProcessExecutor m_processExecutor = new ProcessExecutor();

    final private boolean m_useGraphicsMagick;

    final private ImageProcessorStatus m_imageProcessorStatus;

    final private String m_userComment;
}
