/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.fileitem.impl;

import static com.openexchange.fileitem.impl.FileItemUtils.LOG;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.NoSuchElementException;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;

import com.google.common.base.Throwables;
import com.openexchange.annotation.NonNull;
import com.openexchange.exception.OXException;
import com.openexchange.filestore.FileStorage;
import com.openexchange.imageconverter.api.AccessOption;
import com.openexchange.imageconverter.api.ElementLocker;
import com.openexchange.imageconverter.api.FileItemException;
import com.openexchange.imageconverter.api.IFileItemWriteAccess;


/**
 * {@link FileItemWriteAccess}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.0
 */
public class FileItemWriteAccess extends FileItemReadAccess implements IFileItemWriteAccess  {

    final private static InputStream EMPTY_INPUT_STREAM = new ByteArrayInputStream(new byte[] {});

    /**
     * Initializes a new {@link FileItemWriteAccess}.
     * @param fileItem
     * @param fileItem
     * @throws FileItemException
     */
    protected FileItemWriteAccess(@NonNull FileItemService fileItemService,
        @NonNull final FileItemDatabase database,
        @NonNull final FileItem fileItem,
        AccessOption... accessOption) {

        super(fileItemService, database, fileItem, accessOption);
    }

    /* (non-Javadoc)
     * @see java.io.Closeable#close()
     */
    @Override
    public void close() throws IOException {
        try {
            // local lock
            ElementLocker.lock(m_fileItemKey);

            if (m_createNewFileItem) {
                final FileStoreData fileStoreData = createNewFileStoreFile();

                if (null != fileStoreData) {
                    m_fileItem.setFileStoreData(fileStoreData);

                    try {
                        m_database.createEntry(m_fileItem, fileStoreData, m_fileItemProperties);
                    } catch (FileItemException e) {
                        try {
                            // delete FileStore file in case of Database error
                            FileItemUtils.deleteFileStoreFile(m_fileItemService, fileStoreData);
                        } catch (FileItemException e1) {
                            LOG.warn("Unable to remove FileStore file after error occured: {}", Throwables.getRootCause(e1).getMessage());
                            // ok, there's no more, we can do
                        } finally {
                            // reset FileStoreData in case of failure in every case
                            m_fileItem.setFileStoreData(null);
                        }

                        throw new IOException("Error while creating database entry", e);
                    }
                }
            } else {
                final FileStoreData fileStoreData = m_fileItem.getFileStoreData();
                final long newFileLength = updateFileStoreFile(fileStoreData);

                // append to beginning or existing file content appendToFilein every case
                m_fileItemProperties.setLength(newFileLength);
                m_fileItemProperties.setModificationDateMillis(System.currentTimeMillis());

                // update the appropriate database entry with the changed file properties
                try {
                    m_database.updateProperties(fileStoreData, m_fileItem.getGroupId(), m_fileItemProperties.getChangedProperties());

                } catch (FileItemException e) {
                    throw new IOException("Error while updating database entry", e);
                }
            }
        } finally {
            // close output stream
            IOUtils.closeQuietly(m_fileContentOutputStm);
            m_fileContentOutputStm = null;

            // get rid of temp. output file
            FileUtils.deleteQuietly(m_fileContentOutputFile);
            m_fileContentOutputFile = null;

            // calling close method from superclass to cleanup object.
            // m_fileItem and m_fileItemProperties are not set to null
            // to be still accessible after closing the item access
            try {
                super.close();
            } finally {
                // local unlock
                ElementLocker.unlock(m_fileItemKey);

                // unlock access lifetime lock (was previously locked in #open method)
                ElementLocker.unlock(m_fileItemKey);
           }
        }
    }

    // - IFileItemWriteAccess ------------------------------------------------------

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemWriteAccess#getOutputStream()
     */
    @Override
    public OutputStream getOutputStream() {
        return m_fileContentOutputStm;
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemWriteAccess#getOutputFile()
     */
    @Override
    public File getOutputFile() throws IOException {
        if (null == m_fileContentOutputFile) {
            if (null != m_fileContentOutputStm) {
                if (null != (m_fileContentOutputFile = FileItemUtils.createTempFile())) {
                    FileUtils.writeByteArrayToFile(m_fileContentOutputFile, ((ByteArrayOutputStream) m_fileContentOutputStm).toByteArray());

                    // the previously used output stream is closed and not valid anymore
                    IOUtils.closeQuietly(m_fileContentOutputStm);
                    m_fileContentOutputStm = null;
                }
            } else {
                FileUtils.deleteQuietly(m_fileContentOutputFile);
                m_fileContentOutputFile = null;
            }
        }

        return m_fileContentOutputFile;
    }

    /* (non-Javadoc)
     * @see com.openexchange.imageconverter.api.IFileItemWriteAccess#setKeyValue(java.lang.String, java.lang.String)
     */
    @Override
    public void setKeyValue(final String key, final String value) {
        if (null != m_fileItemProperties) {
            m_fileItemProperties.setCustomKeyValue(key, value);
        }
    }

    // - Implementation --------------------------------------------------------

    /**
     * @return
     * @throws FileItemException
     * @throws NoSuchElementException
     * @throws IOException
     */
    @Override
    protected void open() throws FileItemException, IOException {
        try {
            // local lock
            ElementLocker.lock(m_fileItemKey);

            if (!tryOpen()) {
                // no file existing, it will be created by us in #close method;
                // in order to work this way, we need exclusive file access locking
                m_createNewFileItem = true;
                m_fileItemProperties = new FileItemProperties(m_database.getCustomPropertyKeys(m_fileItem.getGroupId()), System.currentTimeMillis());
                m_fileContentInputStm = EMPTY_INPUT_STREAM;
            }

            // create output file/stream objects to work with
            if (null != (m_fileContentOutputFile = FileItemUtils.createTempFile())) {
                m_fileContentOutputStm = FileUtils.openOutputStream(m_fileContentOutputFile);
            }
        } finally {
            // Global lock for the lifetime of this access object
            // => will finally be unlocked in #close call)
            ElementLocker.lock(m_fileItemKey);

            // local unlock
            ElementLocker.unlock(m_fileItemKey);
        }
    }

    /**
     * @return
     * @throws IOException
     */
    protected FileStoreData createNewFileStoreFile() throws IOException {
        FileStoreData ret = null;
        boolean writeOutput = (null != m_fileContentOutputFile);

        if (null != m_fileContentOutputStm) {
            m_fileContentOutputStm.flush();
            m_fileContentOutputStm.close();
            writeOutput = true;
        }

        if (writeOutput) {
            try (final InputStream newContentInputStm = (null != m_fileContentOutputFile) ?
                new BufferedInputStream(FileUtils.openInputStream(m_fileContentOutputFile)) :
                    new ByteArrayInputStream(((ByteArrayOutputStream) m_fileContentOutputStm).toByteArray())) {

                ret = m_fileItemService.createNewFileStoreFile(m_fileItem, m_fileItemProperties, newContentInputStm);
            }
        }

        return ret;
    }

    /**
     * @return The resulting size of the updated FileStore item
     * @throws IOException
     */
    protected long updateFileStoreFile(FileStoreData fileStoreData) throws IOException {
        long ret = 0;
        boolean writeOutput = (null != m_fileContentOutputFile);

        if (null != m_fileContentOutputStm) {
            m_fileContentOutputStm.flush();
            m_fileContentOutputStm.close();
            writeOutput = true;
        }

        if (writeOutput && FileItemUtils.isValid(fileStoreData)) {
            final FileStorage fileStorage = m_fileItemService.getFileStorage(fileStoreData);
            final String fileStoreId = fileStoreData.getFileStoreId();

            try (final InputStream newContentInputStm = (null != m_fileContentOutputFile) ?
                new BufferedInputStream(new FileInputStream(m_fileContentOutputFile)) :
                    new ByteArrayInputStream(((ByteArrayOutputStream) m_fileContentOutputStm).toByteArray())) {

                // get existing file size
                ret = fileStorage.getFileSize(fileStoreId);

                // truncate possible existing file content if WRITE_MODE_TRUNCATE option is set
                if ((ret > 0) && AccessOption.hasOption(m_accessOptions, AccessOption.WRITE_MODE_TRUNCATE)) {
                    fileStorage.setFileLength(ret = 0, fileStoreId);
                }

                // append new content at end of existing file
                ret = fileStorage.appendToFile(newContentInputStm, fileStoreId, ret);
            } catch (OXException e) {
                throw new IOException("Error while updating FileStore file" + m_fileItem.toString(), e);
            }
        }

        return ret;
    }

    // - Members ---------------------------------------------------------------

    protected OutputStream m_fileContentOutputStm = null;

    protected File m_fileContentOutputFile = null;

    protected boolean m_createNewFileItem = false;
 }
