/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.ox_json.impl;

import java.io.IOException;
import java.io.InputStream;
import org.apache.commons.httpclient.Header;
import org.apache.commons.httpclient.HttpMethod;
import com.openexchange.usm.api.contenttypes.resource.ResourceInputStream;
import com.openexchange.usm.api.session.Session;

/**
 * {@link ResourceInputStreamImpl}
 * 
 * @author <a href="mailto:afe@microdoc.de">Alexander Feess</a>
 */
public class ResourceInputStreamImpl extends ResourceInputStream {

    private static long getContentLength(HttpMethod method) throws IOException {
        Header header = method.getResponseHeader("Content-Length");
        if (header != null) {
            String value = header.getValue();
            if (value != null) {
                try {
                    return Long.parseLong(value);
                } catch (NumberFormatException e) {
                    throw new IOException("Content-Length header is no valid number: " + value);
                }
            }
        }
        return -1L; // Set to -1 if not specified by server
    }

    private final OXJSONAccessImpl _oxJSONAccess;

    private final Session _session;

    private final OXConnectionData _connectionData;

    private final HttpMethod _method;

    private final String _contentType;

    private final long _contentLength;

    private final String _description;

    private final long _startTime;

    private InputStream _inputStream;

    private long _lastActivity;

    private boolean _closed;

    public ResourceInputStreamImpl(OXJSONAccessImpl oxjsonAccess, Session session, OXConnectionData connectionData, HttpMethod method, String contentType, String description, long startTime) throws IOException {
        super();
        _description = description;
        _oxJSONAccess = oxjsonAccess;
        _session = session;
        _method = method;
        _contentType = contentType;
        _contentLength = getContentLength(method);
        _lastActivity = System.currentTimeMillis();
        _connectionData = connectionData;
        _startTime = startTime;
        connectionData.setCurrentResourceInputStream(this);
    }

    @Override
    public int read() throws IOException {
        _lastActivity = System.currentTimeMillis();
        if (_closed)
            return -1;
        int result = getInputStream().read();
        if (result == -1)
            close();
        return result;
    }

    @Override
    public int available() throws IOException {
        return _closed ? 0 : getInputStream().available();
    }

    @Override
    public void close() throws IOException {
        if (_closed)
            return;
        _closed = true;
        try {
            try {
                if (_inputStream != null)
                    _inputStream.close();
            } finally {
                _method.releaseConnection();
            }
        } finally {
            if (_connectionData.getCurrentResourceInputStream() == this)
                _connectionData.setCurrentResourceInputStream(null);
            _oxJSONAccess.releaseLock(_session);
            if (_startTime != 0L)
                OXJSONAccessImpl.collectOXStatistics(_session, _startTime);
        }
    }

    @Override
    public int read(byte[] b, int off, int len) throws IOException {
        _lastActivity = System.currentTimeMillis();
        if (_closed)
            return -1;
        int result = getInputStream().read(b, off, len);
        if (result == -1)
            close();
        return result;
    }
    
    @Override
    public synchronized void reset() throws IOException {
        getInputStream().reset();
    }

    @Override
    public long skip(long n) throws IOException {
        _lastActivity = System.currentTimeMillis();
        return getInputStream().skip(n);
    }

    public long getLastActivity() {
        return _lastActivity;
    }

    @Override
    public long getContentLength() {
        return _contentLength;
    }

    @Override
    public String getContentType() {
        return _contentType;
    }

    public HttpMethod getMethod() {
        return _method;
    }

    private InputStream getInputStream() throws IOException {
        if (_inputStream == null)
            _inputStream = _method.getResponseBodyAsStream();
        return _inputStream;
    }

    @Override
    public String toString() {
        return "ResourceInputStream(" + _closed + ',' + _description + ')';
    }
}
