/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.database.ox.update;

import java.sql.Connection;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Set;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import com.openexchange.exception.OXException;
import com.openexchange.groupware.update.PerformParameters;
import com.openexchange.tools.update.Tools;
import com.openexchange.usm.database.ox.USMUpdateExceptionCodes;
import com.openexchange.usm.database.ox.USMUpdateTaskProvider;

/**
 * {@link USMCreatePrimaryKeyUpdateTask} Replaces unique index with primary key for those USM table for which the primary key was missing.
 * Adds primary key or removes unique index if a previous execution was partially successful.
 *
 * @author <a href="mailto:afe@microdoc.de">Alexander Feess</a>
 */
public class USMCreatePrimaryKeyUpdateTask extends USMBaseUpdateTaskV2 {

    private static final Log LOG = LogFactory.getLog(USMCreatePrimaryKeyUpdateTask.class);

    private static class PrimaryKeyMappingInfo {

        public final String _tableName;

        public final String _indexName;

        public final String _primaryKeyColumns;

        public PrimaryKeyMappingInfo(String table, String index, String primaryKeyColumns) {
            _tableName = table;
            _indexName = index;
            _primaryKeyColumns = primaryKeyColumns;
        }
    }

    private static final PrimaryKeyMappingInfo[] _TABLES_WITHOUT_PRIMARY_KEY = {
        new PrimaryKeyMappingInfo("usmIdMapping", "usmIdMappingUIdx", "cid, usmSessionId, shortId"),
        new PrimaryKeyMappingInfo("easUIDMapping", "easUIDMappingUIDX", "cid, usmSessionId, folderId, shortId"),
        new PrimaryKeyMappingInfo("USMDataStorage", "USMDataStorageUIDX", "cid, usmSessionId, objectId, syncKey"),
        new PrimaryKeyMappingInfo("USMSessionFields", "cid", "cid, usmSessionId, field") };

    public USMCreatePrimaryKeyUpdateTask(USMUpdateTaskProvider provider) {
        super(provider);
    }

    @Override
    public String[] getDependencies() {
        return new String[] { USMTablesUpdateTaskV2.class.getName(), USMUUIDTablesUpdateTaskV2.class.getName() };
    }

    @Override
    protected void doPerform(PerformParameters params) throws OXException {
        Connection con = params.getConnection();
        Statement stmt = null;
        try {
            con.setAutoCommit(false);
            for (PrimaryKeyMappingInfo info : _TABLES_WITHOUT_PRIMARY_KEY) {
                stmt = dropIndexAndAddPrimaryKey(con, stmt, info);
            }
            con.commit();
        } catch (SQLException e) {
            rollback(con);
            throw USMUpdateExceptionCodes.SQL_ERROR.create(e);
        } finally {
            enableAutocommit(con, stmt);
        }
    }

    private static Statement dropIndexAndAddPrimaryKey(Connection con, Statement stmt, PrimaryKeyMappingInfo info) throws SQLException {
        Set<String> indexes = USMBaseUpdateTaskV2.getTableIndexes(info._tableName, con);
        String alterTableOperations = buildAlterTableOperations(info, Tools.hasPrimaryKey(con, info._tableName), indexes);
        if (alterTableOperations != null) {
            if (stmt == null) {
                stmt = con.createStatement();
            }
            String alterTableStatement = "ALTER TABLE " + info._tableName + ' ' + alterTableOperations;
            LOG.debug(alterTableStatement);
            stmt.execute(alterTableStatement);
        }
        return stmt;
    }

    private static String buildAlterTableOperations(PrimaryKeyMappingInfo info, boolean hasPrimaryKey, Set<String> indexes) {
        if (indexes.contains(info._indexName)) {
            if (hasPrimaryKey) {
                return "DROP INDEX " + info._indexName;
            }
            return "ADD PRIMARY KEY(" + info._primaryKeyColumns + "), DROP INDEX " + info._indexName;
        }
        if (!hasPrimaryKey) {
            return "ADD PRIMARY KEY(" + info._primaryKeyColumns + ')';
        }
        return null;
    }
}
