/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.usm.api.database;

import java.sql.CallableStatement;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.sql.SQLWarning;
import java.sql.Savepoint;
import java.sql.Statement;
import java.util.Map;

/**
 * Simple wrapper class for Connection that can be used by the DatabaseAccess to provide a "safe"
 * Connection to callers, i.e. a Connection that calls releaseConnection() on the DatabaseAccess
 * when it is closed and doesn't allow any calls after it got closed. Additionally the
 * EncapsulatedException performs a check on close() if the Connection is autoCommit==true, and
 * throws an IllegalStateException if it is not.
 *
 * @author afe
 *
 */
public class EncapsulatedConnection {

    /** The nature of the underlying connection */
    public static enum Nature {
        /**
         * Signals a read-only connection.
         */
        READ_ONLY,
        /**
         * Signals a read-write connection.
         */
        READ_WRITE,
        /**
         * Signals a read-write connection w/o a timeout.
         */
        NO_TIMEOUT;
    }

    // ------------------------------------------------------------------------------------------------------------- //

	private final Connection _connection;
	private final int _contextID;
	private final Nature _mode;
	private final DatabaseAccess _dbAccess;
	private boolean _closed = false;
	private boolean _isAutoCommit = true;

	/**
	 * Initializes a new {@link EncapsulatedConnection}.
	 */
	public EncapsulatedConnection(DatabaseAccess dbAccess, Connection connection, int contextID, boolean readOnly) {
		this(dbAccess, connection, contextID, readOnly ? Nature.READ_ONLY : Nature.READ_WRITE);
	}

	/**
     * Initializes a new {@link EncapsulatedConnection}.
     */
    public EncapsulatedConnection(DatabaseAccess dbAccess, Connection connection, int contextID, Nature mode) {
        super();
        _dbAccess = dbAccess;
        _connection = connection;
        _contextID = contextID;
        _mode = mode;
    }

	public void clearWarnings() throws SQLException {
		checkNotClosed();
		_connection.clearWarnings();
	}

	public void close() throws SQLException, StorageAccessException {
		if (!_isAutoCommit) {
            throw new IllegalStateException("Closing connection while autoCommit==false");
        }
		checkNotClosed();
		_closed = true;
		switch (_mode) {
            case READ_ONLY:
                _dbAccess.backReadOnly(_contextID, _connection);
                break;
            case READ_WRITE:
                _dbAccess.backWritable(_contextID, _connection);
                break;
            case NO_TIMEOUT:
                if (_dbAccess instanceof DatabaseAccessWithNoTimeoutSupport) {
                    ((DatabaseAccessWithNoTimeoutSupport) _dbAccess).backNoTimeout(_contextID, _connection);
                } else {
                    _dbAccess.backWritable(_contextID, _connection);
                }
                break;
            default:
                throw new IllegalStateException("Unknown mode: " + _mode.name());
        }
	}

	public void commit() throws SQLException {
		checkNotClosed();
		_connection.commit();
	}

	public Statement createStatement() throws SQLException {
		checkNotClosed();
		return _connection.createStatement();
	}

	public Statement createStatement(int resultSetType, int resultSetConcurrency, int resultSetHoldability)
			throws SQLException {
		checkNotClosed();
		return _connection.createStatement(resultSetType, resultSetConcurrency, resultSetHoldability);
	}

	public Statement createStatement(int resultSetType, int resultSetConcurrency) throws SQLException {
		checkNotClosed();
		return _connection.createStatement(resultSetType, resultSetConcurrency);
	}

	public boolean getAutoCommit() throws SQLException {
		checkNotClosed();
		return _connection.getAutoCommit();
	}

	public String getCatalog() throws SQLException {
		checkNotClosed();
		return _connection.getCatalog();
	}

	public int getHoldability() throws SQLException {
		checkNotClosed();
		return _connection.getHoldability();
	}

	public DatabaseMetaData getMetaData() throws SQLException {
		checkNotClosed();
		return _connection.getMetaData();
	}

	public int getTransactionIsolation() throws SQLException {
		checkNotClosed();
		return _connection.getTransactionIsolation();
	}

	public Map<String, Class<?>> getTypeMap() throws SQLException {
		checkNotClosed();
		return _connection.getTypeMap();
	}

	public SQLWarning getWarnings() throws SQLException {
		checkNotClosed();
		return _connection.getWarnings();
	}

	public boolean isClosed() {
		return _closed;
	}

	public boolean isReadOnly() throws SQLException {
		checkNotClosed();
		return _connection.isReadOnly();
	}

	public String nativeSQL(String sql) throws SQLException {
		checkNotClosed();
		return _connection.nativeSQL(sql);
	}

	public CallableStatement prepareCall(String sql, int resultSetType, int resultSetConcurrency,
			int resultSetHoldability) throws SQLException {
		checkNotClosed();
		return _connection.prepareCall(sql, resultSetType, resultSetConcurrency, resultSetHoldability);
	}

	public CallableStatement prepareCall(String sql, int resultSetType, int resultSetConcurrency) throws SQLException {
		checkNotClosed();
		return _connection.prepareCall(sql, resultSetType, resultSetConcurrency);
	}

	public CallableStatement prepareCall(String sql) throws SQLException {
		checkNotClosed();
		return _connection.prepareCall(sql);
	}

	public PreparedStatement prepareStatement(String sql, int resultSetType, int resultSetConcurrency,
			int resultSetHoldability) throws SQLException {
		checkNotClosed();
		return _connection.prepareStatement(sql, resultSetType, resultSetConcurrency, resultSetHoldability);
	}

	public PreparedStatement prepareStatement(String sql, int resultSetType, int resultSetConcurrency)
			throws SQLException {
		checkNotClosed();
		return _connection.prepareStatement(sql, resultSetType, resultSetConcurrency);
	}

	public PreparedStatement prepareStatement(String sql, int autoGeneratedKeys) throws SQLException {
		checkNotClosed();
		return _connection.prepareStatement(sql, autoGeneratedKeys);
	}

	public PreparedStatement prepareStatement(String sql, int[] columnIndexes) throws SQLException {
		checkNotClosed();
		return _connection.prepareStatement(sql, columnIndexes);
	}

	public PreparedStatement prepareStatement(String sql, String[] columnNames) throws SQLException {
		checkNotClosed();
		return _connection.prepareStatement(sql, columnNames);
	}

	public PreparedStatement prepareStatement(String sql) throws SQLException {
		checkNotClosed();
		return _connection.prepareStatement(sql);
	}

	public void releaseSavepoint(Savepoint savepoint) throws SQLException {
		checkNotClosed();
		_connection.releaseSavepoint(savepoint);
	}

	public void rollback() throws SQLException {
		checkNotClosed();
		_connection.rollback();
	}

	public void rollback(Savepoint savepoint) throws SQLException {
		checkNotClosed();
		_connection.rollback(savepoint);
	}

	public void setAutoCommit(boolean autoCommit) throws SQLException {
		checkNotClosed();
		_connection.setAutoCommit(autoCommit);
		_isAutoCommit = autoCommit;
	}

	public void setCatalog(String catalog) throws SQLException {
		checkNotClosed();
		_connection.setCatalog(catalog);
	}

	public void setHoldability(int holdability) throws SQLException {
		checkNotClosed();
		_connection.setHoldability(holdability);
	}

	public void setReadOnly(boolean readOnly) throws SQLException {
		checkNotClosed();
		_connection.setReadOnly(readOnly);
	}

	public Savepoint setSavepoint() throws SQLException {
		checkNotClosed();
		return _connection.setSavepoint();
	}

	public Savepoint setSavepoint(String name) throws SQLException {
		checkNotClosed();
		return _connection.setSavepoint(name);
	}

	public void setTransactionIsolation(int level) throws SQLException {
		checkNotClosed();
		_connection.setTransactionIsolation(level);
	}

	public void setTypeMap(Map<String, Class<?>> map) throws SQLException {
		checkNotClosed();
		_connection.setTypeMap(map);
	}

	private void checkNotClosed() throws SQLException {
		if (_closed) {
            throw new SQLException("Connection closed");
        }
	}
}
