/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.tools.doc;

import java.util.HashMap;
import java.util.Map;
import com.openexchange.documentconverter.Properties;


public class DocumentFormatHelper {


    public enum OfficeDocumentType {
        NONE,
        MS_DOCUMENT_WORD,
        MS_DOCUMENT_SPREADSHEET,
        MS_DOCUMENT_PRESENTATION,
        ODF_DOCUMENT_WRITER,
        ODF_DOCUMENT_CALC,
        ODF_DOCUMENT_IMPRESS
    }

    public enum OfficeTemplateDocumentType {
        NONE,
        MS_DOCUMENT_WORD_TEMPLATE,
        MS_DOCUMENT_SPREADSHEET_TEMPLATE,
        MS_DOCUMENT_PRESENTATION_TEMPLATE,
        ODF_DOCUMENT_WRITER_TEMPLATE,
        ODF_DOCUMENT_CALC_TEMPLATE,
        ODF_DOCUMENT_IMPRESS_TEMPLATE
    }

    /**
     * Provides conversion format info for mime type and extension type,
     * where the mime type is preferred.
     *
     * @param mimeType
     *  The mine type where we want to get the conversion format info. Can be
     *  null.
     *
     * @param extensionType
     *  The extension type of the file where we want to get conversion info
     *  for.
     *
     * @return
     *  The conversion format info or null if there is no known conversion
     *  information available.
     */
    static public Map<String, String> getConversionFormatInfo(String mimeType, String extensionType) {
        HashMap<String, String> result = null;

        // get default conversion for mime type first
        if (mimeType != null) {
            if (mimeType.startsWith("application/msword") || mimeType.equals("application/rtf") || mimeType.equals("text/rtf")) {
                result = getFormatData(OfficeDocumentType.MS_DOCUMENT_WORD);
            } else if (mimeType.startsWith("application/vnd.ms-excel")) {
                // xlsm files have a binary mime type set, although they
                // are xml files => do not convert xlsm files (see #31245)
                if ((null == extensionType) || !extensionType.equals("xlsm")) {
                    result = getFormatData(OfficeDocumentType.MS_DOCUMENT_SPREADSHEET);
                }
            } else if (mimeType.startsWith("application/vnd.ms-word")) {
                // dotm and docm files have a binary mime type set, although they are xml files => do not convert them
                if ((null == extensionType) || !(extensionType.equals("dotm")||extensionType.equals("docm"))) {
                    result = getFormatData(OfficeDocumentType.ODF_DOCUMENT_WRITER);
                }
            } else if (mimeType.startsWith("application/vnd.ms-powerpoint")) {
                result = getFormatData(OfficeDocumentType.MS_DOCUMENT_PRESENTATION);
            }
        }

        // get default conversion for extension type, if no valid mime type was given before
        if ((null == result) && (null != extensionType)) {
            if (extensionType.equals("doc") || extensionType.equals("dot") || extensionType.equals("rtf")) {
                result = getFormatData(OfficeDocumentType.MS_DOCUMENT_WORD);
            }
            else if (extensionType.equals("xls") || extensionType.equals("xlt") || extensionType.equals("xlsb")) {
                result = getFormatData(OfficeDocumentType.MS_DOCUMENT_SPREADSHEET);
            }
            else if (extensionType.equals("ppt") || extensionType.equals("pot") || extensionType.equals("pps")) {
                result = getFormatData(OfficeDocumentType.MS_DOCUMENT_PRESENTATION);
            }
        }

        return result;
    }

    /**
     * Provides the default document format info for a certain document type.
     *
     * @param type
     *  The document template type where the default template format info
     *  should be provided.
     *
     * @return
     *  The default document format info for the provided type or null if there
     *  is no known format info available.
     */
    static public HashMap<String, String> getFormatData(OfficeDocumentType officeDocumentType) {
        HashMap<String, String> data = new HashMap<String, String>();

        switch (officeDocumentType) {
        case MS_DOCUMENT_WORD: {
            data.put(Properties.PROP_MIME_TYPE, MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSING);
            data.put(Properties.PROP_INPUT_TYPE, "docx");
            data.put(Properties.PROP_FILTER_SHORT_NAME, "ooxml");
            data.put("DocumentType", DocumentType.TEXT.toString());
            break;
        }

        case MS_DOCUMENT_SPREADSHEET: {
            data.put(Properties.PROP_MIME_TYPE, MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET);
            data.put(Properties.PROP_INPUT_TYPE, "xlsx");
            data.put(Properties.PROP_FILTER_SHORT_NAME, "ooxml");
            data.put("DocumentType", DocumentType.SPREADSHEET.toString());
            break;
        }

        case MS_DOCUMENT_PRESENTATION: {
            data.put(Properties.PROP_MIME_TYPE, MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATION);
            data.put(Properties.PROP_INPUT_TYPE, "pptx");
            data.put(Properties.PROP_FILTER_SHORT_NAME, "ooxml");
            data.put("DocumentType", DocumentType.PRESENTATION.toString());
            break;
        }

        case ODF_DOCUMENT_WRITER: {
            data.put(Properties.PROP_MIME_TYPE, MimeTypeHelper.MIMETYPE_VND_OASIS_OPENDOCUMENT_TEXT);
            data.put(Properties.PROP_INPUT_TYPE, "odt");
            data.put(Properties.PROP_FILTER_SHORT_NAME, "odf");
            data.put("DocumentType", DocumentType.TEXT.toString());
            break;
        }

        case ODF_DOCUMENT_CALC: {
            data.put(Properties.PROP_INPUT_TYPE, "ods");
            data.put(Properties.PROP_MIME_TYPE, MimeTypeHelper.MIMETYPE_VND_OASIS_OPENDOCUMENT_SPREADSHEET);
            data.put(Properties.PROP_FILTER_SHORT_NAME, "odf");
            data.put("DocumentType", DocumentType.SPREADSHEET.toString());
            break;
        }

        case ODF_DOCUMENT_IMPRESS: {
            data.put(Properties.PROP_INPUT_TYPE, "odp");
            data.put(Properties.PROP_MIME_TYPE, MimeTypeHelper.MIMETYPE_VND_OASIS_OPENDOCUMENT_PRESENTATION);
            data.put(Properties.PROP_FILTER_SHORT_NAME, "odf");
            data.put("DocumentType", DocumentType.PRESENTATION.toString());
            break;
        }

        default: {
            break;
        }
        }

        return data;
    }

    /**
     * Provides the default template format info for a certain document template type.
     *
     * @param templateType
     *  The document template type where the default template format info
     *  should be provided.
     *
     * @return
     *  The template document format info for the provided type or null if there
     *  is no known format info available.
     */
    static public HashMap<String, String> getTemplateFormatData(OfficeTemplateDocumentType templateType) {
        HashMap<String, String> data = new HashMap<String, String>();

        switch (templateType) {
        case MS_DOCUMENT_WORD_TEMPLATE: {
            data.put(Properties.PROP_MIME_TYPE, MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_WORDPROCESSING_TEMPLATE);
            data.put(Properties.PROP_INPUT_TYPE, "dotx");
            data.put(Properties.PROP_FILTER_SHORT_NAME, "ooxml");
            data.put("DocumentType", DocumentType.TEXT.toString());
            break;
        }

        case MS_DOCUMENT_SPREADSHEET_TEMPLATE: {
            data.put(Properties.PROP_MIME_TYPE, MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET_TEMPLATE);
            data.put(Properties.PROP_INPUT_TYPE, "xltx");
            data.put(Properties.PROP_FILTER_SHORT_NAME, "ooxml");
            data.put("DocumentType", DocumentType.SPREADSHEET.toString());
            break;
        }

        case MS_DOCUMENT_PRESENTATION_TEMPLATE: {
            data.put(Properties.PROP_MIME_TYPE, MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATION_TEMPLATE);
            data.put(Properties.PROP_INPUT_TYPE, "potx");
            data.put(Properties.PROP_FILTER_SHORT_NAME, "ooxml");
            data.put("DocumentType", DocumentType.PRESENTATION.toString());
            break;
        }

        case ODF_DOCUMENT_WRITER_TEMPLATE: {
            data.put(Properties.PROP_MIME_TYPE, MimeTypeHelper.MIMETYPE_VND_OASIS_OPENDOCUMENT_TEXT_TEMPLATE);
            data.put(Properties.PROP_INPUT_TYPE, "ott");
            data.put(Properties.PROP_FILTER_SHORT_NAME, "odf");
            data.put("DocumentType", DocumentType.TEXT.toString());
            break;
        }

        case ODF_DOCUMENT_CALC_TEMPLATE: {
            data.put(Properties.PROP_INPUT_TYPE, "ots");
            data.put(Properties.PROP_MIME_TYPE, MimeTypeHelper.MIMETYPE_VND_OASIS_OPENDOCUMENT_SPREADSHEET_TEMPLATE);
            data.put(Properties.PROP_FILTER_SHORT_NAME, "odf");
            data.put("DocumentType", DocumentType.SPREADSHEET.toString());
            break;
        }

        case ODF_DOCUMENT_IMPRESS_TEMPLATE: {
            data.put(Properties.PROP_INPUT_TYPE, "otp");
            data.put(Properties.PROP_MIME_TYPE, MimeTypeHelper.MIMETYPE_VND_OASIS_OPENDOCUMENT_PRESENTATION_TEMPLATE);
            data.put(Properties.PROP_FILTER_SHORT_NAME, "odf");
            data.put("DocumentType", DocumentType.PRESENTATION.toString());
            break;
        }

        default: {
            break;
        }
        }

        return data;
    }

    /**
     * Provides the document format info that results if a document is created
     * based on a certain template format.
     *
     * @param mimeType
     *  The mime type of a document template file.
     *
     * @param extensionType
     *  The extension type of a document template file.
     *
     * @return
     *  The document format info or null if there is no known mapping.
     */
    static public HashMap<String, String> getTemplateTargetFormatInfo(String mimeType, String extensionType) {
        HashMap<String, String> result = null;

        // get default template target format for mime type first
        if (null != mimeType) {
            if (null == result) {
                // Spreadsheet
                if (mimeType.equalsIgnoreCase(MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET_TEMPLATE)) {
                    result = getFormatData(OfficeDocumentType.MS_DOCUMENT_SPREADSHEET);
                } else if (mimeType.equalsIgnoreCase(MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET_TEMPLATE_MACRO)) {
                    result = getFormatData(OfficeDocumentType.MS_DOCUMENT_SPREADSHEET);
                } else if (mimeType.equalsIgnoreCase(MimeTypeHelper.MIMETYPE_VND_OASIS_OPENDOCUMENT_SPREADSHEET_TEMPLATE)) {
                    result = getFormatData(OfficeDocumentType.ODF_DOCUMENT_CALC);
                }
            }

            if (null == result) {
                // Text
                if (mimeType.equalsIgnoreCase(MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET_TEMPLATE)) {
                    result = getFormatData(OfficeDocumentType.MS_DOCUMENT_WORD);
                } else if (mimeType.equalsIgnoreCase(MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET_TEMPLATE_MACRO)) {
                    result = getFormatData(OfficeDocumentType.MS_DOCUMENT_WORD);
                } else if (mimeType.equalsIgnoreCase(MimeTypeHelper.MIMETYPE_VND_OASIS_OPENDOCUMENT_TEXT_TEMPLATE)) {
                    result = getFormatData(OfficeDocumentType.ODF_DOCUMENT_WRITER);
                }
            }

            if (null == result) {
                // Presentation
                if (mimeType.equalsIgnoreCase(MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATION_TEMPLATE)) {
                    result = getFormatData(OfficeDocumentType.MS_DOCUMENT_PRESENTATION);
                } else if (mimeType.equalsIgnoreCase(MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATION_TEMPLATE_MACRO)) {
                    result = getFormatData(OfficeDocumentType.MS_DOCUMENT_PRESENTATION);
                } else if (mimeType.equalsIgnoreCase(MimeTypeHelper.MIMETYPE_VND_OASIS_OPENDOCUMENT_TEXT_TEMPLATE)) {
                    result = getFormatData(OfficeDocumentType.ODF_DOCUMENT_IMPRESS);
                }
            }
        }

        // get default template target format for extension type, if no valid mime type was given
        if ((null == result) && (null != extensionType)) {
            if ((extensionType.equalsIgnoreCase("xltx")) || (extensionType.equalsIgnoreCase("xltm"))) {
                // Spreadsheet
                result = getFormatData(OfficeDocumentType.MS_DOCUMENT_SPREADSHEET);
            } else if (extensionType.equalsIgnoreCase("ots")) {
                result = getFormatData(OfficeDocumentType.ODF_DOCUMENT_CALC);
            } else if ((extensionType.equalsIgnoreCase("dotx")) || (extensionType.equalsIgnoreCase("dotm"))) {
                // Text
                result = getFormatData(OfficeDocumentType.MS_DOCUMENT_WORD);
            } else if (extensionType.equalsIgnoreCase("ott")) {
                result = getFormatData(OfficeDocumentType.ODF_DOCUMENT_WRITER);
            } else if ((extensionType.equalsIgnoreCase("potx")) || (extensionType.equalsIgnoreCase("potm"))) {
                // Presentation
                result = getFormatData(OfficeDocumentType.MS_DOCUMENT_PRESENTATION);
            } else if (extensionType.equalsIgnoreCase("otp")) {
                result = getFormatData(OfficeDocumentType.ODF_DOCUMENT_IMPRESS);
            }
        }

        return result;
    }

    /**
     * Provides the mapping from a normal document format to a pre-defined
     * document template format. The mapping uses the mime type and the
     * extension type into account, preferring the mine type.
     *
     * @param mimeType
     *  The mime type of the document if known or null.
     *
     * @param extensionType
     *  The extension type of the document.
     *
     * @return
     *  The document template format info or null if there is no mapping.
     */
    static public HashMap<String, String> getTemplateFormatInfoForDocument(String mimeType, String extensionType) {
        HashMap<String, String> result = null;

        if (null != mimeType) {
            if (null == result) {
                // Spreadsheet
                if (mimeType.equalsIgnoreCase(MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET)) {
                    result = getTemplateFormatData(OfficeTemplateDocumentType.MS_DOCUMENT_SPREADSHEET_TEMPLATE);
                } else if (mimeType.equalsIgnoreCase(MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET_MACRO)) {
                    result = getTemplateFormatData(OfficeTemplateDocumentType.MS_DOCUMENT_SPREADSHEET_TEMPLATE);
                } else if (mimeType.equalsIgnoreCase(MimeTypeHelper.MIMETYPE_VND_OASIS_OPENDOCUMENT_SPREADSHEET)) {
                    result = getTemplateFormatData(OfficeTemplateDocumentType.ODF_DOCUMENT_CALC_TEMPLATE);
                }
            }

            if (null == result) {
                // Text
                if (mimeType.equalsIgnoreCase(MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET)) {
                    result = getTemplateFormatData(OfficeTemplateDocumentType.MS_DOCUMENT_WORD_TEMPLATE);
                } else if (mimeType.equalsIgnoreCase(MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET_MACRO)) {
                    result = getTemplateFormatData(OfficeTemplateDocumentType.MS_DOCUMENT_WORD_TEMPLATE);
                } else if (mimeType.equalsIgnoreCase(MimeTypeHelper.MIMETYPE_VND_OASIS_OPENDOCUMENT_TEXT)) {
                    result = getTemplateFormatData(OfficeTemplateDocumentType.ODF_DOCUMENT_WRITER_TEMPLATE);
                }
            }

            if (null == result) {
                // Presentation
                if (mimeType.equalsIgnoreCase(MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATION)) {
                    result = getTemplateFormatData(OfficeTemplateDocumentType.MS_DOCUMENT_PRESENTATION_TEMPLATE);
                } else if (mimeType.equalsIgnoreCase(MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATION_MACRO)) {
                    result = getTemplateFormatData(OfficeTemplateDocumentType.MS_DOCUMENT_PRESENTATION_TEMPLATE);
                } else if (mimeType.equalsIgnoreCase(MimeTypeHelper.MIMETYPE_VND_OASIS_OPENDOCUMENT_TEXT)) {
                    result = getTemplateFormatData(OfficeTemplateDocumentType.ODF_DOCUMENT_IMPRESS_TEMPLATE);
                }
            }
        }

        if ((null == result) && (null != extensionType)) {

            // get default template target format for extension type, if no valid mime type was given
            if ((extensionType.equalsIgnoreCase("xlsx")) || (extensionType.equalsIgnoreCase("xlsm"))) {
                // Spreadsheet
                result = getTemplateFormatData(OfficeTemplateDocumentType.MS_DOCUMENT_SPREADSHEET_TEMPLATE);
            } else if (extensionType.equalsIgnoreCase("ods")) {
                result = getTemplateFormatData(OfficeTemplateDocumentType.ODF_DOCUMENT_CALC_TEMPLATE);
            } else if ((extensionType.equalsIgnoreCase("docx")) || (extensionType.equalsIgnoreCase("docm"))) {
                // Text
                result = getTemplateFormatData(OfficeTemplateDocumentType.MS_DOCUMENT_WORD_TEMPLATE);
            } else if (extensionType.equalsIgnoreCase("odt")) {
                result = getTemplateFormatData(OfficeTemplateDocumentType.ODF_DOCUMENT_WRITER_TEMPLATE);
            } else if ((extensionType.equalsIgnoreCase("pptx")) || (extensionType.equalsIgnoreCase("pptm"))) {
                // Presentation
                result = getTemplateFormatData(OfficeTemplateDocumentType.MS_DOCUMENT_PRESENTATION_TEMPLATE);
            } else if (extensionType.equalsIgnoreCase("odp")) {
                result = getTemplateFormatData(OfficeTemplateDocumentType.ODF_DOCUMENT_IMPRESS_TEMPLATE);
            }
        }

        return result;
    }

    /**
     * Determines if the provided mime type or extension type specify a
     * supported document template format. Supported means that it can be
     * loaded without conversion needed.
     *
     * @param mimeType
     *  The mime type of the document if known or null.
     *
     * @param extensionType
     *  The extension type of the document.
     *
     * @return
     *  TRUE if the mime/extension type describe a supported document template
     *  format, FALSE if not.
     */
    static public boolean isSupportedTemplateFormat(String mimeType, String extensionType) {
        // get default template target format for mime type first
        if (null != mimeType) {
            // Spreadsheet
            if (mimeType.equalsIgnoreCase(MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET_TEMPLATE)) {
                return true;
            } else if (mimeType.equalsIgnoreCase(MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET_TEMPLATE_MACRO)) {
                return true;
            } else if (mimeType.equalsIgnoreCase(MimeTypeHelper.MIMETYPE_VND_OASIS_OPENDOCUMENT_SPREADSHEET_TEMPLATE)) {
               return true;
            }

            // Text
            if (mimeType.equalsIgnoreCase(MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET_TEMPLATE)) {
                return true;
            } else if (mimeType.equalsIgnoreCase(MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_SPREADSHEET_TEMPLATE_MACRO)) {
                return true;
            } else if (mimeType.equalsIgnoreCase(MimeTypeHelper.MIMETYPE_VND_OASIS_OPENDOCUMENT_TEXT_TEMPLATE)) {
                return true;
            }

            // Presentation
            if (mimeType.equalsIgnoreCase(MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATION_TEMPLATE)) {
                return true;
            } else if (mimeType.equalsIgnoreCase(MimeTypeHelper.MIMETYPE_VND_OPENXMLFORMATS_OFFICEDOCUMENT_PRESENTATION_TEMPLATE_MACRO)) {
                return true;
            } else if (mimeType.equalsIgnoreCase(MimeTypeHelper.MIMETYPE_VND_OASIS_OPENDOCUMENT_PRESENTATION_TEMPLATE)) {
                return true;
            }
        }

        // get default template target format for extension type, if no valid mime type was given
        if (null != extensionType) {
            if ((extensionType.equalsIgnoreCase("xltx")) || (extensionType.equalsIgnoreCase("xltm")) || (extensionType.equalsIgnoreCase("ots"))) {
                // Spreadsheet
                return true;
            } else if ((extensionType.equalsIgnoreCase("dotx")) || (extensionType.equalsIgnoreCase("dotm")) || (extensionType.equalsIgnoreCase("ott"))) {
                // Text
                return true;
            } else if ((extensionType.equalsIgnoreCase("potx")) || (extensionType.equalsIgnoreCase("potm")) || (extensionType.equalsIgnoreCase("otp"))) {
                // Presentation
                return true;
            }
        }

        return false;
    }
}
