/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.tools.config;

import org.apache.commons.logging.Log;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import com.openexchange.exception.OXException;
import com.openexchange.jslob.DefaultJSlob;
import com.openexchange.jslob.JSlobService;
import com.openexchange.office.tools.osgi.ServiceLookupRegistry;
import com.openexchange.server.ServiceLookup;
import com.openexchange.session.Session;

/**
 * A helper class to read/write user configuration entries in the
 * context of a specific user.
 *
 * @author Carsten Driesner
 */
public class UserConfigurationHelper {

	public enum Mode {
		WRITE_THROUGH,
		WRITE_BACK
	}

    /**
     * Initializes a new {@link UserConfigurationHelper}
     *
     * @param services
     *  The service lookup instance to be used by the new
     *  UserConfigurationHelper instance.
     *
     * @param session
     *  The session of the client in that context the user configuration data
     *  should be retrieved/set.
     *
     * @param id
     *  The root path for the configuration bundle, e.g. (io.ox/office). Must
     *  not be null.
     *
     * @param mode
     *  The work mode of the configuration helper instance. WRITE_THROUGH
     *  always reads/writes from the jslob configuration service, where
     *  WRITE_BACK caches data until flushCache() is called.
     */
	public UserConfigurationHelper(ServiceLookup servicesDEPRECATED, Session session, String id, Mode mode) {
        m_session = session;
        m_id = id;
        m_mode = mode;
    }

	/**
	 * Retrieves a JSONObject from the user configuration referenced by the
	 * provided relative path.
	 *
	 * @param path
	 *  The relative path to he user configuration entry represented as a
	 *  JSONObject.
	 *
	 * @return
	 *  The value as a JSONObject or null if the entry does not exists, or
	 *  the type is not a JSONObject.
	 */
	public JSONObject getJSONObject(String path) {
        Object value = getGenericValue(path);
        return (value instanceof JSONObject) ? (JSONObject)value : null;
	}

	/**
	 * Retrieves a JSONArray from the user configuration referenced by the
	 * provided relative path.
	 *
	 * @param path
	 *  The relative path to he user configuration entry represented as a
	 *  JSONArray.
	 *
	 * @return
	 *  The value as a JSONArray or null if the entry does not exists, or
	 *  the type is not a JSONArray.
	 */
	public JSONArray getJSONArray(String path) {
		Object value = getGenericValue(path);
		return (value instanceof JSONArray) ? (JSONArray)value : null;
	}

	/**
	 * Retrieves a String from the user configuration referenced by the
	 * provided relative path.
	 *
	 * @param path
	 *  The relative path to he user configuration entry represented as a
	 *  String.
	 *
	 * @return
	 *  The value as a String or null if the entry does not exists, or
	 *  the type is not a String.
	 */
	public String getString(String path) {
		Object value = getGenericValue(path);
		return (value instanceof String) ? (String)value : null;
	}

	/**
	 * Retrieves a String from the user configuration referenced by the
	 * provided relative path supporting a default value for missing
	 * entries.
	 *
	 * @param path
	 *  The relative path to he user configuration entry represented as a
	 *  String.
	 *
	 * @param defaultValue
	 *  The default value to be used if the entry is missing.
	 *
	 * @return
	 *  The value as a String or defaultValue if the entry does not
	 *  exists, or the type is not a String.
	 */
	public String getString(String path, String defaultValue) {
		String value = getString(path);
		return (null == value) ? defaultValue : value;
	}
	
	/**
	 * Retrieves a Double from the user configuration referenced by the
	 * provided relative path.
	 *
	 * @param path
	 *  The relative path to he user configuration entry represented as a
	 *  Double.
	 *
	 * @return
	 *  The value as a Double or null if the entry does not exists, or
	 *  the type is not a Double.
	 */
	public Double getDouble(String path) {
		Object value = getGenericValue(path);
		return (value instanceof Double) ? (Double)value : null;
	}

	/**
	 * Retrieves a Double from the user configuration referenced by the
	 * provided relative path supporting a default value for missing
	 * entries.
	 *
	 * @param path
	 *  The relative path to he user configuration entry represented as a
	 *  String.
	 *
	 * @param defaultValue
	 *  The default value to be used if the entry is missing.
	 *
	 * @return
	 *  The value as a Double or null if the entry does not exists, or
	 *  the type is not a Double.
	 */
    public Double getDouble(String path, Double defaultValue) {
    	Double value = getDouble(path);
		return (null == value) ? defaultValue : value;
    }

	/**
	 * Retrieves a Long from the user configuration referenced by the
	 * provided relative path.
	 *
	 * @param path
	 *  The relative path to he user configuration entry represented as a
	 *  Long.
	 *
	 * @return
	 *  The value as a Long or null if the entry does not exists, or
	 *  the type is not a Long.
	 */
	public Long getLong(String path) {
		Object value = getGenericValue(path);
		return ((value instanceof Long) || (value instanceof Integer)) ? (Long)value : null;
	}

	/**
	 * Retrieves a Long from the user configuration referenced by the
	 * provided relative path supporting a default value for missing
	 * entries.
	 *
	 * @param path
	 *  The relative path to he user configuration entry represented as a
	 *  String.
	 *
	 * @param defaultValue
	 *  The default value to be used if the entry is missing.
	 *
	 * @return
	 *  The value as a Long or null if the entry does not exists, or
	 *  the type is not a Long.
	 */
    public Long getLong(String path, Long defaultValue) {
    	Long value = getLong(path);
		return (null == value) ? defaultValue : value;
    }

	/**
	 * Retrieves a Integer from the user configuration referenced by the
	 * provided relative path.
	 *
	 * @param path
	 *  The relative path to he user configuration entry represented as a
	 *  Integer.
	 *
	 * @return
	 *  The value as a Integer or null if the entry does not exists, or
	 *  the type is not a Integer.
	 */
	public Integer getInteger(String path) {
		Object value = getGenericValue(path);
		return (value instanceof Integer) ? (Integer)value : null;
	}

	/**
	 * Retrieves a Integer from the user configuration referenced by the
	 * provided relative path supporting a default value for missing
	 * entries.
	 *
	 * @param path
	 *  The relative path to he user configuration entry represented as a
	 *  String.
	 *
	 * @param defaultValue
	 *  The default value to be used if the entry is missing.
	 *
	 * @return
	 *  The value as a Integer or null if the entry does not exists, or
	 *  the type is not a Integer.
	 */
    public Integer getInteger(String path, Integer defaultValue) {
    	Integer value = getInteger(path);
		return (null == value) ? defaultValue : value;
    }

	/**
	 * Retrieves a Boolean from the user configuration referenced by the
	 * provided relative path.
	 *
	 * @param path
	 *  The relative path to he user configuration entry represented as a
	 *  Boolean.
	 *
	 * @return
	 *  The value as a Boolean or null if the entry does not exists, or
	 *  the type is not a Boolean.
	 */
	public Boolean getBoolean(String path) {
		Object value = getGenericValue(path);
		return (value instanceof Boolean) ? (Boolean)value : null;
	}


	/**
	 * Retrieves a Boolean from the user configuration referenced by the
	 * provided relative path supporting a default value for missing
	 * entries.
	 *
	 * @param path
	 *  The relative path to he user configuration entry represented as a
	 *  String.
	 *
	 * @param defaultValue
	 *  The default value to be used if the entry is missing.
	 *
	 * @return
	 *  The value as a Boolean or null if the entry does not exists, or
	 *  the type is not a Boolean.
	 */
    public Boolean getBoolean(String path, Boolean defaultValue) {
    	Boolean value = getBoolean(path);
		return (null == value) ? defaultValue : value;
    }

    /**
     * Sets the value of the user configuration entry.
     *
     * @param path
	 *  The relative path to the user configuration entry which should get
	 *  the new provided boolean value.
	 *
     * @param value
     *  The new boolean value.
     *
     * @return
     *  TRUE if the value was set successfully, otherwise FALSE.
     */
	public boolean setValue(String path, Boolean value) {
        return setGenericValue(path, value);
    }

    /**
     * Sets the value of the user configuration entry.
     *
     * @param path
	 *  The relative path to the user configuration entry which should get
	 *  the new provided string value.
	 *
     * @param value
     *  The new string value.
     *
     * @return
     *  TRUE if the value was set successfully, otherwise FALSE.
     */
    public boolean setValue(String path, String value) {
        return setGenericValue(path, value);
    }

    /**
     * Sets the value of the user configuration entry.
     *
     * @param path
	 *  The relative path to the user configuration entry which should get
	 *  the new provided integer value.
	 *
     * @param value
     *  The new integer value.
     *
     * @return
     *  TRUE if the value was set successfully, otherwise FALSE.
     */
    public boolean setValue(String path, Integer value) {
    	return setGenericValue(path, value);
    }

    /**
     * Sets the value of the user configuration entry.
     *
     * @param path
	 *  The relative path to the user configuration entry which should get
	 *  the new provided long value.
	 *
     * @param value
     *  The new long value.
     *
     * @return
     *  TRUE if the value was set successfully, otherwise FALSE.
     */
    public boolean setValue(String path, Long value) {
    	return setGenericValue(path, value);
    }

    /**
     * Sets the value of the user configuration entry.
     *
     * @param path
	 *  The relative path to the user configuration entry which should get
	 *  the new provided double value.
	 *
     * @param value
     *  The new double value.
     *
     * @return
     *  TRUE if the value was set successfully, otherwise FALSE.
     */
    public boolean setValue(String path, Double value) {
    	return setGenericValue(path, value);
    }

    /**
     * Sets the value of the user configuration entry.
     *
     * @param path
	 *  The relative path to the user configuration entry which should get
	 *  the new provided JSONObject value.
	 *
     * @param value
     *  The new JSONObject value.
     *
     * @return
     *  TRUE if the value was set successfully, otherwise FALSE.
     */
    public boolean setValue(String path, JSONObject value) {
    	return setGenericValue(path, value);
    }

    /**
     * Sets the value of the user configuration entry.
     *
     * @param path
	 *  The relative path to the user configuration entry which should get
	 *  the new provided JSONArray value.
	 *
     * @param value
     *  The new JSONArray value.
     *
     * @return
     *  TRUE if the value was set successfully, otherwise FALSE.
     */
    public boolean setValue(String path, JSONArray value) {
    	return setGenericValue(path, value);
    }

    /**
     * Refreshes the cache content reading the user configuration again.
     * Be careful as the code doesn't check if the cache is dirty and
     * changes will be lost. Therefore call flushCache() before using
     * refreshCache().
     *
     * @return
     *  TRUE if the cache has been refreshed (also true if no cache is
     *  used -> mode = WRITE_THROUGH), otherwise FALSE.
     */
    public boolean refreshCache() {
    	boolean result = false;

        if ((null != m_cache) && (null != m_jslobService)) {
        	try {
        	    m_cache = m_jslobService.get(m_id, m_session).getJsonObject();
        	    result = true;
        	} catch (OXException e) {
                LOG.error("RT connection: JSONException caught while reading data from JSlobService", e);
       	    }
        } else if (null == m_cache) {
        	result = true;
        }

        return result;
    }
    
    /**
     * Flushes the cache to the jslob configuration service.
     *
     * @return
     *  TRUE if writing back data was successful, otherwise FALSE.
     */
    public boolean flushCache() {
    	boolean result = false;

    	if ((null != m_cache) && (null != m_jslobService)) {
    		try {
    		    m_jslobService.set(m_id, new DefaultJSlob(m_cache), m_session);
    		    m_cache = m_jslobService.get(m_id, m_session).getJsonObject();
    		    result = true;
    		} catch (OXException e) {
                LOG.error("RT connection: JSONException caught while writing data to JSlobService", e);
    		}
    	}

    	return result;
    }

    /**
     * Retrieves a value from the user configuration.
     *
     * @param path
     *  The relative path to the user configuration entry, e.g.
     *  "portal/recents" or "spreadsheet/maxCells".
     *
     * @return
     *  The value or null if the entry doesn't exists. The type of the value
     *  could be a simple type (Boolean, String, Double, Long, Integer), or a
     *  JSON type if the entry has a complex structure (JSONObject, JSONArray).
     */
	private Object getGenericValue(String path) {
        Object value = null;

        initOnDemand();
        if ((null != m_jslobService) && (null != path)) {
            JSONObject configTree = readJSlobConfigTree();
            JSONObject jsonObject = getParentObject(path, configTree);
            String attrName = getAttributeNameFromPath(path);

            if ((null != attrName) && (null != jsonObject)) {
            	// retrieve value from the parent object
        	    value = jsonObject.opt(attrName);
            }
        }

        return value;
    }

	/**
	 * Sets a value to the configuration referenced by the provided path.
	 *
	 * @param path
	 *  The relative path to the configuration entry to be written into the
	 *  user configuration.
	 *
	 * @param value
	 *  The value that should be written to the configuration entry. It must
	 *  be a Boolean/Integer/String for simple types or a JSONObject/JSONArray
	 *  for complex types.
	 *
	 * @return
	 */
    private boolean setGenericValue(String path, Object value) {
        initOnDemand();

        if (null != m_jslobService) {
        	try {
                JSONObject configTree = readJSlobConfigTree();
                JSONObject parent = getParentObject(path, configTree);
                String attrName = getAttributeNameFromPath(path);

                if ((null != attrName) && (null != parent)) {
    			    parent.put(attrName, value);
    			    writeJSlobConfigTree(configTree);
    			    return true;
                }
            } catch (JSONException e) {
                LOG.error("RT connection: JSONException caught while accessing JSlobService", e);
        	}
        }

        return false;
    }

	/**
	 * Retrieves the parent object containing the referenced entry defined
	 * by the provided path.
	 *
	 * @param path
	 *  A relative path to the user configuration entry.
	 *
	 * @return
	 *  The JSONObject which contains the configuration entry or null if the
	 *  parent object does not exists.
	 */
	private JSONObject getParentObject(String path, JSONObject configTree) {
		JSONObject value = null;

		if (null != configTree) {

	        // cascade through the hierarchical structure to find the correct setting
            int i = 0;
            String[] entries = path.split("/");
            Object parent = configTree;

            while (i < (entries.length - 1)) {
                String entry = entries[i++];
                 // ignore empty strings
                if (entry.length() > 0) {
                    Object obj = configTree.opt(entry);
                    if (obj instanceof JSONObject) {
                        // step down on JSONObject
                    	configTree = (JSONObject)obj;
                        parent = configTree;
                    } else {
                    	parent = obj;
                    }
                }
            }

            if ((null != parent) && (parent instanceof JSONObject)) {
            	value = (JSONObject)parent;
            }
		}

		return value;
	}

	/**
	 *  Provides the attribute name of a relative configuration entry path.
	 *
	 * @param path
	 *  A relative configuration entry path.
	 *
	 * @return
	 *  The name of the attribute referenced by the provided path or null
	 *  if the attribute name cannot be determined.
	 */
	private String getAttributeNameFromPath(String path) {
		String attributeName = null;

		if ((null != path) && (path.length() > 0)) {
			int lastIndex = path.lastIndexOf('/');
			if ((lastIndex >= 0) && ((lastIndex + 1) < path.length())) {
				attributeName = path.substring(lastIndex+1);
			} else {
				attributeName = path;
			}
		}

		return attributeName;
	}

    /**
     * Initializes the instance on demand. Reading the configuration values
     * from the user configuration service.
     *
     * @return
     *  TRUE if the initialization was successful otherwise FALSE.
     */
    private boolean initOnDemand() {
        boolean result = false;

        if (!m_bInit && (null != m_session) && (null != m_id)) {
            m_jslobService = ServiceLookupRegistry.get().getService(JSlobService.class);
            if (null != m_jslobService) {
                try {
                	// just to be sure that id exists and system works correctly
                    JSONObject cache = m_jslobService.get(m_id, m_session).getJsonObject();
                    if (m_mode == Mode.WRITE_BACK){
                    	m_cache = cache;
                    }
                    result = true;
                } catch (OXException e) {
                    LOG.error("RT connection: Exception caught while accesing JSlobService", e);
                }
            }

            m_bInit = true;
        }

        return result;
    }
    
    /**
     * Reads the configuration tree dependent on the mode. In mode WRITE_TROUGH
     * the data is automatically read from the jslob service, otherwise the
     * current cache data is returned.
     *
     * @return
     *  The JSONObject containing all configuration tree entries.
     */
    private JSONObject readJSlobConfigTree() {
    	try {
	    	if (null != m_cache) {
	    		return m_cache;
	    	} else if (null != m_jslobService) {
	    		return m_jslobService.get(m_id, m_session).getJsonObject();
	    	}
    	} catch (OXException e) {
    		LOG.error("RT connection: Exception caught while reading JSON config tree from JSlobService", e);
    	}
    	
    	return null;
    }
    
    /**
     * Writes the configuration tree dependent on the mode. In mode
     * WRITE_TROUGH the data is automatically stored via the jslob service,
     * otherwise the data is cached in the internal cache.
     * In case of mode WRITE_BACK, the data must be written back via the
     * jslob service using the flushCache() method.
     *
     * @param configTree
     *  The JSONObject containing all configuration tree entries.
     */
    private void writeJSlobConfigTree(JSONObject configTree) {
    	try {
    		if (null != m_cache) {
    			m_cache = configTree;
    		} else if (null != m_jslobService) {
    			m_jslobService.set(m_id, new DefaultJSlob(configTree), m_session);
    		}
    	} catch (OXException e) {
    		LOG.error("RT connection: Exception caught while write JSON config tree to JSlobService", e);
    	}
    }
    

    private Session m_session = null;
    private String m_id = null;
    private boolean m_bInit = false;
    private JSlobService m_jslobService = null;
    private JSONObject m_cache = null;
    private Mode m_mode = Mode.WRITE_THROUGH;

    private static final Log LOG = com.openexchange.log.Log.loggerFor(UserConfigurationHelper.class);
}
