package com.openexchange.office.tools.common.memory;

import java.lang.management.ManagementFactory;
import java.lang.management.MemoryMXBean;
import java.lang.management.MemoryNotificationInfo;
import java.lang.management.MemoryPoolMXBean;
import java.lang.management.MemoryType;
import java.lang.management.MemoryUsage;
import javax.management.Notification;
import javax.management.NotificationEmitter;
import javax.management.NotificationListener;
import javax.management.openmbean.CompositeData;

import org.slf4j.Logger;
import com.openexchange.exception.ExceptionUtils;
import com.openexchange.java.ConcurrentList;
import com.openexchange.office.tools.common.system.SystemInfoHelper;

/**
 * memory observer class which tracks the heap usage of the
 * greatest heap memory pool.
 *
 * @author Carsten Driesner
 *
 */
public class MemoryObserver implements NotificationListener
{
    private static final Logger LOG = org.slf4j.LoggerFactory.getLogger(MemoryObserver.class);

    private static final MemoryPoolMXBean thresholdMemPool = getThresholdMemPool();
    private static final MemoryObserver memoryObserver = new MemoryObserver();
    private final ConcurrentList<MemoryListener> registeredListeners = new ConcurrentList<MemoryListener>();
    private final double maxPercentageLimit = 0.8; // 80%
    private final long   minFreeHeapMemory  = 131072000; // 128MB minimum free heap space
    private long threshold = 0;

    public interface MemoryListener {
        void memoryTresholdExceeded(long usedMemory, long maxMemory);
    }

    private MemoryObserver() {
        MemoryMXBean mbean = ManagementFactory.getMemoryMXBean();
        NotificationEmitter emitter = (NotificationEmitter) mbean;
        emitter.addNotificationListener(this, null, null);

        setUsageThreshold();
    }

    public static synchronized MemoryObserver getMemoryObserver() {
        return memoryObserver;
    }

    public boolean addListener(MemoryListener listener) {
        return registeredListeners.add(listener);
    }

    public boolean removeListener(MemoryListener listener) {
        return registeredListeners.remove(listener);
    }

    public boolean isUsageThresholdExceeded() {
        boolean exceeded = thresholdMemPool.isCollectionUsageThresholdExceeded();
        if (exceeded) {
            // check memory usage manually as the call is true although
            // memory consumption has fallen below threshold
            final MemoryUsage memUsage = thresholdMemPool.getUsage();
            long memThreshold = getMemPoolThreshold();
            exceeded = (memUsage.getUsed() > memThreshold);
            if (exceeded) {
                final long maxMemory = memUsage.getMax();
                final long usedMemory = memUsage.getUsed();
                LOG.debug("RT connection: Memory observer detects exceeding memory threshold! Max memory: {}, limit: {}, used memory: {}", maxMemory, memThreshold, usedMemory);
            }

            // Attention: treshold needs to be configured again ... JVM wont notify us about new treshold exceeds otherwise !
            //            It's some kind of "refresh" we trigger here ;-)
            thresholdMemPool.setCollectionUsageThreshold(memThreshold);
        }

        return exceeded;
    }

    public boolean willMemoryThresholdExceeded(long additonalMemNeeded) {
        final MemoryUsage memUsage = thresholdMemPool.getUsage();
        long memThreshold = getMemPoolThreshold();
        return ((memUsage.getUsed() + additonalMemNeeded) > memThreshold);
    }

    private void setUsageThreshold() {
        long maxMemory = thresholdMemPool.getUsage().getMax();
        long thresh1 = (long)(maxMemory * maxPercentageLimit);
        long thresh2 = maxMemory - minFreeHeapMemory;

        // we only want to be notified after a comprehensive gc run
        threshold = (thresh2 > thresh1) ? thresh2 : thresh1;

        thresholdMemPool.setCollectionUsageThreshold(threshold);
        LOG.debug("RT connection: Memory observer uses memory pool: {} with maxMemory: {}, threshold: {}", thresholdMemPool.getName(), maxMemory, threshold);
    }

    private static MemoryPoolMXBean getThresholdMemPool() {
        MemoryPoolMXBean thresholdMemPool = null;
        for (MemoryPoolMXBean pool : ManagementFactory.getMemoryPoolMXBeans()) {
            if (pool.getType() == MemoryType.HEAP && pool.isUsageThresholdSupported()) {
                thresholdMemPool = pool;
                break;
            }
        }
        return thresholdMemPool;
    }

    private long getMemPoolThreshold() {
        return threshold;
    }

    /**
     * Invoked when a JMX notification occurs. The registered listeners are
     * called to react on exceeding the memory threshold.
     */
    @Override
    public void handleNotification(Notification n, Object jb) {
        try {
            if (n.getType().equals(MemoryNotificationInfo.MEMORY_COLLECTION_THRESHOLD_EXCEEDED)) {
                final CompositeData cd = (CompositeData) n.getUserData();
                final MemoryNotificationInfo info = MemoryNotificationInfo.from(cd);
                final MemoryUsage memUsage = info.getUsage();

                // retrieve memory data from the notification
                final long maxMemory = memUsage.getMax();
                final long usedMemory = memUsage.getUsed();
                final long memLimit = threshold;

                // Check limits manually, there are situations were the notification is sent
                // although memory threshold is NOT reached.
                LOG.debug("RT connection: Memory observer notified due to exceeding memory threshold! Max memory: {}, limit: {}, used memory: {}", maxMemory, memLimit, usedMemory);

                final java.util.List<MemoryListener> listeners = registeredListeners.getSnapshot();
                for (MemoryListener listener : listeners) {
                    try {
                        listener.memoryTresholdExceeded(usedMemory, maxMemory);
                    } catch (Throwable e) {
                        // nothing can be done, but do not swallow severe VM errors
                        LOG.warn("{} failed to handle exceeded memory threshold", listener.getClass().getName(), e);
                        ExceptionUtils.handleThrowable(e);
                    }
                }
            }
        } catch (Throwable t) {
            LOG.warn("Failed to handle JMX notification", t);
            throw t;
        }
    }

    public static long calcMaxHeapSize (int percent) {
        final SystemInfoHelper.MemoryInfo memInfo = SystemInfoHelper.getMemoryInfo();
        if (null == memInfo) {
            return 0L;
        }
        final long maxHeapSize = Math.round(((double)memInfo.maxHeapSize * (double)percent) / 100);
        return maxHeapSize;
    }
}
