/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.tools.common;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.Serializable;

import org.apache.commons.codec.binary.Base64;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.Validate;

//=============================================================================
/** Class serialization inside an OSGI environment can be some kind of pane.
 *
 *  A typical problem : on deserialization the wrong (default) class loader
 *  is used and is not able to load right class within it's context.
 *
 *  This helper solve the problem by hooking into the class loader hierarchy
 *  and select the right loader instead of using the default class loader.
 */
public class ClassContextAwareSerializationUtils
{
	//-------------------------------------------------------------------------
	/** Utility classes should not provide a way to create object instances
	 */
	private ClassContextAwareSerializationUtils ()
	{}

	//-------------------------------------------------------------------------
	/** serialize the given object to a base64 encoded string.
	 *
	 *  @param	aObject [IN]
	 *  		the object to be serialized here.
	 *  		Has not to be null.
	 *
	 *	@return the base64 encoded data set of those object.
	 */
	public static < T > String serializeObject (final T aObject)
	    throws Exception
	{
		Validate.notNull(aObject, "Object for serialization has not to be null.");

		final byte[] lRaw    = org.apache.commons.lang3.SerializationUtils.serialize((Serializable)aObject);
		final byte[] lBase64 = Base64.encodeBase64(lRaw);
		final String sData   = new String (lBase64);
		return sData;
	}

	//-------------------------------------------------------------------------
	@SuppressWarnings("unchecked")
	public static < T > T deserializeObject (final String sData   ,
											 final Object aContext)
	    throws Exception
	{
		Validate.notEmpty(sData   , "Data for de-serialization has not to be empty."  );
		Validate.notNull (aContext, "Serialization context object has not to be null.");

		final byte[]            lBase64     = sData.getBytes();
		final byte[]            lRaw        = Base64.decodeBase64(lBase64);
		      InputStream       aRawData    = null;
		      ObjectInputStream aObjectData = null;

		try
		{
			aRawData    = new ByteArrayInputStream(lRaw);
			aObjectData = new ClassContextAwareObjectInputStream(aRawData, aContext);

			final Object aObject = aObjectData.readObject();
			return (T) aObject;
		}
		finally
		{
			IOUtils.closeQuietly(aObjectData);
			IOUtils.closeQuietly(aRawData   );
		}
	}
}
