/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.rest;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.TimeZone;
import java.util.UUID;
import com.openexchange.office.message.OperationHelper;
import com.openexchange.office.tools.common.TimeZoneHelper;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.json.JSONObject;

public class LogPerformanceDataWriter extends AsyncFileWriter<JSONObject> {
    public static final long MAX_PERFORMANCE_LOG_FILESIZE  = 32 * 1024 * 1024; // 32 MB

    @SuppressWarnings("deprecation")
    private static final Log LOG = com.openexchange.log.Log.loggerFor(LogPerformanceDataWriter.class);
    private static final String LOG_PERFORMANCE_THREAD_NAME = "com.openexchange.office.rest.AsyncPerformanceLogger";
    private static final int  MAX_NUM_OF_RING_LOGFILES      = 2; // must be at least 1 (2 or more for real ring-buffer)
    private static final char NUM_SEPARATOR_CHAR            = '.';

    private int currentFileNumber = 0;
    private String currentFilePath;

    public LogPerformanceDataWriter() {
        super(LOG_PERFORMANCE_THREAD_NAME);

    }

    @Override
    public void init(String filePath) throws Exception {
        super.init(filePath);

        if (StringUtils.isNotEmpty(filePath)) {
            currentFileNumber = -1;
            currentFilePath   = getNextFilePath();
        }
    }

    /**
     * Writes the performance data to the file referenced by the
     * provided logging file path.
     *
     * @param loggingFilePath a file path to the logging file MUST NOT be null or empty
     * @param performanceData the performance data to be written to the log file
     */
    @Override
    protected void writeData(JSONObject performanceData) {
        String uid = UUID.randomUUID().toString().substring(0, 8); // only using 8 digits

        String app = performanceData.optString("application", "");

        // Saving performance data in log file
        try (BufferedWriter bufferedWriter = new BufferedWriter(new FileWriter(getCurrentRingFile(), true))) {
            if (app.equals("viewer")) {
                OperationHelper.logViewerPerformanceData(performanceData, bufferedWriter);
            } else if (app.equals("text") || app.equals("presentation") || app.equals("spreadsheet")) {
                OperationHelper.logEditorPerformanceData(performanceData, bufferedWriter);
            }
        } catch (Exception e) {
            LOG.debug("LogPerformanceData writing to log file failed due to exception", e);
        }

    }

    @Override
    protected void logErrorMessage(final String sMsg, final Exception e) {

        if (null != e)
            LOG.error(sMsg, e);
        else
            LOG.error(sMsg);
    }

    private File getCurrentRingFile() throws Exception {
        File aFile = new File(currentFilePath);

        if (aFile.exists() && aFile.length() > MAX_PERFORMANCE_LOG_FILESIZE) {
            currentFilePath = getNextFilePath();
            aFile = new File(currentFilePath);
            truncateOrCreateFile(aFile);
        }

        return aFile;
    }

    private String getNextFilePath() throws Exception {
        currentFileNumber = ((currentFileNumber + 1) % MAX_NUM_OF_RING_LOGFILES);

        String filePath = getFilePath();
        if (currentFileNumber == 0)
            return filePath;

        final StringBuilder aTmp = new StringBuilder(getFilePath());
        aTmp.append(NUM_SEPARATOR_CHAR);
        aTmp.append(currentFileNumber);
        return aTmp.toString();
    }

    private void truncateOrCreateFile(final File aFile) throws Exception {
        if (aFile.exists()) {
            try (final FileOutputStream oStream = new FileOutputStream(aFile)) {
                oStream.getChannel().truncate(0);
                oStream.close();
            }
        } else {
            aFile.createNewFile();
        }
    }

    /**
     * The current date in ISO 8601 format.
     *
     * @param useSeconds
     *  Whether seconds shall be included into the result string or not.
     *
     * @return
     *  The date as string in ISO 8601 format.
     */
    private static String getISODateFormat(Boolean useSeconds) {

        TimeZone tz = TimeZoneHelper.getTimeZone("UTC");
        DateFormat df = null;

        if (useSeconds) {
            df = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss'Z'");
        } else {
            df = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm':00Z'");
        }
        df.setTimeZone(tz);
        return df.format(new Date());
    }
}
