/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.rest;

import java.io.ByteArrayInputStream;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.Collections;
import java.util.HashSet;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;

import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.validator.routines.UrlValidator;
import org.json.JSONException;
import org.json.JSONObject;

import com.openexchange.ajax.requesthandler.AJAXRequestData;
import com.openexchange.ajax.requesthandler.AJAXRequestResult;
import com.openexchange.config.ConfigurationService;
import com.openexchange.exception.OXException;
import com.openexchange.groupware.upload.UploadFile;
import com.openexchange.imagetransformation.ImageMetadata;
import com.openexchange.imagetransformation.ImageMetadataOptions;
import com.openexchange.imagetransformation.ImageMetadataService;
import com.openexchange.net.HostList;
import com.openexchange.office.document.DocFileHelper;
import com.openexchange.office.imagemgr.Resource;
import com.openexchange.office.imagemgr.ResourceManager;
import com.openexchange.office.rest.tools.RESTException;
import com.openexchange.office.tools.common.memory.MemoryObserver;
import com.openexchange.office.tools.doc.StreamInfo;
import com.openexchange.office.tools.error.ErrorCode;
import com.openexchange.office.tools.files.StorageHelper;
import com.openexchange.office.tools.osgi.ServiceLookupRegistry;
import com.openexchange.office.tools.rest.RequestDataHelper;
import com.openexchange.server.ServiceLookup;
import com.openexchange.tools.encoding.Base64;
import com.openexchange.tools.net.URITools;
import com.openexchange.tools.session.ServerSession;

/**
 * {@link AddFileAction}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 */
// @Action(method = RequestMethod.POST, name = "addfile", description = "Get the whole document as PDF file or get the substream of an infostore file in its naive format.", parameters = {
//    @Parameter(name = "session", description = "A session ID previously obtained from the login module."),
//    @Parameter(name = "add_fileid", description = "Object ID of the requested infoitem."),
//    @Parameter(name = "add_folderid", description = "Folder ID of the requested infoitem."),
//    @Parameter(name = "uid", description = "The unique id of the client application."),
//    @Parameter(name = "version", optional=true, description = "If present, the infoitem data describes the given version. Otherwise the current version is returned."),
//    @Parameter(name = "filename", optional=true, description = "If present, this parameter contains the name of the infostore item to be used as initial part of the filename for the returned file, other the filename is set to 'file'."),
//    @Parameter(name = "add_filename", optional=true, description = "If present, this parameter contains the name of the just uploaded file to be set as filename for the new file content of the document. Otherwise a system time based name is generated.")
//}, responseDescription = "Response with timestamp: A JSON object containing the actual file metadata and the fragment name of the just uploaded file within the document: 'id', 'folder_id', 'version' 'filename' and 'added_fragment'.")

public class AddFileAction extends DocumentRESTAction {

    public static final String HOST_BLACKLIST_DEFAULT = "127.0.0.1-127.255.255.255,localhost";

    private static final Set<String> SUPPORTED_IMAGE_TYPES;
    static {
        final Set<String> supportedImageTypes = new HashSet<>();
        supportedImageTypes.add("png");
        supportedImageTypes.add("jpg");
        supportedImageTypes.add("jpeg");
        supportedImageTypes.add("gif");
        supportedImageTypes.add("tif");
        supportedImageTypes.add("tiff");
        supportedImageTypes.add("bmp");
        SUPPORTED_IMAGE_TYPES = Collections.unmodifiableSet(supportedImageTypes);
    }

    private static final int MAX_IMAGE_SIZE_IN_MB = 10;
    private static final int MAX_URL_LENGTH = 255;
    private static final Log LOG = com.openexchange.log.Log.loggerFor(AddFileAction.class);

    static final protected String m_resourceIdKey = "added_fileid";
    static final protected String m_resourceNameKey = "added_filename";
    static final protected String m_connectionResourceIdKey = "resourceid";

    private long maxImageSizeSupported = MAX_IMAGE_SIZE_IN_MB * 1024L * 1024L;

    // - Members ---------------------------------------------------------------

    private HostList blackImageUrlHostlist;
    private String blackImageUrlHostlistAsStr = HOST_BLACKLIST_DEFAULT;
    private HostList whiteImageUrlHostlist;
    private String whiteImageUrlHostlistAsStr = "";
    private int maxUrlLength;


    /**
     * Initializes a new {@link AddFileAction}.
     *
     * @param services
     * @param oqm
     */
    public AddFileAction(ServiceLookup servicesDEPRECATED, ResourceManager resMgr) {
        super(null);
        resourceManager = resMgr;

        final ConfigurationService configService = ServiceLookupRegistry.get().getService(ConfigurationService.class);
        if (null != configService) {
            maxImageSizeSupported = configService.getIntProperty("com.openexchange.office.maxImageSize", MAX_IMAGE_SIZE_IN_MB) * 1024L * 1024L;
            blackImageUrlHostlistAsStr = configService.getProperty("com.openexchange.office.upload.blacklist", HOST_BLACKLIST_DEFAULT);
            whiteImageUrlHostlistAsStr = configService.getProperty("com.openexchange.office.upload.whitelist", "");
        }

        blackImageUrlHostlist = HostList.valueOf(blackImageUrlHostlistAsStr);
        whiteImageUrlHostlist = HostList.valueOf(whiteImageUrlHostlistAsStr);
        maxUrlLength = configService.getIntProperty("com.openexchange.office.maxUrlLength", MAX_URL_LENGTH);
    }

    /*
     * (non-Javadoc)
     * @see com.openexchange.ajax.requesthandler.AJAXActionService#perform(com.openexchange.ajax.requesthandler.AJAXRequestData,
     * com.openexchange.tools.session.ServerSession)
     */
    @Override
    public AJAXRequestResult perform(AJAXRequestData request, ServerSession session) throws OXException {
        final RequestDataHelper requestDataHelper = new RequestDataHelper(request);
        final JSONObject jsonObj = new JSONObject();
        AJAXRequestResult ajaxResult = null;
        Integer crc32 = null;

        try {
            final String p = requestDataHelper.getParameter("add_crc32");
            if (p != null) {
                crc32 = Integer.valueOf((int) (Long.parseLong(p, 16)));
            }
        } catch (NumberFormatException e) {
            //
        }

        final String extension = requestDataHelper.getParameter("add_ext");
        final String data64 = requestDataHelper.getParameter("add_filedata");
        final String fileId = requestDataHelper.getParameter("add_fileid");
        final String folderId = requestDataHelper.getParameter("add_folderid");
        final String imageUrlAsStr = requestDataHelper.getParameter("add_imageurl");

        byte[] data = null;

        if ((null != fileId) && (null != folderId)) {
            final DocFileHelper fileHelper = new DocFileHelper(null);
            final StorageHelper storageHelper = new StorageHelper(null, session, folderId);
            final StreamInfo streamInfo = fileHelper.getDocumentStream(session, storageHelper, folderId, fileId, "");
            final InputStream inputStream = streamInfo.getDocumentStream();
            if (null != inputStream) {
                try {
                    data = IOUtils.toByteArray(inputStream);
                } catch (IOException e) {
                    LOG.error("Could not read file stream", e);
                } finally {
                    IOUtils.closeQuietly(inputStream);
                    try {
                        streamInfo.close();
                    } catch (IOException e) {
                        LOG.warn("Could not correctly close IDBasedFileAccess instance", e);
                    }
                }
            } else {
                try {
                    streamInfo.close();
                } catch (IOException e) {
                    LOG.warn("Could not correctly close IDBasedFileAccess instance", e);
                }
            }
        } else if (StringUtils.isNotEmpty(imageUrlAsStr)) {
            try {
                data = getImageDataFromUrl(imageUrlAsStr);
            } catch (RESTException e) {
                final ErrorCode errorCode = e.getErrorCode();
                final JSONObject requestResult = errorCode.getAsJSONResultObject();
                ajaxResult = getAjaxRequestResult(requestResult);
            }
        } else {
            if (null != data64) {

                final String searchPattern = "base64,";
                int pos = data64.indexOf(searchPattern);

                if ((-1 != pos) && ((pos += searchPattern.length()) < (data64.length() - 1))) {
                    data = Base64.decode(data64.substring(pos));
                }
            } else { // in IE9 data64 is not readable from request
                InputStream uploadStm = null;

                try {
                    if (request.hasUploads()) {
                        final UploadFile uploadFile = request.getFiles().get(0);
                        uploadStm = new FileInputStream(uploadFile.getTmpFile());
                    }
                } catch (final Exception e) {
                    LOG.error("Could not create file stream", e);
                }

                if (null != uploadStm) {
                    try {
                        data = IOUtils.toByteArray(uploadStm);
                    } catch (IOException e) {
                        LOG.error("Could not read file stream", e);
                    } finally {
                        IOUtils.closeQuietly(uploadStm);
                    }
                }
            }
        }

        if ((data != null) && (data.length > 0)) {

            if (crc32 == null)
                crc32 = Integer.valueOf(Resource.getCRC32(data));

            long uid = resourceManager.addResource(crc32.intValue(), data);

            try {
                String resname = null;
                switch (DocFileHelper.getDocumentFormat(request.getParameter("filename"))) {
                case DOCX: {
                    resname = "word/media/uid" + Long.toHexString(uid) + "." + extension;
                    break;
                }
                case PPTX: {
                    resname = "ppt/media/uid" + Long.toHexString(uid) + "." + extension;
                    break;
                }
                case ODT: case ODS: case ODP: {
                    resname = "Pictures/uid" + Long.toHexString(uid) + "." + extension;
                    break;
                }
                case XLSX: {
                    resname = "xl/media/uid" + Long.toHexString(uid) + "." + extension;
                    break;
                }

                case ODG:
                case NONE:
                default:
                    break;
                }

                if (resname != null) {
                    final Resource resource = resourceManager.getResource(uid);
                    String managedResourceId = null;

                    // add the managed file id to the result object
                    if ((null != resource) && (null != (managedResourceId = resource.getManagedId()))) {
                        jsonObj.put(m_resourceIdKey, managedResourceId);
                    }

                    // add the filename to the result object
                    jsonObj.put(m_resourceNameKey, resname);
                }
            } catch (Exception e1) {
                LOG.error("Could not add to resource manager", e1);
            }
        }

        // set resource id at connection object; return the
        // object with fragment name and id in case of success
        if (jsonObj.has(m_resourceIdKey)) {
            final JSONObject requestData = new JSONObject();

            try {
                requestData.put(m_connectionResourceIdKey, jsonObj.getString(m_resourceIdKey));

                JSONObject resultData = sendConnectionMessageSynchronously(request, session, "addResource", "resource", requestData);

                if ((null != resultData) && resultData.has(m_connectionResourceIdKey) && (resultData.getString(m_connectionResourceIdKey).length() > 0) && jsonObj.has(m_resourceNameKey)) {
                    resultData.put(m_resourceNameKey, jsonObj.get(m_resourceNameKey));
                } else {
                    resultData = null;
                }

                ajaxResult = getAjaxRequestResult(resultData);
            } catch (JSONException e) {
                LOG.error("Could not send resource to realtime session", e);
            }
        }

        return ajaxResult;
    }

    /**
     * Validates the given URL according to whitelisted prtocols ans blacklisted hosts.
     *
     * @param url The URL to validate
     * @return An optional OXException
     */
    private Function<URL, Optional<OXException>> validator = (url) -> {

      	String urlasStr = url.toString();
    	if (urlasStr.length() > maxUrlLength) {
    		return Optional.of(new OXException(new RESTException("The length " + blackImageUrlHostlistAsStr.length() + " is to long!", ErrorCode.GENERAL_ARGUMENTS_ERROR)));
    	}    	    	
    	
        final UrlValidator defaultValidator = new UrlValidator();
        if (!defaultValidator.isValid(url.toString())) {
            return Optional.of(new OXException(new RESTException(ErrorCode.GENERAL_ARGUMENTS_ERROR)));
        }
        String host = url.getHost();
        if (StringUtils.isBlank(whiteImageUrlHostlistAsStr)) {
            if (blackImageUrlHostlist.contains(host)) {
                return Optional.of(new OXException(new RESTException("Host " + host + " is on blacklist!", ErrorCode.GENERAL_ARGUMENTS_ERROR)));
            }
        } else {
            if (!whiteImageUrlHostlist.contains(host)) {
                if (blackImageUrlHostlist.contains(host)) {
                    return Optional.of(new OXException(new RESTException("Host " + host + " is not on whithlist and is on blacklist!", ErrorCode.GENERAL_ARGUMENTS_ERROR)));
                }
            }
        }
        return Optional.empty();
    };

    private byte[] getImageDataFromUrl(String fileURL) throws RESTException {
    	byte[] result = null;
    	try {
        	HttpURLConnection httpConn;
        	try {
        		httpConn = (HttpURLConnection) URITools.getTerminalConnection(fileURL, Optional.of(validator), Optional.empty());
            } catch (Exception e) {
                throw new RESTException(e.getMessage(), ErrorCode.GENERAL_ARGUMENTS_ERROR);
            }
    		try {
		        int responseCode = httpConn.getResponseCode();
		 
		        // always check HTTP response code first
		        if (responseCode == HttpURLConnection.HTTP_OK) {
		            int contentLength = httpConn.getContentLength();
		 
		            if (contentLength > maxImageSizeSupported) {
		                final ErrorCode errorCode = ErrorCode.GENERAL_IMAGE_SIZE_EXCEEDS_LIMIT_ERROR;
		                errorCode.setValue(new Long(maxImageSizeSupported / (1024L * 1024L)).toString());
		                throw new RESTException("Image size exceeds limit " + maxImageSizeSupported, errorCode);
		            }
		            final boolean notEnoughMemory = MemoryObserver.getMemoryObserver().willMemoryThresholdExceeded(contentLength);
		            if (notEnoughMemory) {
		                throw new RESTException("Memory threshold will be exceeded due to reading image data in memory", ErrorCode.GENERAL_MEMORY_TOO_LOW_ERROR);
		            }    		
		            // opens input stream from the HTTP connection
		            InputStream inputStream = httpConn.getInputStream();            
		            result = IOUtils.toByteArray(inputStream, contentLength);
		            String imageType = null;
		            if (null != result) {
		                imageType = imageTypeDetection(new ByteArrayInputStream(result));
		                if (StringUtils.isEmpty(imageType) || !SUPPORTED_IMAGE_TYPES.contains(imageType)) {
		                    throw new RESTException(ErrorCode.GENERAL_IMAGE_TYPE_NOT_SUPPORTED_ERROR);
		                }
		            } else {
		                throw new RESTException(ErrorCode.GENERAL_UNKNOWN_ERROR);
		            }
		        }
	        } finally {
	        	httpConn.disconnect();
			}	                
	    } catch (RESTException e) {
	        LOG.debug("RESTException caught while trying to get image data from provided url", e);
	        throw e;
	    } catch (Exception e) {
	        LOG.error("Error while processing the retrieved information: " + e.getMessage(), e);
	        throw new RESTException(ErrorCode.GENERAL_UNKNOWN_ERROR);
	    }        
    	return result;
    }    

    private static String imageTypeDetection(InputStream imageInputStm) throws OXException {

        final ImageMetadataService metadataService = ServiceLookupRegistry.get().getService(ImageMetadataService.class);

        String imageFormatName = null;
        if ((null != imageInputStm) && (null != metadataService)) {
            try {
                final ImageMetadata imageMetadata = metadataService.getMetadataFor(imageInputStm, null, null, ImageMetadataOptions.builder().withDimension().withFormatName().build());
                imageFormatName = imageMetadata.getFormatName();
            } catch (IOException e) {
                LOG.trace(e.getMessage());
            }
        }

        return imageFormatName;
    }

    // - Members ---------------------------------------------------------------

    private ResourceManager resourceManager = null;
}
