/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.realtime.impl;

import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.lang3.StringUtils;

import com.openexchange.config.ConfigurationService;
import com.openexchange.office.realtime.impl.Connection.FailSafeSaveReason;
import com.openexchange.office.tools.error.ErrorCode;
import com.openexchange.office.tools.osgi.ServiceLookupRegistry;
import com.openexchange.server.ServiceLookup;

/**
 * A connection processor which checks, if connection instances should be saved
 * in the background using certain criteria.
 *
 * @author <a href="mailto:carsten.driesner@open-xchange.com">Carsten Driesner</a>
 * @since v7.8.0
 */
public class FailSafeSaveProcessor implements IConnectionProcessor {
    private static final org.apache.commons.logging.Log LOG = com.openexchange.log.LogFactory.getLog(FailSafeSaveProcessor.class);
    private static final int MIN_OPERATIONS_FOR_FAILSAFESAVE = 50;
    private static final int MIN_TIME_FOR_MANY_OPS_FAILSAFESAVE = 60 * 1000; // time in milliseconds
    // DOCS-1063 - lower time for a fail safe save to 5 minutes
    private static final int MAX_TIME_FOR_MODIFIED_FAILSAFESAVE = 5 * 60 * 1000; // time in milliseconds

    private Date m_lastProcessed = null;
    private boolean m_clearInsteadOfSave = false;

    public FailSafeSaveProcessor(final ServiceLookup servicesDEPRECATED) {
        final ConfigurationService configuration = ServiceLookupRegistry.get().getService(com.openexchange.config.ConfigurationService.class);
        if (null != configuration) {
            m_clearInsteadOfSave = configuration.getBoolProperty("io.ox/office//module/fssResetOperationQueue", false);
            if (m_clearInsteadOfSave) {
                LOG.warn("RT connection: Warning - fail safe save is configured to clear the operation queue of the documents - this is for sizing tests only!");
            }
        }
    }

    /**
     * Starts a processing round for fail safe save.
     *
     * @param name A name which describes what document type should be checked
     *             for modified documents, e.g. text, spreadsheet
     */
    @Override
    public void processStart(final String name) {
        LOG.trace("RT connection: Fail safe save checks for documents to save for: " + (StringUtils.isEmpty(name) ? "unknown" : name));
        m_lastProcessed = new Date();
    }

    /**
     * Process a specific connection, while trying to use not more time than
     * provided.
     *
     * @param connection The connection which should be checked and saved by
     *                   fail safe save.
     * @param maxTime    The maximum time to be used for the check/save process
     *                   in milliseconds.
     */
    @Override
    public boolean process(final Connection connection, long maxTime)
    {
        boolean continueToProcess = true;

        try {
            Date nowStart = new Date();
            long timeElapsed = 0;

            if (null != m_lastProcessed) {
                timeElapsed = getTimeDifference(m_lastProcessed, nowStart);
            }

            if (timeElapsed < maxTime) {
                final FailSafeSaveReason failSafeSaveReason[] = { null };

                // Don't try to save, if the connection has already been disposed, we are
                // in a middle of a save or we were not able to save the document correctly.
                if (mustBeProcessed(connection, nowStart, failSafeSaveReason)) {
                    if (m_clearInsteadOfSave) {
                        connection.resetOperationQueue();
                        final Date nowEnd = new Date();
                        LOG.debug("RT connection: Fail safe save in sizing-test-mode - clearing operation queue needed: " + getTimeDifference(nowStart, nowEnd) + "ms");
                    } else {
                        connection.failSafeSaveDocument(failSafeSaveReason[0]);
                        final Date nowEnd = new Date();
                        long diff = getTimeDifference(nowStart, nowEnd);
                        if (diff > (maxTime / 2)) {
                            LOG.info("RT connection: Fail safe save - saving document needed: " + diff + "ms");
                        } else {
                            LOG.debug("RT connection: Fail safe save - saving document needed: " + diff + "ms");
                        }
                    }
                }
            } else {
                continueToProcess = false;
                LOG.trace("RT connection: Fail safe save - time is up");
            }
        } catch (Exception e) {
            // Ignore if saving doesn't work. This is just an additional
            // periodically save in the background.
            LOG.error("RT connection: Exception caught during failSafeSaveDocument", e);
        }

        return continueToProcess;
    }

    /**
     * Completes the processing round.
     */
    @Override
    public void processEnd() {
        LOG.trace("RT connection: Fail safe save process check round finished - waiting for next activation");
        m_lastProcessed = null;
    }

    /**
     * Provides a list of connections which should be processed by the fail
     * safe save processor.
     */
    @Override
    public List<WeakReference<Connection>> getConnectionsToBeProcessed(List<WeakReference<Connection>> sourceCollection) {
        List<WeakReference<Connection>> filteredList = sourceCollection;

        if ((null != sourceCollection) && (sourceCollection.size() > 0)) {
            final ArrayList<WeakReference<Connection>> filteredCollection = new ArrayList<WeakReference<Connection>>();
            final Iterator<WeakReference<Connection>> iter = sourceCollection.iterator();
            final Date nowStart = new Date();

            while (iter.hasNext()) {
                final WeakReference<Connection> connection = iter.next();
                if (mustBeProcessed(connection.get(), nowStart, null)) {
                    filteredCollection.add(connection);
                }
            }
            filteredList = filteredCollection;
        }

        return filteredList;
    }

    /**
     * Determines, if a connection must be processed or not by fail safe save.
     *
     * @param connection A connection to be checked for processing.
     * @param now The current time
     * @param failSafeSaveReason[] Output parameter, whose first array value is
     *           set to the reason for the save that must be processed or null
     *           when there's no need to process saving. This parameter may be null
     * @return TRUE, if the connection must be processed in the current round, otherwise
     *         FALSE.
     */
    private boolean mustBeProcessed(final Connection connection, final Date now, FailSafeSaveReason[] failSafeSaveReason) {
        boolean bProcess = false;

        if  (((connection != null) &&
             !connection.isDisposed() &&
             !connection.isSaveInProgress() &&
             (connection.getLastFailSafeSaveError().getCode() == ErrorCode.NO_ERROR.getCode()))) {
            long diff = getTimeDifference(connection.getSaveTimeStamp(), now);
            long ops = connection.getPendingOperationsCount();
            FailSafeSaveReason saveReason = ((diff >= MIN_TIME_FOR_MANY_OPS_FAILSAFESAVE) && (ops >= MIN_OPERATIONS_FOR_FAILSAFESAVE)) ?
                                                FailSafeSaveReason.TOO_MANY_OPS : (((diff >= MAX_TIME_FOR_MODIFIED_FAILSAFESAVE) && (ops > 0)) ?
                                                    FailSafeSaveReason.MODIFICATION_TIMEOUT : null);

            // set possible saveReason or null at output parameter
            if ((null != failSafeSaveReason) && (failSafeSaveReason.length > 0)) {
                failSafeSaveReason[0] = saveReason;
            }

            bProcess = (null != saveReason);
        }

        return bProcess;
    }

    /**
     * Calculates the absolute difference between two Dates
     *
     * @param d1 Date1
     * @param d2 Date2
     * @return The difference in milliseconds.
     */
    private long getTimeDifference(Date d1, Date d2) {
        return Math.abs(d1.getTime() - d2.getTime());
    }
}
