/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.message;

import java.io.BufferedWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.Writer;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.ListIterator;
import java.util.TimeZone;
import java.util.UUID;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.Validate;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.exception.OXException;
import com.openexchange.office.filter.api.OCKey;
import com.openexchange.office.filter.api.OCValue;
import com.openexchange.office.imagemgr.Resource;
import com.openexchange.office.imagemgr.ResourceManager;
import com.openexchange.office.tools.common.TimeZoneHelper;
import com.openexchange.office.tools.error.ErrorCode;
import com.openexchange.office.tools.json.JSONDebugWriter;
import com.openexchange.office.tools.json.JSONHelper;
import com.openexchange.realtime.packet.ID;
import com.openexchange.tools.encoding.Base64;

/**
 * {@link RealtimeHelper}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 */
public class OperationHelper {

    public final static String KEY_ACTIONS = "actions";
    public final static String KEY_OPERATIONS = "operations";
    public final static String KEY_OSN = "osn";
    public final static String KEY_OPL = "opl";

    /**
     * Comparing, if two positions in the form of JSONArrays have the same
     * parents, grand parents, ... parentLevel = 1 means, both positions
     * have the same parent. parentLevel = 2 means, the grand parents are
     * identical and so on.
     *
     * @param position1
     *  The first position to be compared
     *
     * @param position2
     *  The second position to be compared
     *
     * @param parentLevel
     *  The level, until which the ancestors will be compared. parentLevel = 1
     *  means, that the direct parents must be identical.
     *
     * @return
     *  If the two positions have the same ancestor until the specified
     *  level, true is returned. Otherwise false
     */
    static private boolean hasSameParentComponent(JSONArray position1, JSONArray position2, int parentLevel) {

        boolean sameParent = true;

        if ((position1 == null) || (position2 == null)) {
            return false;
        }

        int length = position1.length();

        if ((length < parentLevel) || (length != position2.length())) {
            return false;
        }

        for (int index = length - parentLevel - 1; index >= 0; index -= 1) {

            try {
                if (position1.getInt(index) != position2.getInt(index)) {
                    return false;
                }
            } catch (JSONException e) {
                return false;
            }
        }

        return sameParent;
    }

    /**
     * Trying to merge two directly following insertText operations. If the
     * merge is possible the operation 'next' that follows the operation
     * 'last' must not be added to the operation array. Therefore the return
     * value 'insertNextOp' will be set to false in the case of successful
     * merging. In this case the content of the operation 'last' will be
     * modified.
     *
     * @param last
     *  The preceding insertText operation. This was already added to the
     *  operations array. Therefore it will be modified within this operation.
     *  And therefore it is necessary, that the insertText operations are
     *  copied before they are added to the operations array, so that the
     *  original operations are not modified.
     *
     * @param position2
     *  The following insertText operation. If merging this two operations
     *  is possible, then this operation will not be added to the operations
     *  array. Therefore the return value of this function is set to
     *  'insertNextOp' false.
     *
     * @return
     *  If the two insertText operations can be merged, it is not necessary
     *  to add the second operation to the operations array. Therefore the
     *  return value will be false. Otherwise it is true.
     *
     * @throws JSONException
     */
    static private boolean mergeInsertOperations(JSONObject last, JSONObject next) throws JSONException {

        boolean insertNextOp = true;

        int parentLevel = 1;
        String nextText = next.getString(OCKey.TEXT.value());
        JSONArray nextStart = next.getJSONArray(OCKey.START.value());
        JSONArray lastStart = last.getJSONArray(OCKey.START.value());
        // additional requirement is to check if operations have target, and if they are equal
        String lastTarget = last.optString(OCKey.TARGET.value(), "");
        String nextTarget = next.optString(OCKey.TARGET.value(), "");

        if ((lastStart != null) && (nextStart != null) && (hasSameParentComponent(lastStart, nextStart, parentLevel) && lastTarget.equals(nextTarget))) {

            int lastTextPosStart = lastStart.getInt(lastStart.length() - 1);
            int nextTextPosStart = nextStart.getInt(nextStart.length() - 1);

            JSONObject nextAttrs = null;
            JSONObject lastAttrs = null;

            try {
                nextAttrs = next.getJSONObject(OCKey.ATTRS.value());
            } catch (Exception e) {
                nextAttrs = null;
            }

            try {
                lastAttrs = last.getJSONObject(OCKey.ATTRS.value());
            } catch (Exception e) {
                lastAttrs = null;
            }

            if ((nextAttrs == null) && (!((lastAttrs != null) && (lastAttrs.has(OCKey.CHANGES.value()))))) {

            	// TODO: Furter optimization: Both have 'changes' attribute and next has no other attributes

                String lastText = last.getString(OCKey.TEXT.value());

                if ((lastTextPosStart + lastText.length() == nextTextPosStart)) {
                    // the new text is inserted directly behind an already existing text
                    last.put(OCKey.TEXT.value(), lastText + nextText);
                    insertNextOp = false;
                } else if ((lastTextPosStart <= nextTextPosStart) && (nextTextPosStart <= lastTextPosStart + lastText.length())) {
                    // the new text is inserted into an already existing text
                    int relativeStart = nextTextPosStart - lastTextPosStart;
                    last.put(OCKey.TEXT.value(), lastText.substring(0, relativeStart) + nextText + lastText.substring(relativeStart));
                    insertNextOp = false;
                }
            }
        }

        return insertNextOp;
    }

    /**
     * Trying to merge an insertText operation with a directly following delete
     * operation. If the merge is possible the 'delete' operation 'next' that
     * follows the insertText operation 'last' must not be added to the operation
     * array. Therefore the return value 'insertNextOp' will be set to false in
     * the case of successful merging. In this case the content of the operation
     * 'last' will be modified.
     *
     * @param last
     *  The preceding insertText operation. This was already added to the operations
     *  array. Therefore it will be modified within this operation. And therefore it
     *  is necessary, that the insertText operations are copied before they are added
     *  to the operations array, so that the original operations are not modified.
     *
     * @param next
     *  The following delete operation. If merging this two operations is possible,
     *  then this operation will not be added to the operations array. Therefore
     *  the return value of this function is set to 'insertNextOp' false.
     *
     * @return
     *  If the 'insertText' and the 'delete' operations can be merged, it
     *  is not necessary to add the 'delete' operation to the operations array.
     *  Therefore the return value will be false. Otherwise it is true.
     *
     * @throws JSONException
     */
    static private boolean mergeInsertAndDeleteOperations(JSONObject last, JSONObject next) throws JSONException {

        boolean insertNextOp = true;
        int parentLevel = 1;
        JSONArray nextStart = next.getJSONArray(OCKey.START.value());
        JSONArray lastStart = last.getJSONArray(OCKey.START.value());
        // delete has an optional end parameter
        JSONArray nextEnd = null;
        // additional requirement is to check if operations have target, and if they are equal
        String lastTarget = last.optString(OCKey.TARGET.value(), "");
        String nextTarget = next.optString(OCKey.TARGET.value(), "");

        try {
            nextEnd = next.getJSONArray(OCKey.END.value());
        } catch (Exception e) {
            nextEnd = null;
        }

        if (nextEnd == null) {
            nextEnd = nextStart;
        }

        if ((lastStart != null) && (nextStart != null) && (hasSameParentComponent(lastStart, nextStart, parentLevel)) && (hasSameParentComponent(
            nextStart,
            nextEnd,
            parentLevel)) && (lastTarget.equals(nextTarget))) {

            int lastTextPosStart = lastStart.getInt(lastStart.length() - 1);
            int nextTextPosStart = nextStart.getInt(nextStart.length() - 1);
            int nextTextPosEnd = nextEnd.getInt(nextEnd.length() - 1);
            String lastText = last.getString(OCKey.TEXT.value());

            if ((lastTextPosStart <= nextTextPosStart) && (nextTextPosEnd < lastTextPosStart + lastText.length())) {
                // using 'nextTextPosEnd < lastTextPosStart + lastText.length()', not '<=' !
                // insertText at [0,3], Text 'abc' -> length is 3.
                // deleteText 'abc' is delete from [0,3] to [0,5], not till [0,6]!

                try {
                    // the deleted text is part of an already existing text
                    int relativeStart = nextTextPosStart - lastTextPosStart;
                    int relativeEnd = nextTextPosEnd - lastTextPosStart + 1;
                    String newText = "";
                    if ((relativeStart > 0) && (relativeStart <= lastText.length())) {
                        newText = lastText.substring(0, relativeStart);
                    }
                    if ((relativeEnd >= 0) && (relativeEnd < lastText.length())) {
                        newText += lastText.substring(relativeEnd);
                    }

                    last.put(OCKey.TEXT.value(), newText);
                    insertNextOp = false;

                } catch (IndexOutOfBoundsException e) {
                    LOG.error("RT connection: Wrong index used within mergeInsertAndDeleteOperations", e);
                }
            }
        }

        return insertNextOp;
    }

    /**
     * Optimizing the operations by trying to merge an insertText operation
     * with a directly following insertText or delete operation.
     *
     * @param jsonResult
     *  The JSONObject that contains the operations below the key "operations"
     *  as a JSONArray. The content of this JSONArray is modified. No return
     *  value is required.
     */
    static private void optimizeOperationsArray(JSONObject jsonResult) {

        if ((null != jsonResult) && (jsonResult.has(KEY_OPERATIONS))) {
            try {
                JSONArray operations = jsonResult.getJSONArray(KEY_OPERATIONS);
                JSONArray operationArr = new JSONArray();

                if (!operations.isEmpty()) {

                    int nOps = operations.length();

                    for (int i = 0; i < nOps; i++) {

                        JSONObject nextOp = operations.getJSONObject(i);
                        JSONObject next = null;

                        boolean insertNextOp = true;

                        // Optimizing the content of the JSONArray 'operationArr'.
                        // For example several insertText operations can be merged.
                        // INFO: It might be necessary to make this optimization optional,
                        // so that operations from several clients will not be merged.

                        try {

                            String nextName = nextOp.getString(OCKey.NAME.value());

                            if (nextName.equals(OCValue.INSERT_TEXT.value())) {
                                // Always make a copy of insertText operations, because they might be modified in
                                // optimization process. Taking care, that the original operations are not modified.
                                next = new JSONObject(nextOp, nextOp.keySet().toArray(new String[0]));
                                // next = new JSONObject(nextOrig.shortName()); -> alternative way
                            } else {
                                next = nextOp; // no need to copy operation, because it will not be modified
                            }

                            final JSONObject last = (operationArr.length() > 0) ? operationArr.getJSONObject(operationArr.length() - 1) : null;

                            if (last != null) {

                                String lastName = last.getString(OCKey.NAME.value());

                                if ((lastName.equals(OCValue.INSERT_TEXT.value())) && (nextName.equals(OCValue.INSERT_TEXT.value()))) {
                                    // First optimization: Merging two following insertText operations
                                    insertNextOp = mergeInsertOperations(last, next);
                                } else if ((lastName.equals(OCValue.INSERT_TEXT.value())) && (nextName.equals(OCValue.DELETE.value()))) {
                                    // Second optimization: Merging delete operation following insertText operation
                                    insertNextOp = mergeInsertAndDeleteOperations(last, next);

                                    if ((!insertNextOp) && (last.getString(OCKey.TEXT.value()).length() == 0)) {
                                        // taking care of empty new texts -> removing the last operation completely
                                        operationArr = JSONHelper.removeItemFromJSONArray(operationArr, operationArr.length() - 1);
                                    }
                                }

                                if (! insertNextOp) {
                                    if (last.has("opl") && (next.has("opl"))) {
                                        last.put("opl", last.getInt("opl") + next.getInt("opl"));
                                    }
                                }

                            }
                        } catch (JSONException e) {
                            LOG.error("RT connection: Exception caught while setting up JSON result within optimizeOperationsArray", e);
                        }

                        if (insertNextOp) {
                            operationArr.put(next);
                        }
                    }

                    jsonResult.put(KEY_OPERATIONS, operationArr);
                }
            } catch (JSONException e) {
                LOG.error("RT connection: Exception caught while setting up JSON result within optimizeOperationsArray", e);
            }
        }
    }

    /**
     * Concatenating two JSON objects containing actions
     *
     * @param jsonResult
     *
     * @param jsonAppender
     *
     * @return
     */
    public static JSONObject appendJSON(JSONObject jsonResult, JSONObject jsonAppender) {
        return appendJSON(jsonResult, jsonAppender, false);
    }

    /**
     * Concatenating two JSON objects containing actions
     *
     * @param jsonResult
     *
     * @param jsonAppender
     *
     * @param keepActions
     *
     * @return
     */
    public static JSONObject appendJSON(JSONObject jsonResult, JSONObject jsonAppender, boolean keepActions) {
        if ((null != jsonResult) && (null != jsonAppender)) {
            try {
                final Iterator<String> keys = jsonAppender.keys();

                while (keys.hasNext()) {
                    final String curKey = keys.next();
                    final Object curObject = jsonAppender.get(curKey);

                    if (curKey.equals(KEY_ACTIONS)) {
                        if (!keepActions) {
                            if (!jsonResult.has(KEY_OPERATIONS)) {
                                jsonResult.put(KEY_OPERATIONS, new JSONArray());
                            }

                            if (curObject instanceof JSONArray) {
                                final JSONArray operationsGroups = (JSONArray) curObject;

                                for (int i = 0, size = operationsGroups.length(); i < size; ++i) {
                                    final JSONObject operationsObject = operationsGroups.getJSONObject(i);
                                    if (operationsObject.has(KEY_OPERATIONS)) {
                                        final JSONArray newOperations = operationsObject.getJSONArray(KEY_OPERATIONS);

                                        for (int j = 0; j < newOperations.length(); j++) {
                                            jsonResult.append(KEY_OPERATIONS, newOperations.get(j));
                                        }
                                    } else {
                                        jsonResult.append(KEY_OPERATIONS, operationsGroups.get(i));
                                    }
                                }
                            } else {
                                jsonResult.append(KEY_OPERATIONS, curObject);
                            }
                        } else {
                            appendJSONElement(jsonResult, curKey, curObject);
                        }
                    } else {
                        appendJSONElement(jsonResult, curKey, curObject);
                    }
                }
            } catch (JSONException e) {
                LOG.error("RT connection: Exception caught while setting up JSON result within appendJSON", e);
            }
        }

        return jsonResult;
    }

    /**
     * Appends operations of a message chunk to a JSONObject which represents
     * the complete operations of the current modification session. The
     * jsonResult will contain the combinations of both operations, while
     * the function will try to remove superfluous no-op operations, which
     * were inserted for synchronization purposes.
     *
     * @param jsonResult The json object which contains the operations of the
     *                   current modification session. Can be be empty.
     * @param senderOperations generated operations without osn/opl. Can be null.
     * @param chunk      A message chunk containing operations or not. Must not be null.
     * @param serverOSN  The current server OSN.
     */
    public static void appendPackedMessageChunk(final JSONObject jsonResult, final JSONArray senderOperations, final MessageChunk chunk, int serverOSN) {
        final JSONObject chunkOpsObject = chunk.getOperations();

        try {
            final JSONArray actionsArray = chunkOpsObject.optJSONArray(MessagePropertyKey.KEY_ACTIONS);

            if (null != actionsArray && actionsArray.length() > 0) {
                int actions = actionsArray.length();

                if (!jsonResult.has(KEY_OPERATIONS)) {
                    jsonResult.put(KEY_OPERATIONS, new JSONArray());
                } else {
                    JSONArray oldOperations = jsonResult.getJSONArray(KEY_OPERATIONS);
                    if(oldOperations.length() > 0){
                        JSONObject lastOp = oldOperations.getJSONObject(oldOperations.length() - 1);
                        lastOp.remove("opl");
                        lastOp.remove("osn");
                    }
                }

                if (null != senderOperations) {
                    for( int sIdx = 0; sIdx < senderOperations.length(); ++sIdx) {
                        final JSONObject opsObject = senderOperations.getJSONObject(sIdx);
                        final String opName = opsObject.optString(OCKey.NAME.value());

                        if (!opName.equals(OCValue.NO_OP.value())) {
                            jsonResult.append(KEY_OPERATIONS, new JSONObject(opsObject));
                        }
                    }
                }

                for (int i=0; i < actions; i++) {
                    final JSONObject actionsObject = actionsArray.getJSONObject(i);
                    final JSONArray operationsArray = actionsObject.optJSONArray(KEY_OPERATIONS);

                    if ((null != operationsArray) && (operationsArray.length() > 0)) {
                        final int operations = operationsArray.length();

                        for (int j=0; j < operations; j++) {
                            final JSONObject opsObject = new JSONObject(operationsArray.getJSONObject(j));
                            final String opName = opsObject.optString(OCKey.NAME.value());
                            opsObject.remove("opl");
                            opsObject.remove("osn");

                            if (!opName.equals(OCValue.NO_OP.value())) {
                                jsonResult.append(KEY_OPERATIONS, opsObject);
                            }

                        }
                    }
                }

                JSONObject lastResultOpsObject = null;
                final JSONArray operationsArray = jsonResult.optJSONArray(KEY_OPERATIONS);
                if ((null != operationsArray) && (operationsArray.length() > 0)) {
                    lastResultOpsObject = operationsArray.getJSONObject(operationsArray.length() - 1);
                }

                // modify last operation in the combined list to reflect the correct osn
                if ((null != lastResultOpsObject)) {
                    lastResultOpsObject.put("opl", 1);
                    lastResultOpsObject.put("osn", serverOSN - 1);
                }
            }
        } catch (final Exception e) {
            LOG.warn("RT connection: appendPackedMessageChunk catches exception trying to append packed operations.", e);
        }
    }

    /**
     * Create a message chunk object filled with the provided operations and
     * ID.
     *
     * @param operations the operations to be used for the message chunk object
     * @param creatorId the creator of the operations
     * @return a message chunk object if operations could be added otherwise NULL is returned
     */
    public static MessageChunk createMessageChunkFromOperations(JSONArray operations, ID creatorId) {
        MessageChunk aMsgChunk = null;

        if (JSONHelper.isNotNullAndEmpty(operations)) {
            try {
                final JSONObject wrappedActions = new JSONObject();
                wrappedActions.put("actions", operations);

                aMsgChunk = MessageChunk.createFrom(wrappedActions, creatorId);
            } catch (JSONException e) {
                LOG.warn("RT connection: createMessageChunkFromOperations catches exception trying to create new message chunk.", e);
            }
        }

        return aMsgChunk;
    }

    /**
     * @param jsonResult
     * @param key
     * @param element
     */
    static private void appendJSONElement(JSONObject jsonResult, String key, Object element) {
        try {
            if (key.equals(KEY_OPERATIONS)) {
                if (!jsonResult.has(KEY_OPERATIONS)) {
                    jsonResult.put(KEY_OPERATIONS, new JSONArray());
                }

                if (element instanceof JSONArray) {
                    final JSONArray jsonArray = (JSONArray) element;

                    for (int i = 0; i < jsonArray.length(); ++i) {
                        jsonResult.append(KEY_OPERATIONS, jsonArray.get(i));
                    }
                } else {
                    jsonResult.append(KEY_OPERATIONS, element);
                }
            } else {
                jsonResult.put(key, element);
            }
        } catch (JSONException e) {
            LOG.error("RT connection: Exception caught while setting up JSON result within appendJSONElement", e);
        }
    }

    /**
     * @param jsonResult
     * @param operationChunks
     * @return
     */
    public static JSONObject appendOperationChunks(final JSONObject jsonResult, final List<MessageChunk> operationChunks, boolean allowOptimization) {
        if ((null != jsonResult) && (null != operationChunks)) {
            final ListIterator<MessageChunk> listIter = operationChunks.listIterator();

            while (listIter.hasNext()) {
                appendJSON(jsonResult, listIter.next().toJSON());
            }

            // Check if it's safe to optimize the operations array. There are
            // situations where this is NOT safe:
            // If we have just loaded a document and set a OSN for the last operation,
            // this operation could be exchanged by a different operation missing
            // the OSN.
            if (allowOptimization) {
                // optimize operations
                optimizeOperationsArray(jsonResult);
            }
        }

        return jsonResult;
    }

    /**
     * @param jsonResult
     * @param jsonAppender
     * @return
     */
    public static JSONObject appendData(ResourceManager resourceManager, JSONObject jsonResult, InputStream contentStm, String returnType, String mimeType) {
        final JSONObject jsonObject = (null != jsonResult) ? jsonResult : new JSONObject();

        if (null != contentStm) {
            byte[] data = null;

            try {
                data = IOUtils.toByteArray(contentStm);
            } catch (IOException e) {
                LOG.error("RT connection: Exception caught while reading content stream for appendData", e);
            }

            if ((null != data) && (data.length > 0)) {
                if (returnType.equals("resourceid")) {
                    final long uid = resourceManager.addResource(data);

                    if (0 != uid) {
                        try {
                            jsonObject.put("resourceid", Resource.getManagedIdFromUID(uid));
                        } catch (JSONException e) {
                            LOG.error("RT connection: Exception caught while creating resourceid JSON object for appendData", e);
                        }
                    }
                } else if (returnType.equals("dataurl")) {
                    final StringBuilder dataUrlBuilder = new StringBuilder("data:");

                    dataUrlBuilder.append(mimeType).append(";base64,").append(Base64.encode(data));

                    try {
                        jsonObject.put("dataurl", dataUrlBuilder.toString());
                    } catch (JSONException e) {
                        LOG.error("RT connection: Exception caught while creating dataurl JSON object for appendData", e);
                    }
                }
            }
        }

        return jsonObject;
    }

    /**
     * Transforms an operation JSONObject to a string using a special writer to ensure
     * that the output won't be bigger that a maximum number of characters.
     *
     * @param operation The json object containing the data of an operation.
     *
     * @return A string representing the operation.
     */
    public static String operationToString(JSONObject operation) {
        String result = "";

        if (null != operation) {
            JSONDebugWriter myWriter = null;

            try {
                myWriter = new JSONDebugWriter();
                operation.write(myWriter);
                result = myWriter.getData();
            } catch (final Exception e) {
                LOG.warn("RT connection: operationshortName catches exception writing operation to string. ", e);
            } finally {
                IOUtils.closeQuietly(myWriter);
            }
        }

        return result;
    }

    /**
     * Provides the json data of the first operation of the first chunk within
     * the provided MessageChunk list.
     *
     * @param chunkList The message chunk list to retrieve the first operation of the first chunk.
     *
     * @return The json data of the first operation of the first chunk or null if no chunk/operation is
     * available.
     */
    public static JSONObject debugGetFirstOperationFromMessageChunkList(final List<MessageChunk> chunkList) {
        JSONObject result = null;

        if (chunkList.size() > 0) {
            MessageChunk chunk = chunkList.get(0);
            JSONObject opsObject = chunk.getOperations();
            try {
                JSONArray actions = opsObject.getJSONArray(MessagePropertyKey.KEY_ACTIONS);
                if (actions.length() > 0) {
                    opsObject = actions.getJSONObject(0);
                    int operationCount = opsObject.getJSONArray("operations").length();
                    if (operationCount > 0) {
                        result = opsObject.getJSONArray("operations").getJSONObject(0);
                    }
                }
            } catch (final JSONException e) {
                LOG.warn("RT connection: debugGetFirstOperationFromMessageChunkList catches exception trying to retrieve operations from chunkList. ", e);
            } catch (final Throwable e) {
                LOG.warn("RT connection: debugGetFirstOperationFromMessageChunkList catches exception trying to retrieve operations from chunkList. ", e);
            }
        }

        return result;
    }

    /**
     * Provides the json data of the last operation of the last chunk within
     * the provided MessageChunk list.
     *
     * @param chunkList The message chunk list to retrieve the last operation of the last chunk.
     *
     * @return The json data of the last operation of the last chunk or null if no chunk/operation is
     * available.
     */
    public static JSONObject debugGetLastOperationFromMessageChunkList(final List<MessageChunk> chunkList) {
        JSONObject result = null;

        if (chunkList.size() > 0) {
            MessageChunk chunk = chunkList.get(chunkList.size() - 1);
            JSONObject opsObject = chunk.getOperations();
            try {
                JSONArray actions = opsObject.getJSONArray(MessagePropertyKey.KEY_ACTIONS);
                if (actions.length() > 0) {
                    opsObject = actions.getJSONObject(actions.length() - 1);
                    int operationCount = opsObject.getJSONArray("operations").length();
                    if (operationCount > 0) {
                        result = opsObject.getJSONArray("operations").getJSONObject(operationCount - 1);
                    }
                }
            } catch (final JSONException e) {
                LOG.warn("RT connection: debugGetLastOperationFromMessageChunkList catches exception trying to retrieve operations from chunkList. ", e);
            } catch (final Throwable e) {
                LOG.warn("RT connection: debugGetLastOperationFromMessageChunkList catches exception trying to retrieve operations from chunkList. ", e);
            }
        }

        return result;
    }

    /**
     * Checks that the operations of a message chunk comply to
     * our minimal requirements:
     * 1. always ascending
     * 2. without any gap in osn/opl by consecutive operations
     *
     * @param chunk
     *  The MessageChunk which should be checked.
     *
     * @return
     *  The error code of the check. ErrorCode.NO_ERROR, if nothing
     *  is wrong.
     */
    public static ErrorCode hasValidOperations(MessageChunk chunk, int serverOSN) {
        ErrorCode errorCode = ErrorCode.NO_ERROR;
        JSONObject chunkOpsObject = chunk.getOperations();

        try {
            JSONArray actionsArray = chunkOpsObject.optJSONArray(MessagePropertyKey.KEY_ACTIONS);
            if (null != actionsArray && actionsArray.length() > 0) {
                int actions = actionsArray.length();

                for (int i=0; i < actions; i++) {
                    JSONObject actionsObject = actionsArray.getJSONObject(i);
                    JSONArray operationsArray = actionsObject.optJSONArray(MessagePropertyKey.KEY_OPERATIONS);

                    if ((null != operationsArray) && (operationsArray.length() > 0)) {
                        int operations = operationsArray.length();
                        for (int j=0; j < operations; j++) {
                            JSONObject opsObject = operationsArray.getJSONObject(j);
                            int opsOPL = opsObject.optInt("opl");
                            int opsOSN = opsObject.optInt("osn");

                            if (opsOSN != serverOSN) {
                                // found inconsistent osn in chunk
                                return ErrorCode.HANGUP_INVALID_OSN_DETECTED_ERROR;
                            }
                            serverOSN = opsOSN + opsOPL;
                        }
                    }
                }
            }
        } catch (final Exception e) {
            LOG.warn("RT connection: hasValidOperations catches exception trying to retrieve operations from chunkList. ", e);
        }

        return errorCode;
    }

    /**
     * Creates a List which contains the operations JSONObjects. Be careful: This is a
     * shallow copy of the operations.
     *
     * @param chunk
     * @return
     */
    static public boolean appendOperationsFromMessageChunk(final JSONArray target, final MessageChunk chunk) {
        Validate.notNull(target);

        boolean result = false;
        final JSONObject chunkOpsObject = chunk.getOperations();

        try {
            final JSONArray actionsArray = chunkOpsObject.optJSONArray(MessagePropertyKey.KEY_ACTIONS);
            if (null != actionsArray && actionsArray.length() > 0) {
                int actions = actionsArray.length();

                for (int i=0; i < actions; i++) {
                    final JSONObject actionsObject = actionsArray.getJSONObject(i);
                    final JSONArray operationsArray = actionsObject.optJSONArray(MessagePropertyKey.KEY_OPERATIONS);

                    // read the operationsArray and copy the JSONObject reference
                    if ((null != operationsArray) && (operationsArray.length() > 0)) {
                        JSONHelper.appendArray(target, operationsArray);
                    }
                }
            }

            result = true;
        } catch (final Exception e) {
            LOG.warn("RT connection: getOperationsFromMessageChunk catches exception trying to retrieve operations from chunkList.", e);
        }

        return result;
    }

    /**
     * Creates a synchronization no-operation to be added to a list of
     * operations.
     *
     * @param osn
     *  The OSN to be used by the no-op.
     *
     * @return
     *  A JSONObject filled with the necessary properties.
     */
    public static JSONObject createSyncNoOperation(int osn) throws JSONException {
        final JSONObject noOpSync = new JSONObject();
        noOpSync.put(OCKey.NAME.value(), OCValue.NO_OP.value());
        noOpSync.put("osn", (osn - 1));
        noOpSync.put("opl", 1);
        return noOpSync;
    }

    /**
     * Saving performance data into log file on server.
     *  with a directly following insertText or delete operation.
     *
     * @param performanceData
     *  The JSONObject containing all performance information sent
     *  from the client.
     *
     * @param writer
     *  The buffered writer for content input.
     */
    public static void logEditorPerformanceData(JSONObject performanceData, BufferedWriter writer) {

    	String uid = UUID.randomUUID().toString().substring(0, 8);  // only using 8 digits

    	// Saving performance data in log file
    	try {
        	writer.append(uid + ":UTC:" + getISODateFormat(true));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":LAUNCHSTART:" + performanceData.optInt("launchStartDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":APPCONSTRUCTED:" + performanceData.optInt("appConstructedDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":GUIPREPARED:" + performanceData.optInt("guiPreparedDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":IMPORTHANDLERSTART:" + performanceData.optInt("importHandlerStartDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":CREATENEWDOCSTART:" + performanceData.optInt("createNewDocStartDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":CREATENEWDOCAFTERREQUEST:" + performanceData.optInt("createNewDocAfterRequestDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":PREVIEWSTART:" + performanceData.optInt("previewStartDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":PREVIEWEND:" + performanceData.optInt("previewEndDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":AFTERLEAVEBUSYEARLY:" + performanceData.optInt("afterLeaveBusyEarlyDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":IMPORTDOCSTART:" + performanceData.optInt("importDocStartDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":CONNECTSTART:" + performanceData.optInt("connectStartDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":CONNECTEND:" + performanceData.optInt("connectEndDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":PREPROCESSINGSTART:" + performanceData.optInt("preProcessingStartDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":PREPROCESSINGEND:" + performanceData.optInt("preProcessingEndDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":APPLYACTIONSSTART:" + performanceData.optInt("applyActionsStartDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":APPLYACTIONSEND:" + performanceData.optInt("applyActionsEndDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":POSTPROCESSINGSTART:" + performanceData.optInt("postProcessingStartDuration", 0));
            writer.append(System.getProperty("line.separator"));
            writer.append(uid + ":DOCUMENTVISIBLEDURATION:" + performanceData.optInt("documentVisibleDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":POSTPROCESSINGEND:" + performanceData.optInt("postProcessingEndDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":EDITMODE:" + performanceData.optInt("editModeDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":DOCUMENTIMPORTED:" + performanceData.optInt("documentImportedDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":LEAVEBUSY:" + performanceData.optInt("leaveBusyDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":GUIINITIALIZED:" + performanceData.optInt("guiInitializedDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":TRIGGERIMPORTSUCCESS:" + performanceData.optInt("triggerImportSuccessDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":LAUNCHEND:" + performanceData.optInt("launchEndDuration", 0));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":LOCALSTORAGE:" + performanceData.optBoolean("LocalStorage", false));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":FASTEMPTYLOAD:" + performanceData.optBoolean("FastEmptyLoad", false));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":USERAGENT:" + performanceData.optString("user-agent", ""));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":PLATFORM:" + performanceData.optString("platform", ""));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":USER:" + performanceData.optString("user", ""));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":OXVERSION:" + performanceData.optString("version", ""));
            writer.append(System.getProperty("line.separator"));
        	writer.append(uid + ":SERVERHOSTNAME:" + performanceData.optString("server", ""));
            writer.append(System.getProperty("line.separator"));
            writer.append(uid + ":APPLICATION:" + performanceData.optString("application", ""));
            writer.append(System.getProperty("line.separator"));
            writer.append(uid + ":FILENAME:" + performanceData.optString("filename", ""));

            writer.append(System.getProperty("line.separator"));
            writer.append(System.getProperty("line.separator"));
            writer.close();
		} catch (Exception e) {
            LOG.debug("LogEditorPerformanceData writing to log file failed due to exception", e);
			// Do nothing, simply not logging performance data
	    }

    }

    /**
     * Saving performance data into log file on server.
     *  with a directly following insertText or delete operation.
     *
     * @param performanceData
     *  The JSONObject containing all performance information sent
     *  from the client.
     *
     * @param writer
     *  The buffered writer for content input.
     */
    public static void logViewerPerformanceData(JSONObject performanceData, BufferedWriter writer) {

        String uid = UUID.randomUUID().toString().substring(0, 8);  // only using 8 digits
        int counter = 1;

        String beforePageZoom = "DocumentView:refresh_before_set_pagezoom_";
        String beforeGetPdfJsPage = "pdfView:renderPDFPage_before_getPDFJSPage_";
        String thenOfGetPdfJsPage = "pdfView:renderPDFPage_getPDFJSPage_then_handler_";
        String beforeRenderPdfJsPage = "pdfView:renderPDFPage_before_pdfjsPage_render_";
        String thenOfRenderPdfJsPage = "pdfView:renderPDFPage_pdfjsPage_render_then_handler_";
        String thenOfPageZoom = "DocumentView:refresh_then_from_set_pagezoom_";

        String beforePageZoomValue = beforePageZoom + counter;
        String beforeGetPdfJsPageValue = beforeGetPdfJsPage + counter;
        String thenOfGetPdfJsPageValue = thenOfGetPdfJsPage + counter;
        String beforeRenderPdfJsPageValue = beforeRenderPdfJsPage + counter;
        String thenOfRenderPdfJsPageValue = thenOfRenderPdfJsPage + counter;
        String thenOfPageZoomValue = thenOfPageZoom + counter;

        // Saving performance data in log file
        try {
            writer.append(uid + ":UTC:" + getISODateFormat(true));
            writer.append(System.getProperty("line.separator"));
            writer.append(uid + ":LAUNCHCONTSTART:" + performanceData.optInt("launchContStart", 0));
            writer.append(System.getProperty("line.separator"));
            writer.append(uid + ":MAINVIEW:INITIALIZE:" + performanceData.optInt("MainView:initialize", 0));
            writer.append(System.getProperty("line.separator"));
            writer.append(uid + ":DOCUMENTVIEW:INITIALIZE:" + performanceData.optInt("DocumentView:initialize", 0));
            writer.append(System.getProperty("line.separator"));
            writer.append(uid + ":DOCUMENTVIEW:SHOW:" + performanceData.optInt("DocumentView:show", 0));
            writer.append(System.getProperty("line.separator"));
            writer.append(uid + ":PDFDOCUMENT:PDFJS_GETDOCUMENT_THEN_HANDLER:" + performanceData.optInt("PDFDocument:PDFJs_getDocument_then_handler", 0));
            writer.append(System.getProperty("line.separator"));
            writer.append(uid + ":DOCUMENTVIEW:SHOW_GETLOADPROMISE_DONE_HANDLER:" + performanceData.optInt("DocumentView:show_getLoadPromise_done_handler", 0));
            writer.append(System.getProperty("line.separator"));
            writer.append(uid + ":DOCUMENTVIEW:LOADVISIBLEPAGES:" + performanceData.optInt("DocumentView:loadVisiblePages", 0));
            writer.append(System.getProperty("line.separator"));

            while (performanceData.optInt(beforePageZoomValue, 0) != 0) {
                writer.append(uid + ":" + beforePageZoom.toUpperCase() + counter + ":" + performanceData.optInt(beforePageZoomValue, 0));
                writer.append(System.getProperty("line.separator"));
                beforePageZoomValue = beforePageZoom + ++counter;
            }

            counter = 1;

            while (performanceData.optInt(beforeGetPdfJsPageValue, 0) != 0) {
                writer.append(uid + ":" + beforeGetPdfJsPage.toUpperCase() + counter + ":" + performanceData.optInt(beforeGetPdfJsPageValue, 0));
                writer.append(System.getProperty("line.separator"));
                beforeGetPdfJsPageValue = beforeGetPdfJsPage + ++counter;
            }

            counter = 1;

            while (performanceData.optInt(thenOfGetPdfJsPageValue, 0) != 0) {
                writer.append(uid + ":" + thenOfGetPdfJsPage.toUpperCase() + counter + ":" + performanceData.optInt(thenOfGetPdfJsPageValue, 0));
                writer.append(System.getProperty("line.separator"));
                thenOfGetPdfJsPageValue = thenOfGetPdfJsPage + ++counter;
            }

            counter = 1;

            while (performanceData.optInt(beforeRenderPdfJsPageValue, 0) != 0) {
                writer.append(uid + ":" + beforeRenderPdfJsPage.toUpperCase() + counter + ":" + performanceData.optInt(beforeRenderPdfJsPageValue, 0));
                writer.append(System.getProperty("line.separator"));
                beforeRenderPdfJsPageValue = beforeRenderPdfJsPage + ++counter;
            }

            counter = 1;

            while (performanceData.optInt(thenOfRenderPdfJsPageValue, 0) != 0) {
                writer.append(uid + ":" + thenOfRenderPdfJsPage.toUpperCase() + counter + ":" + performanceData.optInt(thenOfRenderPdfJsPageValue, 0));
                writer.append(System.getProperty("line.separator"));
                thenOfRenderPdfJsPageValue = thenOfRenderPdfJsPage + ++counter;
            }

            counter = 1;

            while (performanceData.optInt(thenOfPageZoomValue, 0) != 0) {
                writer.append(uid + ":" + thenOfPageZoom.toUpperCase() + counter + ":" + performanceData.optInt(thenOfPageZoomValue, 0));
                writer.append(System.getProperty("line.separator"));
                thenOfPageZoomValue = thenOfPageZoom + ++counter;
            }

            writer.append(uid + ":USERAGENT:" + performanceData.optString("user-agent", ""));
            writer.append(System.getProperty("line.separator"));
            writer.append(uid + ":PLATFORM:" + performanceData.optString("platform", ""));
            writer.append(System.getProperty("line.separator"));
            writer.append(uid + ":USER:" + performanceData.optString("user", ""));
            writer.append(System.getProperty("line.separator"));
            writer.append(uid + ":OXVERSION:" + performanceData.optString("version", ""));
            writer.append(System.getProperty("line.separator"));
            writer.append(uid + ":SERVERHOSTNAME:" + performanceData.optString("server", ""));
            writer.append(System.getProperty("line.separator"));
            writer.append(uid + ":FILENAME:" + performanceData.optString("filename", ""));
            writer.append(System.getProperty("line.separator"));
            writer.append(uid + ":APPLICATION:" + performanceData.optString("application", ""));

            writer.append(System.getProperty("line.separator"));
            writer.append(System.getProperty("line.separator"));
            writer.close();

        } catch (Exception e) {
            LOG.debug("LogViewerPerformanceData writing to log file failed due to exception", e);
            // Do nothing, simply not logging performance data
        }

    }

    /**
     * The current date in ISO 8601 format.
     *
     * @param useSeconds
     *  Wheter seconds shall be included into the returnes string or not.
     *
     * @return
     *  The date as string in ISO 8601 format.
     */
    public static String getISODateFormat(Boolean useSeconds) {

    	TimeZone tz = TimeZoneHelper.getTimeZone("UTC");
    	DateFormat df = null;

    	if (useSeconds) {
    	    df = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss'Z'");
    	} else {
    	    df = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm':00Z'");
    	}
    	df.setTimeZone(tz);
    	return df.format(new Date());
    }

    public static int getBaseOSN(final JSONObject operationsObject) {
        int result = -1;

        try {
            int operationCount = operationsObject.getJSONArray(KEY_OPERATIONS).length();
            if (operationCount > 0) {
                result = operationsObject.getJSONArray(KEY_OPERATIONS).getJSONObject(0).getInt(KEY_OSN);
            }
        } catch (final JSONException e) {
            // nothing to do
        } catch (final Exception e) {
            // nothing to do
        }

        return result;
    }

    public static int getFollowUpOSN(final JSONObject operationsObject) {
        int result = -1;

        try {
             result = getFollowUpOSN(operationsObject.getJSONArray(KEY_OPERATIONS));
        } catch (JSONException e) {
             // nothng to do
        }

        return result;
    }

    public static int getFollowUpOSN(final JSONArray operationsArray) {
        int result = -1;

        try {
            int operationCount = operationsArray.length();
            if (operationCount > 0) {
                int osn = operationsArray.getJSONObject(operationCount-1).getInt(KEY_OSN);
                int opl = operationsArray.getJSONObject(operationCount-1).getInt(KEY_OPL);
                result = osn + opl;
            }
        } catch (final JSONException e) {
            // nothing to do
        } catch (final Exception e) {
            // nothing to do
        }

        return result;
    }

    /**
     * Retrieves the last operation from a JSONObject containing
     * a set of operations.
     *
     * @param operationsObject a JSONObject containing operations provided by
     *        a client.
     * @return the last operation JSONObject or null, if not found
     */
    public static JSONObject getLastOperation(final JSONObject operationsObject) {
        JSONObject result = null;

        try {
            final JSONArray operationsArray = operationsObject.getJSONArray(KEY_OPERATIONS);
            int operationCount = operationsArray.length();
            if (operationCount > 0) {
                result = operationsArray.getJSONObject(operationCount-1);
            }
        } catch (final JSONException e) {
            // nothing to do
        } catch (final Exception e) {
            // nothing to do
        }

        return result;
    }

    // - Members ---------------------------------------------------------------
    static protected final org.apache.commons.logging.Log LOG = com.openexchange.log.LogFactory.getLog(OperationHelper.class);
}
