/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.filter.ooxml.components;

import java.lang.annotation.Annotation;
import java.text.ParseException;
import java.util.HashSet;
import java.util.Set;
import javax.xml.bind.JAXBElement;
import javax.xml.bind.JAXBException;
import javax.xml.bind.annotation.XmlRootElement;
import org.docx4j.openpackaging.exceptions.InvalidFormatException;
import org.docx4j.openpackaging.exceptions.PartUnrecognisedException;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.office.filter.api.FilterException;
import com.openexchange.office.filter.core.DLNode;
import com.openexchange.office.filter.core.IContentAccessor;
import com.openexchange.office.filter.core.DLList;
import com.openexchange.office.filter.core.SplitMode;
import com.openexchange.office.filter.core.component.Child;
import com.openexchange.office.filter.core.component.ComponentContext;
import com.openexchange.office.filter.core.component.IComponent;
import com.openexchange.office.filter.ooxml.OperationDocument;

public abstract class Component extends ComponentContext<OperationDocument, Component> implements IComponent<ComponentContext<OperationDocument, Component>, Component> {

    private static HashSet<Class<?>> rootElements = new HashSet<Class<?>>();

    public enum Type {
    	PARAGRAPH,
    	TABLE,
    	TR,
    	TC,
    	TAB,
    	BOOKMARK_START,
    	BOOKMARK_END,
    	COMMENT_REFERENCE,
    	COMMENT_RANGE_START,
    	COMMENT_RANGE_END,
    	SIMPLE_FIELD,
    	COMPLEX_FIELD_START,
    	COMPLEX_FIELD_SEPARATE,
    	COMPLEX_FIELD_END,
    	HARDBREAK_DEFAULT,
    	HARDBREAK_PAGE,
    	HARDBREAK_COLUMN,
    	AC_SHAPE,
    	AC_GROUP,
    	AC_CHART,
    	AC_CONNECTOR,
    	AC_IMAGE,
    	AC_COMMENT
    }

    /*
     * simply returns o, but in case o is a JAXBElement and a RootElement annotation is available
     * then the parent content list is updated and the contentModel is returned.
     * */
    public static Object getContentModel(DLNode<Object> node, Object parent) {

        if (node.getData() instanceof javax.xml.bind.JAXBElement) {
            boolean hasRootElement = rootElements.contains(((javax.xml.bind.JAXBElement<?>)node.getData()).getDeclaredType());
            if(!hasRootElement) {
                final Annotation[] annotations = ((javax.xml.bind.JAXBElement<?>)node.getData()).getDeclaredType().getAnnotations();
                for(Annotation annotation : annotations) {
                    if(annotation instanceof XmlRootElement) {
                        rootElements.add(((javax.xml.bind.JAXBElement<?>)node.getData()).getDeclaredType());
                        hasRootElement = true;
                        break;
                    }
                }
            }
            if(hasRootElement) {
                node.setData(((JAXBElement<?>)node.getData()).getValue());
            }
        }
        if(node.getData() instanceof Child) {
        	((Child)node.getData()).setParent(parent);
        }
        return node.getData();
    }

    protected int componentNumber;

    public Component(OperationDocument operationDocument, DLNode<Object> _node, int _componentNumber) {
        super(operationDocument, _node);
        componentNumber = _componentNumber;
    }

    public Component(ComponentContext<OperationDocument, Component> parentContext, DLNode<Object> _node, int _componentNumber) {
        super(parentContext, _node);
        componentNumber = _componentNumber;
    }

    @Override
    public Component getNextComponent() {
    	if(getParentContext()==null) {
    		return null;
    	}
        ComponentContext<OperationDocument, Component> parentContext = this;
        Component nextComponent = null;
        do {
            ComponentContext<OperationDocument, Component> previousContext = parentContext;
            parentContext = parentContext.getParentContext();
            nextComponent = parentContext.getNextChildComponent(previousContext, this);
        }
        while(nextComponent==null&&!(parentContext instanceof Component));
        return nextComponent;
    }

    public ComponentContext<OperationDocument, Component> getContext(Class<?> contextClass) {
        ComponentContext<OperationDocument, Component> parentContext = getParentContext();
    	while(!(parentContext instanceof Component)&&(parentContext!=null)) {
    		if(parentContext.getClass()==contextClass) {
    			return parentContext;
    		}
    		parentContext = parentContext.getParentContext();
    	}
    	return null;
    }

    // the ComponentContext c is inserted behind one of the classes in the
    // parentContextList, it is inserted at least behind the parent Component,
    // so the parentContextList is allowed to be null
    //
    // the parentContext of c is inserted, so c does not need to have a parentContext
    public void insertContext(ComponentContext<OperationDocument, Component> c, Set<Class<?>> parentContextList) {
        ComponentContext<OperationDocument, Component> childContext = this;
        ComponentContext<OperationDocument, Component> parentContext = getParentContext();
        while(!(parentContext instanceof Component)&&((parentContextList==null)||(!parentContextList.contains(parentContext.getClass())))) {
            childContext = parentContext;
            parentContext = childContext.getParentContext();
        }

        final DLList<Object> parentContent = (DLList<Object>)((IContentAccessor)parentContext.getNode().getData()).getContent();
        parentContent.setNode(childContext.getNode(), c.getNode());
        final DLList<Object> content = (DLList<Object>)((IContentAccessor)c.getNode().getData()).getContent();
        content.addNode(childContext.getNode());
        c.setParentContext(parentContext);
        childContext.setParentContext(c);
        ((Child)c.getNode().getData()).setParent(parentContext.getNode().getData());
        ((Child)childContext.getNode().getData()).setParent(c.getNode().getData());
    }

    public void removeContext(Class<?> contextClass) {
        ComponentContext<OperationDocument, Component> childContext = this;
        ComponentContext<OperationDocument, Component> parentContext = getParentContext();
        while(!(parentContext instanceof Component)) {
            if(parentContext.getClass()==contextClass) {
            	final ComponentContext<OperationDocument, Component> newParentContext = parentContext.getParentContext();
            	((DLList<Object>)((IContentAccessor)newParentContext.getNode().getData()).getContent()).setNodes(parentContext.getNode(), (DLList<Object>)((IContentAccessor)parentContext.getNode().getData()).getContent());
            	childContext.setParentContext(newParentContext);
            	((Child)childContext.getNode().getData()).setParent(newParentContext.getNode().getData());
                break;
            }
            childContext = parentContext;
            parentContext = childContext.getParentContext();
        }
    }

    public ComponentContext<OperationDocument, Component> getContextChild(Set<Class<?>> parentContextList) {
        ComponentContext<OperationDocument, Component> childContext = this;
        ComponentContext<OperationDocument, Component> parentContext = getParentContext();
        while(!(parentContext instanceof Component)&&((parentContextList==null)||(!parentContextList.contains(parentContext.getClass())))) {
            childContext = parentContext;
            parentContext = childContext.getParentContext();
        }
        return childContext;
    }

    @Override
    public Component getParentComponent() {
    	ComponentContext<OperationDocument, Component> parentContext = getParentContext();
    	while(!(parentContext instanceof Component)) {
    		parentContext = parentContext.getParentContext();
    	}
    	return (Component)parentContext;
    }

    @Override
    public abstract Component getNextChildComponent(ComponentContext<OperationDocument, Component> previousChildContext, Component previousChildComponent);

    @Override
    public Component getChildComponent(int childComponentNumber) {
    	Component c = getNextChildComponent(null, null);
    	if(c!=null) {
    		c = c.getComponent(childComponentNumber);
    	}
    	return c;
    }

    @Override
    public Component getComponent(JSONArray position, int positionCount) {
        try {
            if(positionCount==0) {
                return this;
            }
            // a part has to implement the ContentAccessor interface, otherwise no component can be accessed
            Component c = getNextChildComponent(null, null);
            for(int i=0; c!=null&&i<positionCount;) {
                c = c.getComponent(position.getInt(i++));
                if(i!=positionCount) {
                    c = c.getNextChildComponent(null, null);
                }
            }
            return c;
        }
        catch(JSONException e) {
            // ups
        }
        return null;
    }

    public abstract Component insertChildComponent(ComponentContext<OperationDocument, Component> parentContext, DLNode<Object> contextNode, int number, Component child, Type type)
    	throws JAXBException;

    public Component insertChildComponent(int number, JSONObject attrs, Type type)
    	throws UnsupportedOperationException, JAXBException, InvalidFormatException, PartUnrecognisedException, JSONException {

        final Component c = insertChildComponent(this, getNode(), number, getChildComponent(number), type);
        if(attrs!=null) {
            c.applyAttrsFromJSON(attrs);
        }
        return c;
    }

    @Override
    public Component getComponent(int number) {
    	Component c = this;
    	while(c!=null&&c.getNextComponentNumber()<=number) {
    		c = c.getNextComponent();
    	}
    	return c;
    }

    @Override
    public int getComponentNumber() {
        return componentNumber;
    }

    @Override
    public int getNextComponentNumber() {
        return componentNumber + 1;
    }

    public abstract void applyAttrsFromJSON(JSONObject attrs)
        throws JAXBException, JSONException, InvalidFormatException, PartUnrecognisedException;

    public abstract JSONObject createJSONAttrs(JSONObject attrs)
    	throws JSONException, ParseException, FilterException;

    public static void move(OperationDocument operationDocument, JSONArray startPosition, JSONArray endPosition, JSONArray toPosition)
        throws JSONException {

    	if(endPosition==null) {
    		endPosition = startPosition;
    	}
    	else if(startPosition.length()!=endPosition.length())
            throw new JSONException("ooxml export: move operation, size of startPosition != size of endPosition");

        final Component sourceBegComponent = operationDocument.getRootComponent().getComponent(startPosition, startPosition.length());
        final Component sourceEndComponent = sourceBegComponent.getComponent(endPosition.getInt(startPosition.length()-1));
        final DLList<Object> sourceContent = (DLList<Object>)((IContentAccessor)sourceBegComponent.getParentComponent().getNode().getData()).getContent();
        sourceBegComponent.splitStart(startPosition.getInt(startPosition.length()-1), SplitMode.DELETE);
        sourceEndComponent.splitEnd(endPosition.getInt(startPosition.length()-1), SplitMode.DELETE);

        // destComponent is empty if the content is to be appended
        final Component destComponent = operationDocument.getRootComponent().getComponent(toPosition, toPosition.length());
        Component destParentComponent;
        if(destComponent!=null) {
        	destComponent.splitStart(toPosition.getInt(toPosition.length()-1), SplitMode.DELETE);
        	destParentComponent = destComponent.getParentComponent();
        }
        else {
        	destParentComponent = operationDocument.getRootComponent().getComponent(toPosition, toPosition.length()-1);
        }
        final DLList<Object> destContent = (DLList<Object>)((IContentAccessor)destParentComponent.getNode().getData()).getContent();
        final boolean before = sourceContent==destContent && startPosition.getInt(startPosition.length()-1) <= toPosition.getInt(toPosition.length()-1) ? false : true;
        if(destComponent!=null) {
            sourceContent.moveNodes(sourceBegComponent.getContextChild(null).getNode(), sourceEndComponent.getContextChild(null).getNode(), destContent,
                destComponent.getContextChild(null).getNode(), before, destParentComponent.getNode().getData());
        }
        else {
            sourceContent.moveNodes(sourceBegComponent.getContextChild(null).getNode(), sourceEndComponent.getContextChild(null).getNode(), destContent,
                destContent.getLastNode(), false, destParentComponent.getNode().getData());
        }
    }
}
