/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/**
 *
 * @author sven.jacobi@open-xchange.com
 */

package com.openexchange.office.filter.ods.dom.components;

import org.json.JSONException;
import org.json.JSONObject;
import org.xml.sax.SAXException;
import com.openexchange.office.filter.core.DLList;
import com.openexchange.office.filter.core.DLNode;
import com.openexchange.office.filter.core.component.ComponentContext;
import com.openexchange.office.filter.odf.OdfOperationDoc;
import com.openexchange.office.filter.odf.OpAttrs;
import com.openexchange.office.filter.odf.components.Component;
import com.openexchange.office.filter.odf.draw.ConnectorShape;
import com.openexchange.office.filter.odf.draw.CustomShape;
import com.openexchange.office.filter.odf.draw.DrawFrame;
import com.openexchange.office.filter.odf.draw.DrawTextBox;
import com.openexchange.office.filter.odf.draw.DrawingType;
import com.openexchange.office.filter.odf.draw.GroupShape;
import com.openexchange.office.filter.odf.draw.IDrawingType;
import com.openexchange.office.filter.odf.draw.Shape;
import com.openexchange.office.filter.ods.dom.Drawing;
import com.openexchange.office.filter.odt.dom.Paragraph;

public class DrawingComponent extends Component implements IDrawingType {

	final Drawing drawing;

	public DrawingComponent(ComponentContext<OdfOperationDoc, Component> parentContext, DLNode<Object> shapeNode, int componentNumber) {
		super(parentContext, shapeNode, componentNumber);
		drawing = (Drawing)getObject();
	}

    @Override
    public String simpleName() {
        return "Drawing";
    }

    public Shape getShape() {
		return drawing.getShape();
	}

	@Override
    public DrawingType getType() {
		return drawing.getType();
	}

	@Override
	public Component getNextChildComponent(ComponentContext<OdfOperationDoc, Component> previousChildContext, Component previousChildComponent) {
        final int nextComponentNumber = previousChildComponent != null ? previousChildComponent.getNextComponentNumber() : 0;
        DLNode<Object> nextNode = previousChildContext!=null ? previousChildContext.getNode().getNext() : drawing.getShape().getContent().getFirstNode();
        while(nextNode!=null) {
            final Object child = nextNode.getData();
            if(child instanceof GroupShape) {
                return new ShapeGroupComponent(this, nextNode, nextComponentNumber);
            }
            else if(child instanceof DrawFrame) {
                return new FrameComponent(this, nextNode, nextComponentNumber);
            }
            else if(child instanceof ConnectorShape) {
                return new ShapeConnectorComponent(this, nextNode, nextComponentNumber);
            }
            else if(child instanceof Shape) {
                return new ShapeComponent(this, nextNode, nextComponentNumber);
            }
            else if(child instanceof Paragraph) {
                return new ParagraphComponent(this, nextNode, nextComponentNumber);
            }
            nextNode = nextNode.getNext();
        }
        return null;
	}

    @Override
    public Component insertChildComponent(ComponentContext<OdfOperationDoc, Component> parentContext, DLNode<Object> contextNode, int number, Component child, Type type, JSONObject attrs) {
    	DLList<Object> DLList = drawing.getShape().getContent();
        DLNode<Object> referenceNode = child != null && child.getComponentNumber()== number ? child.getNode() : null;

        switch(type) {
            case AC_SHAPE: {
                final DLNode<Object> newShapeNode = new DLNode<Object>(Shape.createShape(operationDocument, attrs, drawing.getShape() instanceof GroupShape ? (GroupShape)drawing.getShape() : null, false, isContentAutoStyle()));
                DLList.addNode(referenceNode, newShapeNode, true);
                return new ShapeComponent(parentContext, newShapeNode, number);
            }
            case AC_CONNECTOR: {
                final DLNode<Object> newShapeNode = new DLNode<Object>(new CustomShape(operationDocument, drawing.getShape() instanceof GroupShape ? (GroupShape)drawing.getShape() : null, false, isContentAutoStyle()));
                DLList.addNode(referenceNode, newShapeNode, true);
                return new ShapeComponent(parentContext, newShapeNode, number);
            }
            case AC_GROUP: {
                final DLNode<Object> newGroupNode = new DLNode<Object>(new GroupShape(operationDocument, drawing.getShape() instanceof GroupShape ? (GroupShape)drawing.getShape() : null, false, isContentAutoStyle()));
                DLList.addNode(referenceNode, newGroupNode, true);
                return new ShapeGroupComponent(parentContext, newGroupNode, number);
            }
            case AC_FRAME: {
                final DrawFrame drawFrame = new DrawFrame(operationDocument, drawing.getShape() instanceof GroupShape ? (GroupShape)drawing.getShape() : null, false, isContentAutoStyle());
                final DLNode<Object> drawFrameNode = new DLNode<Object>(drawFrame);
                DLList.addNode(referenceNode, new DLNode<Object>(drawFrame), true);
                final DrawTextBox drawTextBox = new DrawTextBox(drawFrame);
                drawFrame.addContent(drawTextBox);
                return new FrameComponent(parentContext, drawFrameNode, number);
            }
            case PARAGRAPH : {
                // if this is a normal shape, then paragraphs can be inserted
                final DLNode<Object> newParagraphNode = new DLNode<Object>(new Paragraph(null));
                DLList.addNode(referenceNode, newParagraphNode, true);
                return new ParagraphComponent(parentContext, newParagraphNode, number);
            }
            case AC_CHART:
                DrawFrame drawFrame = getChart(operationDocument, isContentAutoStyle());

                final DLNode<Object> drawFrameNode = new DLNode<Object>(drawFrame);
                DLList.addNode(referenceNode, new DLNode<Object>(drawFrame), true);
                return new FrameComponent(parentContext, drawFrameNode, number);

            default : {
                throw new UnsupportedOperationException();
            }
        }
    }

    @Override
    public Component insertChildComponent(int number, JSONObject attrs, Type type)
    	throws UnsupportedOperationException, JSONException, SAXException {

        final Component c = insertChildComponent(this, getNode(), number, getChildComponent(number), type, attrs);
        if(attrs!=null) {
            c.applyAttrsFromJSON(attrs);
        }
        return c;
    }

	@Override
	public void applyAttrsFromJSON(JSONObject attrs)
			throws JSONException {

	    drawing.applyAttrsFromJSON(operationDocument, attrs, contentAutoStyle);
	}

	@Override
	public void createJSONAttrs(OpAttrs attrs) {

	    drawing.createAttrs(operationDocument, attrs, contentAutoStyle);
	}
}
