/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/**
 *
 * @author sven.jacobi@open-xchange.com
 */

package com.openexchange.office.filter.odp.dom.components;

import org.json.JSONException;
import org.json.JSONObject;
import org.xml.sax.SAXException;
import com.openexchange.office.filter.core.DLList;
import com.openexchange.office.filter.core.DLNode;
import com.openexchange.office.filter.core.INodeAccessor;
import com.openexchange.office.filter.core.component.ComponentContext;
import com.openexchange.office.filter.odf.OdfOperationDoc;
import com.openexchange.office.filter.odf.OpAttrs;
import com.openexchange.office.filter.odf.components.Component;
import com.openexchange.office.filter.odf.draw.ConnectorShape;
import com.openexchange.office.filter.odf.draw.DrawFrame;
import com.openexchange.office.filter.odf.draw.DrawImage;
import com.openexchange.office.filter.odf.draw.DrawTextBox;
import com.openexchange.office.filter.odf.draw.GroupShape;
import com.openexchange.office.filter.odf.draw.Shape;
import com.openexchange.office.filter.odf.styles.StyleBase;
import com.openexchange.office.filter.odf.styles.StyleFamily;
import com.openexchange.office.filter.odf.styles.StyleManager;
import com.openexchange.office.filter.odf.table.Table;
import com.openexchange.office.filter.odp.dom.MasterPage;
import com.openexchange.office.filter.odp.dom.MasterPageWrapper;
import com.openexchange.office.filter.odp.dom.Page;

public class SlideComponent extends Component {

	final Page page;

    public SlideComponent(OdfOperationDoc operationDocument, DLNode<Object> pageNode, int componentNumber, boolean contentStyle) {
        super(operationDocument, pageNode, componentNumber);

        this.page = (Page)getObject();
        contentAutoStyle = contentStyle;
    }

	public SlideComponent(ComponentContext<OdfOperationDoc, Component> parentContext, DLNode<Object> pageNode, int componentNumber, boolean contentStyle) {
		super(parentContext, pageNode, componentNumber);

		this.page = (Page)getObject();
		contentAutoStyle = contentStyle;
	}

    @Override
    public String simpleName() {
        return "Slide";
    }

    public Page getPage() {
		return page;
	}

	@Override
	public Component getNextChildComponent(ComponentContext<OdfOperationDoc, Component> previousChildContext, Component previousChildComponent) {

        final int nextComponentNumber = previousChildComponent != null ? previousChildComponent.getNextComponentNumber() : 0;
        DLNode<Object> nextNode = previousChildContext != null ? previousChildContext.getNode().getNext() : page.getContent().getFirstNode();
		while(nextNode!=null) {
		    final Object o = nextNode.getData();
            if(o instanceof GroupShape) {
                return new ShapeGroupComponent(this, nextNode, nextComponentNumber);
            }
            else if(o instanceof DrawFrame) {
                return new FrameComponent(this, nextNode, nextComponentNumber);
            }
            else if(o instanceof ConnectorShape) {
                return new ShapeConnectorComponent(this, nextNode, nextComponentNumber);
            }
            else if(o instanceof Shape) {
		        return new ShapeComponent(this, nextNode, nextComponentNumber);
		    }
			nextNode = nextNode.getNext();
		}
		return null;
	}

    @Override
    public Component insertChildComponent(ComponentContext<OdfOperationDoc, Component> parentContext, DLNode<Object> contextNode, int number, Component child, Type type, JSONObject attrs) {

        DLList<Object> DLList = ((INodeAccessor)contextNode.getData()).getContent();
        DLNode<Object> referenceNode = child != null && child.getComponentNumber()== number ? child.getNode() : null;

        switch(type) {
            case AC_CONNECTOR:
            case AC_SHAPE: {
                final DLNode<Object> newShapeNode = new DLNode<Object>(Shape.createShape(operationDocument, attrs, null, true, isContentAutoStyle()));
                DLList.addNode(referenceNode, newShapeNode, true);
                return new ShapeComponent(parentContext, newShapeNode, number);
            }
            case AC_GROUP: {
                final DLNode<Object> newGroupNode = new DLNode<Object>(new GroupShape(operationDocument, null, true, isContentAutoStyle()));
                DLList.addNode(referenceNode, newGroupNode, true);
                return new ShapeGroupComponent(parentContext, newGroupNode, number);
            }
            case AC_IMAGE: {
                final DrawFrame drawFrame = new DrawFrame(operationDocument, null, true, isContentAutoStyle());
                final DLNode<Object> drawFrameNode = new DLNode<Object>(drawFrame);
                DLList.addNode(referenceNode, new DLNode<Object>(drawFrame), true);
                final DrawImage drawImage = new DrawImage(drawFrame);
                drawFrame.addContent(drawImage);
                return new FrameComponent(parentContext, drawFrameNode, number);
            }
            case AC_FRAME: {
                final DrawFrame drawFrame = new DrawFrame(operationDocument, null, true, isContentAutoStyle());
                final DLNode<Object> drawFrameNode = new DLNode<Object>(drawFrame);
                DLList.addNode(referenceNode, new DLNode<Object>(drawFrame),  true);
                final DrawTextBox drawTextBox = new DrawTextBox(drawFrame);
                drawFrame.addContent(drawTextBox);
                return new FrameComponent(parentContext, drawFrameNode, number);
            }
            case TABLE: {
                final DrawFrame drawFrame = new DrawFrame(operationDocument, null, true, isContentAutoStyle());
                final DLNode<Object> drawFrameNode = new DLNode<Object>(drawFrame);
                DLList.addNode(referenceNode, new DLNode<Object>(drawFrame), true);
                final Table table = new Table("presentation");
                drawFrame.addContent(table);
                return new FrameComponent(parentContext, drawFrameNode, number);
            }
            default : {
                throw new UnsupportedOperationException();
            }
        }
    }

	@Override
	public void applyAttrsFromJSON(JSONObject attrs) throws JSONException, SAXException {
	    page.applyAttrsFromJSON(operationDocument, contentAutoStyle, attrs);
	}

	@Override
	public void createJSONAttrs(OpAttrs attrs) {
	    final String style = page.getStyle();
	    if(style!=null&&!style.isEmpty()) {
	        final StyleManager styleManager = operationDocument.getDocument().getStyleManager();
	        final StyleBase styleBase = styleManager.getStyle(style, StyleFamily.DRAWING_PAGE, isContentAutoStyle());
	        if(styleBase!=null) {
	            styleBase.createAttrs(styleManager, attrs);
	        }
	    }
	    page.createAttrs(operationDocument, isContentAutoStyle(), attrs);
	}

	@Override
    public void delete(int count) {
	    if(page instanceof MasterPage) {
	        final Object parent = getParentContext().getObject();
	        if(parent instanceof MasterPageWrapper) {
	            operationDocument.getDocument().getStyleManager().getMasterStyles().getContent().remove(page);
	            operationDocument.getDocument().getTargetNodes().remove(((MasterPageWrapper)parent).getTargetId());
	        }
	    }
	    else {
	        super.delete(count);
	    }
    }
}
