/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/**
 *
 * @author sven.jacobi@open-xchange.com
 */

package com.openexchange.office.filter.odp.dom.components;

import java.util.Iterator;
import java.util.Map;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.xml.sax.SAXException;
import com.openexchange.office.filter.api.OCKey;
import com.openexchange.office.filter.core.DLList;
import com.openexchange.office.filter.core.DLNode;
import com.openexchange.office.filter.core.component.ComponentContext;
import com.openexchange.office.filter.odf.ITable;
import com.openexchange.office.filter.odf.OdfOperationDoc;
import com.openexchange.office.filter.odf.OpAttrs;
import com.openexchange.office.filter.odf.components.Component;
import com.openexchange.office.filter.odf.components.TextSpan_Base;
import com.openexchange.office.filter.odf.draw.DrawFrame;
import com.openexchange.office.filter.odf.draw.DrawingType;
import com.openexchange.office.filter.odf.draw.IDrawing;
import com.openexchange.office.filter.odf.draw.IDrawingType;
import com.openexchange.office.filter.odf.styles.StyleBase;
import com.openexchange.office.filter.odf.styles.StyleFamily;
import com.openexchange.office.filter.odf.styles.StyleManager;
import com.openexchange.office.filter.odf.styles.StyleParagraph;
import com.openexchange.office.filter.odf.table.Cell;
import com.openexchange.office.filter.odf.table.Column;
import com.openexchange.office.filter.odf.table.Row;
import com.openexchange.office.filter.odf.table.Table;
import com.openexchange.office.filter.odt.dom.Paragraph;
import com.openexchange.office.filter.odt.dom.TextSpan;

public class FrameComponent extends Component implements IDrawingType, ITable {

	final DrawFrame drawFrame;

	public FrameComponent(ComponentContext<OdfOperationDoc, Component> parentContext, DLNode<Object> textFrameNode, int componentNumber) {
		super(parentContext, textFrameNode, componentNumber);
		drawFrame = (DrawFrame)getObject();
	}

    @Override
    public String simpleName() {
        return "Frame";
    }

	public DrawFrame getDrawFrame() {
		return drawFrame;
	}

	@Override
    public DrawingType getType() {
		return drawFrame.getType();
	}

	@Override
	public Component getNextChildComponent(ComponentContext<OdfOperationDoc, Component> previousChildContext, Component previousChildComponent) {
        final IDrawing iDrawing = drawFrame.getDrawing();
        if(iDrawing!=null) {
            final int nextComponentNumber = previousChildComponent != null ? previousChildComponent.getNextComponentNumber() : 0;
            DLNode<Object> nextNode = previousChildContext!=null ? previousChildContext.getNode().getNext() : iDrawing.getContent().getFirstNode();
            while(nextNode!=null) {
                if(nextNode.getData() instanceof Paragraph) {
                    return new ParagraphComponent(this, nextNode, nextComponentNumber);
                }
                else if(nextNode.getData() instanceof Row) {
                    return new RowComponent(this, nextNode, nextComponentNumber);
                }
                nextNode = nextNode.getNext();
            }
        }
        return null;
	}

    @Override
    public Component insertChildComponent(ComponentContext<OdfOperationDoc, Component> parentContext, DLNode<Object> contextNode, int number, Component child, Type type, JSONObject attrs) {
		final IDrawing iDrawing = drawFrame.getDrawing();
		if(iDrawing!=null) {
	    	DLList<Object> DLList = iDrawing.getContent();
	        DLNode<Object> referenceNode = child != null && child.getComponentNumber()== number ? child.getNode() : null;

	        switch(type) {
	            case PARAGRAPH : {
	            	final DLNode<Object> newParagraphNode = new DLNode<Object>(new Paragraph(null));
	                DLList.addNode(referenceNode, newParagraphNode, true);
	                return new ParagraphComponent(parentContext, newParagraphNode, number);
	            }
	            default : {
	                throw new UnsupportedOperationException();
	            }
	        }
		}
		return null;
    }

    @Override
    public Component insertChildComponent(int number, JSONObject attrs, Type type)
    	throws UnsupportedOperationException, JSONException, SAXException {

        final Component c = insertChildComponent(this, getNode(), number, getChildComponent(number), type, attrs);
        if(c.getComponentNumber()==0&&c instanceof ParagraphComponent) {
        	// inheriting paragraph attributes from FrameStyle
        	final StyleManager styleManager = operationDocument.getDocument().getStyleManager();
    		final StyleParagraph paragraphStyle = new StyleParagraph(styleManager.getUniqueStyleName(StyleFamily.PARAGRAPH, isContentAutoStyle()), true, isContentAutoStyle());

    		final OpAttrs opAttrs = new OpAttrs();
    		styleManager.collectAllParagraphPropertiesFromStyle(opAttrs, drawFrame.getStyleName(), StyleFamily.GRAPHIC, isContentAutoStyle());
    		styleManager.collectAllTextPropertiesFromStyle(opAttrs, drawFrame.getStyleName(), StyleFamily.GRAPHIC, isContentAutoStyle());

    		// taking care of frame margins, they have to be applied to the paragraph style :-(
    		final OpAttrs allDrawingAttrs = new OpAttrs();
    		styleManager.collectAllGraphicPropertiesFromStyle(allDrawingAttrs, drawFrame.getStyleName(), StyleFamily.GRAPHIC, isContentAutoStyle(), false);
    		final Map<String, Object> drawingAttrs = allDrawingAttrs.getMap(OCKey.DRAWING.value(), false);
    		if(drawingAttrs!=null) {
    			final Map<String, Object> paragraphAttrs = opAttrs.getMap(OCKey.PARAGRAPH.value(), true);
    			final Object marginLeft = drawingAttrs.get(OCKey.MARGIN_LEFT.value());
    			final Object marginTop = drawingAttrs.get(OCKey.MARGIN_TOP.value());
    			final Object marginRight = drawingAttrs.get(OCKey.MARGIN_RIGHT.value());
    			final Object marginBottom = drawingAttrs.get(OCKey.MARGIN_BOTTOM.value());
    			if(marginLeft!=null) {
    				paragraphAttrs.put(OCKey.MARGIN_LEFT.value(), marginLeft);
    			}
    			if(marginTop!=null) {
    				paragraphAttrs.put(OCKey.MARGIN_TOP.value(), marginTop);
    			}
    			if(marginRight!=null) {
    				paragraphAttrs.put(OCKey.MARGIN_RIGHT.value(), marginRight);
    			}
    			if(marginBottom!=null) {
    				paragraphAttrs.put(OCKey.MARGIN_BOTTOM.value(), marginBottom);
    			}
    		}
    		if(attrs!=null) {
				StyleManager.deepCopy(attrs.asMap(), opAttrs);
    		}
    		paragraphStyle.applyAttrs(styleManager, new JSONObject(opAttrs));
    		final String existingStyleId = styleManager.getExistingStyleIdForStyleBase(paragraphStyle);
    		if(existingStyleId!=null) {
    			((Paragraph)((ParagraphComponent)c).getObject()).setStyleName(existingStyleId);
    		}
    		else {
    			styleManager.addStyle(paragraphStyle);
        		((Paragraph)((ParagraphComponent)c).getObject()).setStyleName(paragraphStyle.getName());
    		}
        }
        else if(attrs!=null) {
            c.applyAttrsFromJSON(attrs);
        }
        return c;
    }

	@Override
	public void applyAttrsFromJSON(JSONObject attrs)
			throws JSONException {

		drawFrame.applyAttrsFromJSON(operationDocument, attrs, isContentAutoStyle());
	}

	@Override
	public void createJSONAttrs(OpAttrs attrs) {

		drawFrame.createAttrs(operationDocument, attrs, isContentAutoStyle());
	}

    @Override
    public void insertRows(int rowPosition, int count, boolean insertDefaultCells, int referenceRowNumber, JSONObject attrs) throws JSONException, SAXException {
        final Table table = ((Table)drawFrame.getDrawing());
        final RowComponent referenceRowComponent = referenceRowNumber!=-1 ? (RowComponent)getChildComponent(referenceRowNumber) : null;
        final DLList<Object> tblContent = table.getContent();
        final Component oldTr = getChildComponent(rowPosition);
        DLNode<Object> referenceNode = oldTr!=null?oldTr.getNode():null;

        for(int i = 0; i<count; i++) {
            final Row row = !insertDefaultCells && referenceRowComponent != null ? ((Row)referenceRowComponent.getObject()).clone() : new Row();
            final DLNode<Object> rowNode = new DLNode<Object>(row);
            tblContent.addNode(referenceNode, rowNode, true);
            referenceNode = rowNode;
            if(insertDefaultCells) {
                for(int j = 0; j < table.getColumns().size(); j++) {
                    final Cell cell = new Cell();
                    cell.getContent().add(new Paragraph(null));
                    row.getContent().add(cell);
                }
            }
            else if(referenceRowComponent!=null) {       // we try to create the new row from the referenceRow
                CellComponent referenceCellComponent = (CellComponent)referenceRowComponent.getNextChildComponent(null, null);
                while(referenceCellComponent!=null) {
                    final Cell cell = ((Cell)referenceCellComponent.getObject()).clone();
                    final Paragraph p = new Paragraph(null);
                    cell.getContent().add(p);
                    row.getContent().add(cell);
                    applyReferenceCellAttributes(cell, p, referenceCellComponent);
                    referenceCellComponent = (CellComponent)referenceCellComponent.getNextComponent();
                }
            }
        }
        if(attrs!=null) {
            Component c = getNextChildComponent(null, null).getComponent(rowPosition);
            for(int i=0; i<count; i++) {
                c.applyAttrsFromJSON(attrs);
                c = c.getNextComponent();
            }
        }
    }

    private void applyReferenceCellAttributes(Cell c, Paragraph p, CellComponent referenceCellComponent) {
        if(referenceCellComponent!=null) {
            final ParagraphComponent referenceParagraphComponent = (ParagraphComponent)referenceCellComponent.getNextChildComponent(null, null);
            if(referenceParagraphComponent!=null) {
                final Paragraph referenceParagraph = referenceParagraphComponent.getParagraph();
                p.setStyleName(referenceParagraph.getStyleName());
                final Component component = referenceParagraphComponent.getNextChildComponent(null, null);
                if(component instanceof TextSpan_Base) {
                    final TextSpan textSpan = ((TextSpan_Base)component).getTextSpan();
                    if(textSpan!=null) {
                        final OpAttrs characterAttrs = new OpAttrs();
                        try {
                            component.createJSONAttrs(characterAttrs);
                        } catch (SAXException e) {
                            //
                        }
                        try {
                            p.setStyleName(operationDocument.getDocument().getStyleManager().createStyle(StyleFamily.PARAGRAPH, referenceParagraph.getStyleName(), isContentAutoStyle(), new JSONObject(characterAttrs)));
                            c.setStyleName(operationDocument.getDocument().getStyleManager().createStyle(StyleFamily.TABLE_CELL, referenceParagraph.getStyleName(), isContentAutoStyle(), new JSONObject(characterAttrs)));
                        } catch (JSONException e) {
                            //
                        }
                    }
                }
            }
        }
    }

    @Override
    public void insertColumn(JSONArray tableGrid, int gridPosition, String insertMode) throws JSONException {
        final Table table = ((Table)drawFrame.getDrawing());
        boolean before = insertMode.equals("before");
        RowComponent trComponent = (RowComponent)getNextChildComponent(null, null);
        while(trComponent!=null) {
            CellComponent tcReference = (CellComponent)trComponent.getNextChildComponent(null, null);
            CellComponent tcReferenceLast = null;
            CellComponent destination = null;
            while(tcReference!=null) {
                tcReferenceLast = tcReference;
                if(gridPosition>=tcReference.getGridPosition()&&gridPosition<tcReference.getNextGridPosition()) {
                    destination = tcReference;
                    break;
                }
                tcReference = (CellComponent)tcReference.getNextComponent();
            }
            final Cell tc = tcReferenceLast!=null ? tcReferenceLast.getCell().clone() : new Cell();
            final Paragraph p = new Paragraph(null);
            tc.setColumnSpan(1);
            tc.getContent().add(p);
            final DLList<Object> rowContent = trComponent.getRow().getContent();
            if(destination==null) {
                rowContent.add(tc);
            }
            else {
                final ComponentContext<OdfOperationDoc, Component> contextChild = destination.getContextChild();
                rowContent.addNode(contextChild.getNode(), new DLNode<Object>(tc), before);
            }
            trComponent = (RowComponent)trComponent.getNextComponent();
        }
        table.applyTableGrid(operationDocument, null, tableGrid, isContentAutoStyle());
    }

    @Override
    public void deleteColumns(int gridStart, int gridEnd) {
        final Table table = ((Table)drawFrame.getDrawing());
        Component trComponent = getNextChildComponent(null, null);
        while(trComponent!=null) {
            DLNode<Object> startNode = null;
            DLNode<Object> endNode = null;
            CellComponent tcComponent = (CellComponent)trComponent.getNextChildComponent(null, null);
            while(tcComponent!=null) {
                final int x1 = tcComponent.getGridPosition();
                final int x2 = tcComponent.getNextGridPosition();
                if(Math.max(x2, gridEnd + 1) - Math.min(x1, gridStart) < (x2 - x1) + ((gridEnd+1)-gridStart)) {
                    final ComponentContext<OdfOperationDoc, Component> contextChild = tcComponent.getContextChild();
                    if (startNode==null) {
                        startNode = contextChild.getNode();
                    }
                    endNode = contextChild.getNode();
                }
                if(tcComponent.getNextGridPosition()>gridEnd)
                    break;
                tcComponent = (CellComponent)tcComponent.getNextComponent();
            }
            if(startNode!=null) {
                ((Row)trComponent.getObject()).getContent().removeNodes(startNode, endNode);
            }
            trComponent = trComponent.getNextComponent();
        }
        for(int i = gridEnd; i > gridStart; i--) {
            table.getColumns().remove(i);
        }
    }

    @Override
    public void splitTable(int splitPos, ITable splitTableComponent) {
        final Table table = ((Table)drawFrame.getDrawing());
        final Table destTable = splitTableComponent.getTable();
        final DLList<Object> sourceContent = table.getContent();
        final DLList<Object> destContent = destTable.getContent();
        final DLNode<Object> splitNode = sourceContent.getNode(splitPos);

        // cloning table attributes
        destTable.setAttributes(table.getAttributes().clone());

        // moving table rows
        sourceContent.moveNodes(splitNode, destContent);

        // cloning table columns
        final Iterator<Column> columnIter = table.getColumns().iterator();
        while(columnIter.hasNext()) {
            destTable.getColumns().add(columnIter.next().clone());
        }

        // taking care of the style:master-page attribute within the style, (we have to remove it, otherwise the split table
        // begins at a new page :(
        final StyleManager styleManager = operationDocument.getDocument().getStyleManager();
        final StyleBase styleBase = styleManager.getStyle(destTable.getStyleName(), StyleFamily.TABLE, isContentAutoStyle());
        if(styleBase!=null) {
            final String masterPage = styleBase.getAttribute("style:master-page-name");
            if(masterPage!=null&&!masterPage.isEmpty()) {
                final StyleBase clonedStyle = styleBase.clone();
                clonedStyle.getAttributes().remove("style:master-page-name");
                final String existingStyleId = styleManager.getExistingStyleIdForStyleBase(clonedStyle);
                if(existingStyleId!=null) {
                    destTable.setStyleName(existingStyleId);
                }
                else {
                    clonedStyle.setName(styleManager.getUniqueStyleName(StyleFamily.TABLE, isContentAutoStyle()));
                    destTable.setStyleName(clonedStyle.getName());
                    styleManager.addStyle(clonedStyle);
                }
            }
        }
    }

    @Override
    public void mergeTable() {
        final Table table = ((Table)drawFrame.getDrawing());
        final Component nextComponent = getNextComponent();
        final ITable nextTable = (ITable)nextComponent;
        nextTable.getTable().getContent().moveNodes(table.getContent());
        nextComponent.delete(1);
    }

    @Override
    public Table getTable() {
        final Object o = drawFrame.getDrawing();
        return o instanceof Table ? (Table)o : null;
    }
}
