/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/**
 *
 * @author sven.jacobi@open-xchange.com
 */

package com.openexchange.office.filter.odf.components;

import java.util.Set;
import java.util.logging.Logger;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.json.JSONTokener;
import org.odftoolkit.odfdom.doc.OdfChartDocument;
import org.odftoolkit.odfdom.doc.OdfSpreadsheetDocument;
import org.odftoolkit.odfdom.dom.OdfDocumentNamespace;
import org.odftoolkit.odfdom.pkg.OdfPackage;
import org.xml.sax.SAXException;
import com.openexchange.office.filter.api.OCKey;
import com.openexchange.office.filter.core.DLList;
import com.openexchange.office.filter.core.DLNode;
import com.openexchange.office.filter.core.INodeAccessor;
import com.openexchange.office.filter.core.SplitMode;
import com.openexchange.office.filter.core.component.ComponentContext;
import com.openexchange.office.filter.core.component.IComponent;
import com.openexchange.office.filter.odf.AttributesImpl;
import com.openexchange.office.filter.odf.ElementNS;
import com.openexchange.office.filter.odf.MetaData;
import com.openexchange.office.filter.odf.Namespaces;
import com.openexchange.office.filter.odf.OdfOperationDoc;
import com.openexchange.office.filter.odf.OpAttrs;
import com.openexchange.office.filter.odf.draw.DrawFrame;
import com.openexchange.office.filter.odf.draw.DrawObject;
import com.openexchange.office.filter.ods.dom.chart.ChartContent;

public abstract class Component extends ComponentContext<OdfOperationDoc, Component> implements IComponent<ComponentContext<OdfOperationDoc, Component>, Component> {

	protected int componentNumber;

	protected static boolean contentAutoStyle;

    public enum Type {
    	PARAGRAPH,
    	TABLE,
    	TR,
    	TC,
    	TAB,
    	HARDBREAK,
    	FIELD,
    	COMMENT_REFERENCE,
    	COMMENT_RANGE_START,
    	COMMENT_RANGE_END,
    	AC_SHAPE,
    	AC_GROUP,
    	AC_IMAGE,
    	AC_FRAME,
    	AC_CHART,
    	AC_CONNECTOR
    }

    public Component(OdfOperationDoc operationDocument, DLNode<Object> o, int componentNumber) {
        super(operationDocument, o);

        this.componentNumber = componentNumber;
    }

    public Component(ComponentContext<OdfOperationDoc, Component> parentContext, DLNode<Object> o, int componentNumber) {
        super(parentContext, o);

        this.componentNumber = componentNumber;
    }

    @Override
    public int getComponentNumber() {
        return componentNumber;
    }

    @Override
    public int getNextComponentNumber() {
        return componentNumber + 1;
    }

    public boolean isContentAutoStyle() {
        return contentAutoStyle;
    }

    @Override
    public Component getNextComponent() {
    	if(getParentContext()==null) {
    		return null;
    	}
        ComponentContext<OdfOperationDoc, Component> parentContext = this;
        Component nextComponent = null;
        do {
            ComponentContext<OdfOperationDoc, Component> previousContext = parentContext;
            parentContext = parentContext.getParentContext();
            nextComponent = parentContext.getNextChildComponent(previousContext, this);
        }
        while(nextComponent==null&&!(parentContext instanceof Component));
        return nextComponent;
    }

    @Override
    public Component getParentComponent() {
    	ComponentContext<OdfOperationDoc, Component> parentContext = getParentContext();
    	while(!(parentContext instanceof Component)) {
    		parentContext = parentContext.getParentContext();
    	}
    	return (Component)parentContext;
    }

    public ComponentContext<OdfOperationDoc, Component> getContextChild() {
        ComponentContext<OdfOperationDoc, Component> childContext = this;
        ComponentContext<OdfOperationDoc, Component> parentContext = getParentContext();
        while(!(parentContext instanceof Component)) {
            childContext = parentContext;
            parentContext = childContext.getParentContext();
        }
        return childContext;
    }

    public DrawFrame getChart(OdfOperationDoc operationDoc, boolean contentStyle) {

        try {

            final OdfSpreadsheetDocument document = (OdfSpreadsheetDocument)operationDoc.getDocument();

            // the default is two cell anchor ...
            final DrawFrame drawFrame = new DrawFrame(operationDoc, null, true, contentStyle);
            final DrawObject drawObject= new DrawObject(drawFrame);
            drawFrame.getContent().add(drawObject);
            OdfPackage pkg = document.getPackage();

            long index = 1;
            final String prefix = "Object ";

            Set<String> filePaths = pkg.getFilePaths();
            for (String path : filePaths) {
                if (path.startsWith(prefix) && path.endsWith("/content.xml")) {
                    try {
                        long other = Long.parseLong(path.replace(prefix, "").replace("/content.xml", ""));

                        index = Math.max(other + 1, index);

                    } catch (NumberFormatException e) {
                        Logger.getAnonymousLogger().info(e.toString());
                    }
                }
            }

            String name = prefix + index;

            final StringBuilder bContent = new StringBuilder();
            bContent.append("<?xml version=\"1.0\" encoding=\"UTF-8\"?><office:document-content");
            for (OdfDocumentNamespace nameSpace : OdfDocumentNamespace.values()) {
                bContent.append(" xmlns:" + nameSpace.getPrefix() + "=\"" + nameSpace.getUri() + "\"");
            }
            bContent.append("><office:body><office:chart></office:chart></office:body></office:document-content>");
            byte[] contentBytes = bContent.toString().getBytes();

            pkg.insert("".getBytes(), name + "/", OdfChartDocument.OdfMediaType.CHART.getMediaTypeString());
            pkg.insert(contentBytes, name + "/content.xml", "text/xml");

            pkg.insert("<?xml version=\"1.0\" encoding=\"UTF-8\"?><office:document-styles xmlns:office=\"urn:oasis:names:tc:opendocument:xmlns:office:1.0\" xmlns:style=\"urn:oasis:names:tc:opendocument:xmlns:style:1.0\" xmlns:text=\"urn:oasis:names:tc:opendocument:xmlns:text:1.0\" xmlns:table=\"urn:oasis:names:tc:opendocument:xmlns:table:1.0\" xmlns:draw=\"urn:oasis:names:tc:opendocument:xmlns:drawing:1.0\" xmlns:fo=\"urn:oasis:names:tc:opendocument:xmlns:xsl-fo-compatible:1.0\" xmlns:xlink=\"http://www.w3.org/1999/xlink\" xmlns:dc=\"http://purl.org/dc/elements/1.1/\" xmlns:number=\"urn:oasis:names:tc:opendocument:xmlns:datastyle:1.0\" xmlns:svg=\"urn:oasis:names:tc:opendocument:xmlns:svg-compatible:1.0\" xmlns:chart=\"urn:oasis:names:tc:opendocument:xmlns:chart:1.0\" xmlns:dr3d=\"urn:oasis:names:tc:opendocument:xmlns:dr3d:1.0\" xmlns:script=\"urn:oasis:names:tc:opendocument:xmlns:script:1.0\" xmlns:ooo=\"http://openoffice.org/2004/office\" xmlns:ooow=\"http://openoffice.org/2004/writer\" xmlns:oooc=\"http://openoffice.org/2004/calc\" xmlns:dom=\"http://www.w3.org/2001/xml-events\" xmlns:rpt=\"http://openoffice.org/2005/report\" xmlns:of=\"urn:oasis:names:tc:opendocument:xmlns:of:1.2\" xmlns:xhtml=\"http://www.w3.org/1999/xhtml\" xmlns:grddl=\"http://www.w3.org/2003/g/data-view#\" xmlns:tableooo=\"http://openoffice.org/2009/table\" xmlns:chartooo=\"http://openoffice.org/2010/chart\" xmlns:drawooo=\"http://openoffice.org/2010/draw\" xmlns:calcext=\"urn:org:documentfoundation:names:experimental:calc:xmlns:calcext:1.0\" xmlns:loext=\"urn:org:documentfoundation:names:experimental:office:xmlns:loext:1.0\" xmlns:field=\"urn:openoffice:names:experimental:ooo-ms-interop:xmlns:field:1.0\" xmlns:css3t=\"http://www.w3.org/TR/css3-text/\" office:version=\"1.2\"><office:styles/></office:document-styles>".getBytes(), name + "/styles.xml", "text/xml");

            OdfChartDocument chartDoc = (OdfChartDocument)document.loadSubDocument("./" + name);

            // retrieving chart content
            final ChartContent chart = (ChartContent) chartDoc.getContentDom();
            drawObject.setChart(chart);

            ElementNS officeChart = (ElementNS) chart.getElementsByTagName("office:chart").item(0);

            ElementNS chartArea = chart.newChartElement("chart");
            ElementNS plotArea = chart.newChartElement("plot-area");
            ElementNS wall = chart.newChartElement("wall");

            JSONObject fillNone = new JSONObject(new JSONTokener("{" + OCKey.FILL.value() + ":{" + OCKey.TYPE.value() + ":'none'}}"));

            chart.applyStyleAttrs(fillNone, wall);

            officeChart.appendChild(chartArea);
            chartArea.appendChild(plotArea);
            plotArea.appendChild(wall);

            chart.setChartArea(chartArea);
            chart.setPlotArea(plotArea);

            chartDoc.getOfficeMetaData(true);
            chartDoc.getStylesDom();

            MetaData docMeta = document.getOfficeMetaData(false);
            if (docMeta != null) {
                String generator = docMeta.getGenerator();
                if (generator != null && !generator.isEmpty()) {
                    chartDoc.getOfficeMetaData(false).setGenerator(generator);
                }

            }

            final AttributesImpl drawObjectAttributes = drawObject.getAttributes();
            drawObjectAttributes.setValue(Namespaces.XLINK, "href", "xlink:href", "./" + name);
            drawObjectAttributes.setValue(Namespaces.XLINK, "actuate", "xlink:actuate", "onLoad");
            drawObjectAttributes.setValue(Namespaces.XLINK, "show", "xlink:show", "embed");
            drawObjectAttributes.setValue(Namespaces.XLINK, "type", "xlink:type", "simple");
            drawObjectAttributes.setValue(Namespaces.DRAW, "name", "draw:name", name);
            return drawFrame;

        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public abstract Component getNextChildComponent(ComponentContext<OdfOperationDoc, Component> previousChildContext, Component previousChildComponent);

    @Override
    public Component getChildComponent(int number) {
    	Component c = getNextChildComponent(null, null);
    	if(c!=null) {
    		c = c.getComponent(number);
    	}
    	return c;
    }

    @Override
    public Component getComponent(int number) {
    	Component c = this;
    	while(c!=null&&c.getNextComponentNumber() <= number) {
    		c = c.getNextComponent();
    	}
    	return c;
    }

    @Override
    public Component getComponent(JSONArray position, int positionCount) {
        try {
            if(positionCount==0) {
                return this;
            }
            // a part has to implement the ContentAccessor interface, otherwise no component can be accessed
            Component c = getNextChildComponent(null, null);
            for(int i=0; c!=null&&i<positionCount;) {
                c = c.getComponent(position.getInt(i++));
                if(i!=positionCount) {
                    c = c.getNextChildComponent(null, null);
                }
            }
            return c;
        }
        catch(JSONException e) {
            // ups
        }
        return null;
    }

    @Override
    public void delete(int count) {

    	final ComponentContext<OdfOperationDoc, Component> contextChild = getContextChild();

    	final int endComponent = (getComponentNumber()+count)-1;
        Component component = this;
        while(true) {
        	if(component.getNextComponentNumber()>endComponent) {
        		break;
        	}
        	component = component.getNextComponent();
        }
        component.splitEnd(endComponent, SplitMode.DELETE);
        final DLList<Object> content = ((INodeAccessor)contextChild.getParentContext().getObject()).getContent();
        content.removeNodes(contextChild.getNode(), component.getContextChild().getNode());
    }

    public static void move(Component rootComponent, JSONArray start, JSONArray end, JSONArray to)
        throws JSONException {

        if(end==null) {
            end = start;
        }
        else if(start.length()!=end.length())
            throw new JSONException("ooxml export: move operation, size of startPosition != size of endPosition");

        final int s = start.getInt(start.length()-1);
        final int t = to.getInt(to.length()-1);
        Component sourceBegComponent = rootComponent.getComponent(start, start.length());
        Component sourceEndComponent = rootComponent.getComponent(end, end.length());
        sourceBegComponent.splitStart(s, SplitMode.DELETE);
        sourceEndComponent.splitEnd(end.getInt(end.length()-1), SplitMode.DELETE);

        // destComponent is empty if the content is to be appended
        final Component destComponent = rootComponent.getComponent(to, to.length());
        Component destParentComponent;
        if(destComponent!=null) {
            destComponent.splitStart(to.getInt(to.length()-1), SplitMode.DELETE);
            destParentComponent = destComponent.getParentComponent();
        }
        else {
            destParentComponent = rootComponent.getComponent(to, to.length()-1);
        }
        final DLNode<Object> sourceStartContextNode = sourceBegComponent.getContextChild().getNode();
        final DLNode<Object> sourceEndContextNode = sourceEndComponent.getContextChild().getNode();
        final DLList<Object> sourceContent = ((INodeAccessor)sourceBegComponent.getParentComponent().getObject()).getContent();
        final DLList<Object> destContent = ((INodeAccessor)destParentComponent.getObject()).getContent();
        final boolean before = sourceContent==destContent && s <= t ? false : true;
        sourceContent.moveNodes(sourceStartContextNode, sourceEndContextNode, destContent, destComponent!=null ? destComponent.getContextChild().getNode() :  null, before);
    }

    public Component insertChildComponent(ComponentContext<OdfOperationDoc, Component> parentContext, DLNode<Object> contextNode, int number, Component child, Type type, JSONObject attrs) {
    	return null;
    }

    public Component insertChildComponent(int number, JSONObject attrs, Type type)
    	throws UnsupportedOperationException, JSONException, SAXException {

        final Component c = insertChildComponent(this, getNode(), number, getChildComponent(number), type, attrs);
        if(attrs!=null) {
            c.applyAttrsFromJSON(attrs);
        }
        return c;
    }

    public abstract void applyAttrsFromJSON(JSONObject attrs)
        throws JSONException, SAXException;

    public abstract void createJSONAttrs(OpAttrs attrs)
    	throws SAXException;

	public static String componentToString(Component component) {
    	final StringBuffer stringBuffer = new StringBuffer();
    	componentToString(component, stringBuffer);
    	return stringBuffer.toString();
    }

	public abstract String simpleName();

	private static void componentToString(Component component, StringBuffer stringBuffer) {
	    stringBuffer.append(component.simpleName());
    	Component childComponent = component.getChildComponent(0);
    	if(childComponent!=null) {
    		stringBuffer.append('(');
	    	while(childComponent!=null) {
	    		componentToString(childComponent, stringBuffer);
	    		childComponent = childComponent.getNextComponent();
	    		if(childComponent!=null) {
	    			stringBuffer.append(',');
	    		}
	    	}
	    	stringBuffer.append(')');
    	}
    }
}
