/*
*
*    OPEN-XCHANGE legal information
*
*    All intellectual property rights in the Software are protected by
*    international copyright laws.
*
*
*    In some countries OX, OX Open-Xchange, open xchange and OXtender
*    as well as the corresponding Logos OX Open-Xchange and OX are registered
*    trademarks.
*    The use of the Logos is not covered by the GNU General Public License.
*    Instead, you are allowed to use these Logos according to the terms and
*    conditions of the Creative Commons License, Version 2.5, Attribution,
*    Non-commercial, ShareAlike, and the interpretation of the term
*    Non-commercial applicable to the aforementioned license is published
*    on the web site http://www.open-xchange.com/EN/legal/index.html.
*
*    Please make sure that third-party modules and libraries are used
*    according to their respective licenses.
*
*    Any modifications to this package must retain all copyright notices
*    of the original copyright holder(s) for the original code used.
*
*    After any such modifications, the original and derivative code shall remain
*    under the copyright of the copyright holder(s) and/or original author(s)per
*    the Attribution and Assignment Agreement that can be located at
*    http://www.open-xchange.com/EN/developer/. The contributing author shall be
*    given Attribution for the derivative code and a license granting use.
*
*     Copyright (C) 2016 OX Software GmbH
*     Mail: info@open-xchange.com
*
*
*     This program is free software; you can redistribute it and/or modify it
*     under the terms of the GNU General Public License, Version 2 as published
*     by the Free Software Foundation.
*
*     This program is distributed in the hope that it will be useful, but
*     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
*     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
*     for more details.
*
*     You should have received a copy of the GNU General Public License along
*     with this program; if not, write to the Free Software Foundation, Inc., 59
*     Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
*/

package com.openexchange.office.backup.manager.impl;

import java.util.Queue;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ConcurrentLinkedQueue;

import org.apache.commons.lang.StringUtils;

import com.openexchange.filemanagement.ManagedFileManagement;
import com.openexchange.log.Log;
import com.openexchange.office.backup.distributed.GCManagedFiles;
import com.openexchange.office.tools.osgi.ServiceLookupRegistry;

@SuppressWarnings("deprecation")
public class GCObsoleteStreams implements GCManagedFiles {
    static private final org.apache.commons.logging.Log LOG = Log.loggerFor(GCObsoleteStreams.class);

    private static final long GC_TIMER_PERIOD = 16000;
    private static final long GC_ACTIVE_TIME  = 4000;
    private static final long GC_TIMEOUT      = 30000;

    private final Queue<GCObsoleteStreamEntry> gcManagedFileIds = new ConcurrentLinkedQueue<>();
    private final ManagedFileManagement managedFileManagementService;
    private Timer timer = new Timer("com.openexchange.office.backup.GC");
    private TimerTask gcStreamsTimerTask = null;

    /**
     * Initializes a new GCObsoleteStreams instance which uses a timer-based
     * background processing-to remove obsolete managed files.
     *
     */
	public GCObsoleteStreams() {
        managedFileManagementService = ServiceLookupRegistry.get().getService(ManagedFileManagement.class);

        // creates a task which writes pending operations data to managed files
        gcStreamsTimerTask = new TimerTask() {

            @SuppressWarnings("synthetic-access")
            @Override
            public void run() {
                boolean allTimeupRemoved = false;
                boolean timeout = false;
                int nCount = 0;
                long timeStart = System.currentTimeMillis();

                LOG.trace("GCObsoleteStreams starts garbage collection round - " +  gcManagedFileIds.size() + " entries in gc queue");

                while (!timeout && !allTimeupRemoved) {
                    try {
                        final GCObsoleteStreamEntry managedFileToBeGC = gcManagedFileIds.peek();

                        if (null == managedFileToBeGC)
                            break;

                        final long diff = Math.abs(System.currentTimeMillis() - managedFileToBeGC.timeAdded());
                        if (diff >= GC_TIMEOUT) {
                            final String managedFileId = managedFileToBeGC.managedFileId();

                            LOG.trace("Managed file " + managedFileId + " removed by GCObsoleteStreams");

                            nCount++;
                            managedFileManagementService.removeByID(managedFileId);
                            gcManagedFileIds.poll();
                        } else {
                            allTimeupRemoved = true;
                        }
                    } catch (Exception e) {
                        LOG.info("RT2: Exception caught while trying to remove an obsolete managed file", e);
                    }

                    timeout = (System.currentTimeMillis() - timeStart) > GC_ACTIVE_TIME;
                }

                final long nTimeUsed = Math.abs(System.currentTimeMillis() - timeStart);
                LOG.debug("GCObsoleteStreams garbage collected " + nCount + " managed files in " + nTimeUsed + " ms");
            }
        };

        // start the periodic timer with the specified timeout
        timer.schedule(gcStreamsTimerTask, GC_TIMER_PERIOD, GC_TIMER_PERIOD);

        LOG.trace("GCObsoleteStreams timer task created");
    }

	/**
	 * Adds a managed file id to be removed by garbage collector.
	 *
	 * @param fileId a managed file id - empty strings or null will be ignored
	 */
	@Override
    public void addObsoleteManagedFileId(String fileId) {
		if (StringUtils.isNotEmpty(fileId)) {
			final GCObsoleteStreamEntry entry = new GCObsoleteStreamEntry(fileId);
			gcManagedFileIds.add(entry);
		}
	}

	/**
	 * Removes pending jobs to be processed and stop the background
	 * processing.
	 */
	public void dispose() {
		LOG.trace("GCObsoleteStreams.dispose() called");

	    final Timer aTimer = timer;
	    if (aTimer != null) {
	        aTimer.cancel();
	        aTimer.purge();
	        timer = null;
	    }

	    gcStreamsTimerTask = null;
	    gcManagedFileIds.clear();
	}

}
