/*
*
*    OPEN-XCHANGE legal information
*
*    All intellectual property rights in the Software are protected by
*    international copyright laws.
*
*
*    In some countries OX, OX Open-Xchange, open xchange and OXtender
*    as well as the corresponding Logos OX Open-Xchange and OX are registered
*    trademarks.
*    The use of the Logos is not covered by the GNU General Public License.
*    Instead, you are allowed to use these Logos according to the terms and
*    conditions of the Creative Commons License, Version 2.5, Attribution,
*    Non-commercial, ShareAlike, and the interpretation of the term
*    Non-commercial applicable to the aforementioned license is published
*    on the web site http://www.open-xchange.com/EN/legal/index.html.
*
*    Please make sure that third-party modules and libraries are used
*    according to their respective licenses.
*
*    Any modifications to this package must retain all copyright notices
*    of the original copyright holder(s) for the original code used.
*
*    After any such modifications, the original and derivative code shall remain
*    under the copyright of the copyright holder(s) and/or original author(s)per
*    the Attribution and Assignment Agreement that can be located at
*    http://www.open-xchange.com/EN/developer/. The contributing author shall be
*    given Attribution for the derivative code and a license granting use.
*
*     Copyright (C) 2016 OX Software GmbH
*     Mail: info@open-xchange.com
*
*
*     This program is free software; you can redistribute it and/or modify it
*     under the terms of the GNU General Public License, Version 2 as published
*     by the Free Software Foundation.
*
*     This program is distributed in the hope that it will be useful, but
*     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
*     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
*     for more details.
*
*     You should have received a copy of the GNU General Public License along
*     with this program; if not, write to the Free Software Foundation, Inc., 59
*     Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
*/
package com.openexchange.office.backup.manager.impl;

import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Queue;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.stream.Collectors;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.log.Log;
import com.openexchange.office.message.MessageChunk;
import com.openexchange.office.message.MessagePropertyKey;
import com.openexchange.office.message.OperationHelper;
import com.openexchange.office.tools.directory.DocRestoreID;
import com.openexchange.office.tools.json.JSONHelper;

/**
 *
 * @author <a href="mailto:carsten.driesner@open-xchange.com">Carsten Driesner</a>
 * @since 7.10.0
 */
public class AsyncBackupWorker {
    @SuppressWarnings("deprecation")
    static private final org.apache.commons.logging.Log LOG = Log.loggerFor(AsyncBackupWorker.class);

    static private final long WRITEBACK_OPERATIONS_TIMER_PERIOD = 30000;
    static private final long WRITEBACK_ACTIVE_TIME             = 3000;

    private final Queue<OperationDocData> pendingDocData = new ConcurrentLinkedQueue<OperationDocData>();
    private Timer timer = new Timer("com.openexchange.office.backup.DocumentBackupManager");
    private TimerTask writeBackTimerTask = null;
    private final WeakReference<StoreOperationData> storeOperationDataProcessor;

    /**
     * Initializes a new AsyncBackupWorker instance which used a timer-based
     * background processing-
     *
     * @param services
     * @param storeOperationData
     */
	public AsyncBackupWorker(final StoreOperationData storeOpsData) {
		this.storeOperationDataProcessor = new WeakReference<StoreOperationData>(storeOpsData);

        // creates a task which writes pending operations data to managed files
		writeBackTimerTask = new TimerTask() {

            @SuppressWarnings({ "synthetic-access" })
            @Override
            public void run() {
                boolean timeout = false;
                long timeStart = System.currentTimeMillis();

                while (!timeout) {
                    try {
                        final OperationDocData currOpsDocData = pollOperationDocDataFromFront(pendingDocData);

                        if (null == currOpsDocData)
                            break;

                        final StoreOperationData storeOperationData = storeOperationDataProcessor.get();
                        if (null != storeOperationData) {

	                        // store the pending operations to hazelcast/distributed managed file
	                        storeOperationData.storePendingOperationsToDocumentState(
	                            currOpsDocData.getID(), currOpsDocData.getOperations(), currOpsDocData.getOSN());
                        }
                    } catch (Exception e) {
                        LOG.warn("RT connection: Exception caught while trying to save operations to backup file", e);
                    }

                    timeout = (System.currentTimeMillis() - timeStart) > WRITEBACK_ACTIVE_TIME;
                }
            }
        };

        // start the periodic timer with the specified timeout
        timer.schedule(writeBackTimerTask, WRITEBACK_OPERATIONS_TIMER_PERIOD, WRITEBACK_OPERATIONS_TIMER_PERIOD);
	}

	/**
	 *
	 * @param docResId
	 * @param actionChunk
	 */
	void appendOperations(final DocRestoreID docResId, final MessageChunk actionChunk, int newOSN) {
        final OperationDocData opsData = new OperationDocData(docResId, actionChunk, newOSN);

        pendingDocData.add(opsData);
	}

	/**
	 *
	 */
	public void dispose() {
		LOG.trace("AsyncBackupWorker.dispose() called");

	    final Timer aTimer = timer;
	    if (aTimer != null) {
	        aTimer.cancel();
	        aTimer.purge();
	        timer = null;
	    }
		writeBackTimerTask = null;

		synchronized (pendingDocData) {
			pendingDocData.clear();
		}
	}

	/**
	 * Retrieve all operations pending for the front entry in the pending document data
	 * queue.
	 * The implementation tries to collect all operations which are pending for one document
	 * restore id instance and provide a OperationDocData instance with all collected operations.
	 *
	 * @return OperationDocData with all collected operations and updated OSN
	 * @throws JSONException in case something went wrong with the JSON objects
	 */
	private static OperationDocData pollOperationDocDataFromFront(Queue<OperationDocData> queuedDocData) throws JSONException {
        OperationDocData currOpsDocData = queuedDocData.peek();

        if (null != currOpsDocData) {
            final DocRestoreID docRestoreID = currOpsDocData.getID();

            List<OperationDocData> aPendingDocOps = null;
            synchronized (queuedDocData) {
                aPendingDocOps = queuedDocData.stream()
                                              .filter(p -> { return docRestoreID.equals(p.getID()); })
                                              .collect(Collectors.toList());
                queuedDocData.removeAll(aPendingDocOps);
            }

            // compact collection of OperationDocData into one instance
            // containing all operations from the message chunks
            if (null != aPendingDocOps) {
                final List<MessageChunk> operationChunks = new ArrayList<>();
                aPendingDocOps.forEach(m -> operationChunks.add(m.getOperations()));

                final JSONObject msgChunkJSONObject = new JSONObject();
                final JSONObject opsDataObject = new JSONObject();
                final JSONArray combinedActions = new JSONArray();
                final JSONArray combinedOperations = collectOperations(operationChunks);

                opsDataObject.put(MessagePropertyKey.KEY_OPERATIONS, combinedOperations);
                combinedActions.put(opsDataObject);
                msgChunkJSONObject.put(MessagePropertyKey.KEY_ACTIONS, combinedActions);

                final int newOSN = OperationHelper.getFollowUpOSN(opsDataObject);
                final MessageChunk combinedMsgChunk = new MessageChunk(msgChunkJSONObject, null);
                return new OperationDocData(docRestoreID, combinedMsgChunk, newOSN);
            }
        }

        return currOpsDocData;
	}

	/**
	 * Extract all operations from a MessageChunk and append them to the operations
	 * array.
	 *
	 * @param collectedOperations will contain all operations from the list of message chunks
	 * @param messageChunks a list of message chunk
	 *
	 * @throws JSONException
	 */
	private static JSONArray collectOperations(List<MessageChunk> messageChunks) throws JSONException {
	    final JSONArray collectedOperations = new JSONArray();

	    if (null != messageChunks) {
	        final Iterator<MessageChunk> chunkIter = messageChunks.iterator();
	        while (chunkIter.hasNext()) {
	            final JSONObject actionsObject = chunkIter.next().getOperations();

	            if (null != actionsObject) {
	                final JSONArray actions = actionsObject.optJSONArray(MessagePropertyKey.KEY_ACTIONS);

	                if (null != actions) {
	                    for (int i = 0; i < actions.length(); i++) {
	                        final JSONObject opsObject = actions.getJSONObject(i);

	                        if (null != opsObject) {
	                            final JSONArray operations = opsObject.getJSONArray(MessagePropertyKey.KEY_OPERATIONS);

	                            if (null != operations) {
	                                JSONHelper.appendArray(collectedOperations, operations);
	                            }
	                        }
	                    }
	                }
	            }
	        }
	    }

        return collectedOperations;
	}
}
