/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General protected License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General protected License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General protected License
 *     for more details.
 *
 *     You should have received a copy of the GNU General protected License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.fileitem.impl;

import static org.apache.commons.lang.StringUtils.isNotEmpty;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import org.apache.commons.lang.StringUtils;
import com.openexchange.annotation.NonNull;
import com.openexchange.imageconverter.api.IFileItem;

/**
 * {@link FileItemProperties}
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @since v7.10.0
 */
class FileItemProperties {

    /**
     * Initializes a new {@link FileItemProperties}.
     */
    @SuppressWarnings("unused")
    private FileItemProperties() {
        // not to be used
    }

    /**
     * Initializes a new {@link FileItemProperties}.
     */
    protected FileItemProperties(@NonNull final Set<String> availableCustomKeys) {
        super();
        m_customKeys = availableCustomKeys;
    }

    /**
     * Initializes a new {@link FileItemProperties}.
     * @param createDateMillis
     */
    protected FileItemProperties(@NonNull final Set<String> availableCustomKeys, final long createDateMillis) {
        this(availableCustomKeys);

        m_createDateMillis = m_accessDateMillis = createDateMillis;

    }

    // - Implementation ------------------------------------------------

    /**
     * @return
     */
    protected boolean hasChanged() {
        return !m_propertiesChangedSet.isEmpty();
    }

    /**
     *
     */
    protected void reset() {
        m_propertiesChangedSet.clear();
        m_originalLength = m_length;
    }

    /**
     * Gets the m_createDateMillis
     *
     * @return The m_createDateMillis
     */
    protected long getCreateDateMillis() {
        return m_createDateMillis;
    }

    /**
     * Sets the m_accessDateMillis
     *
     * @param createDateMillis The m_createDateMillis to set
     */
    protected void setCreateDateMillis(final long createDateMillis ) {
        if (createDateMillis != m_createDateMillis) {
            m_createDateMillis = createDateMillis;
            m_propertiesChangedSet.add(FileItemDatabase.KEY_CREATEDATE);
        }
    }

    /**
     * @return
     */
    protected boolean hasCreateDateChanged() {
        return m_propertiesChangedSet.contains(FileItemDatabase.KEY_CREATEDATE);
    }

    /**
     * Gets the m_accessDateMillis
     *
     * @return The m_accessDateMillis
     */
    protected long getModificationDateMillis() {
        return m_accessDateMillis;
    }

    /**
     * Sets the m_accessDateMillis
     *
     * @param accessDateMillis The m_accessDateMillis to set
     */
    protected void setModificationDateMillis(final long accessDateMillis ) {
        if (accessDateMillis != m_accessDateMillis) {
            m_accessDateMillis = accessDateMillis;
            m_propertiesChangedSet.add(FileItemDatabase.KEY_MODIFICATIONDATE);
        }
    }

    /**
     * @return
     */
    protected boolean hasModificationDateChanged() {
        return m_propertiesChangedSet.contains(FileItemDatabase.KEY_MODIFICATIONDATE);
    }

    /**
     * Gets the m_length
     *
     * @return The m_length
     */
    protected long getLength() {
        return m_length;
    }

    /**
     * Gets the length difference from the last change of
     * length to the length at the time of resetting these
     * properties
     *
     * @return The length difference
     */
    protected long getLengthDifference() {
        return m_length - m_originalLength;
    }

    /**
     * Gets the m_length
     *
     * @return The m_length
     */
    protected void setLength(final long length) {
        if (length != m_length) {
            m_length = length;
            m_propertiesChangedSet.add(FileItemDatabase.KEY_LENGTH);
        }
    }

    /**
     * @return
     */
    protected boolean hasLengthChanged() {
        return m_propertiesChangedSet.contains(FileItemDatabase.KEY_LENGTH);
    }

    /**
     * @param customKey The customKey name
     * @return
     */
    protected String getCustomKeyValue(final String customKey) {
        final String customKeyValue = isNotEmpty(customKey) ? m_properties.get(customKey) : null;

        return (null != customKeyValue) ? customKeyValue : StringUtils.EMPTY;
    }

    /**
     * @param customKey The customKey name
     * @param value The customKey value
     */
    protected void setCustomKeyValue(final String customKey, final String value) {
        if (isNotEmpty(customKey) && m_customKeys.contains(customKey) && !getCustomKeyValue(customKey).equals(value)) {
            m_properties.put(customKey, value);
            m_propertiesChangedSet.add(customKey);
        }
    }

    /**
     * @param customKey The customKey name
     * @return
     */
    protected boolean hasCustomKeyValueChanged(final String customKey) {
        return ((isNotEmpty(customKey) && m_customKeys.contains(customKey)) ? m_propertiesChangedSet.contains(customKey) : false);
    }

    /**
     * Gets the available {@link Set} of customKeys at the time of opening an {@link IFileItem}
     *
     * @return The m_name
     */
    protected Set<String> getCustomKeys() {
        return m_customKeys;
    }

    /**
     * @return The Properties, containing all available key/value pair
     */
    protected Properties getProperties() {
        return implGetProperties(false, false, false);
    }

    /**
     * @return The Properties, containing all available key/value pair
     */
    protected Properties getDefaultProperties() {
        return implGetProperties(false, false, true);
    }

    /**
     * @return The Properties, containing all available customKey/value pair
     */
    protected Properties getCustomProperties() {
        return implGetProperties(false, true, false);
    }

    /**
     * @return The Properties, containing all changed key/value pair
     */
    protected Properties getChangedProperties() {
        return implGetProperties(true, false, false);
    }

    /**
     * @return The Properties, containing all changed customKey/value pair
     */
    protected Properties getChangedDefaultProperties() {
        return implGetProperties(true, false, true);
    }

    /**
     * @return The Properties, containing all changed customKey/value pair
     */
    protected Properties getChangedCustomProperties() {
        return implGetProperties(true, true, false);
    }

    // - Implementation --------------------------------------------------------

    /**
     * @return
     */
    private Properties implGetProperties(boolean changedOnly, boolean customKeysOnly, boolean defaultKeysOnly) {
        final boolean getAll = !changedOnly;
        final Properties properties = new Properties();

        if (!customKeysOnly || defaultKeysOnly) {
            if (getAll || hasCreateDateChanged()) {
                properties.put(FileItemDatabase.KEY_CREATEDATE, Long.valueOf(getCreateDateMillis()));
            }

            if (getAll || hasModificationDateChanged()) {
                properties.put(FileItemDatabase.KEY_MODIFICATIONDATE, Long.valueOf(getModificationDateMillis()));
            }

            if (getAll || hasLengthChanged()) {
                properties.put(FileItemDatabase.KEY_LENGTH, Long.valueOf(getLength()));
            }
        }

        if (!defaultKeysOnly || customKeysOnly) {
            // get changed customKeys and values
            for (final String curKey : m_customKeys) {
                if (getAll || m_propertiesChangedSet.contains(curKey)) {
                    final String curValue = m_properties.get(curKey);

                    if (isNotEmpty(curValue)) {
                        properties.put(curKey, curValue);
                    }
                }
            }
        }

        return properties;
    }

    // - Members ---------------------------------------------------------------

    protected long m_createDateMillis = 0;

    protected long m_accessDateMillis = 0;

    protected long m_length = 0;

    protected long m_originalLength = 0;

    protected Map<String, String> m_properties = new HashMap<>();

    protected Set<String> m_propertiesChangedSet = new HashSet<>();

    protected Set<String> m_customKeys = null;
}
