/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.admin.storage.mysqlStorage.user.attribute.changer.guipref;

import java.sql.Connection;
import java.util.Collection;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.openexchange.admin.rmi.dataobjects.User;
import com.openexchange.admin.rmi.exceptions.StorageException;
import com.openexchange.admin.storage.mysqlStorage.user.attribute.changer.AbstractAttributeChangers;
import com.openexchange.admin.storage.mysqlStorage.user.attribute.changer.EmptyAttribute;
import com.openexchange.exception.OXException;
import com.openexchange.groupware.settings.Setting;
import com.openexchange.groupware.settings.impl.ConfigTree;
import com.openexchange.groupware.settings.impl.SettingStorage;
import com.openexchange.tools.session.ServerSessionAdapter;

/**
 * {@link GuiPreferenceUserAttributeChangers}
 *
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 * @since v7.10.1
 */
public class GuiPreferenceUserAttributeChangers extends AbstractAttributeChangers {

    private static final Logger LOG = LoggerFactory.getLogger(GuiPreferenceUserAttributeChangers.class);

    /**
     * Initialises a new {@link GuiPreferenceUserAttributeChangers}.
     */
    public GuiPreferenceUserAttributeChangers() {
        super(EnumSet.noneOf(EmptyAttribute.class));
    }

    @Override
    public Set<String> change(User userData, int userId, int contextId, Connection connection, Collection<Runnable> pendingInvocations) throws StorageException {
        Map<String, String> guiPreferences = userData.getGuiPreferences();
        if (guiPreferences == null) {
            return EMPTY_SET;
        }

        // If administrator sets GUI configuration existing GUI configuration is overwritten
        Set<String> changedAttributes = new HashSet<>();
        Iterator<Entry<String, String>> iter = guiPreferences.entrySet().iterator();

        SettingStorage settingStorage = getSettingStorage(userId, contextId);
        while (iter.hasNext()) {
            Entry<String, String> entry = iter.next();
            String key = entry.getKey();
            String value = entry.getValue();
            if (null != key && null != value) {
                try {
                    Setting setting = ConfigTree.getInstance().getSettingByPath(key);
                    setting.setSingleValue(value);
                    settingStorage.save(connection, setting);
                    changedAttributes.add(key);
                } catch (OXException e) {
                    LOG.error("Problem while storing GUI preferences.", e);
                }
            }
        }
        return changedAttributes;
    }

    /**
     * Returns the {@link SettingStorage} instance for the specified user in the specified context
     *
     * @param userId The user identifier
     * @param contextId The context identifier
     * @return the {@link SettingStorage} instance for the specified user in the specified context
     * @throws StorageException if the {@link SettingStorage} instance cannot be initialised
     */
    private SettingStorage getSettingStorage(int userId, int contextId) throws StorageException {
        try {
            return SettingStorage.getInstance(ServerSessionAdapter.valueOf(userId, contextId));
        } catch (OXException e) {
            throw new StorageException("Cannot initialise the SettingStorage for user '" + userId + "' in context '" + contextId + "'", e);
        }
    }
}
