/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.contenttypes.calendar.impl;

import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.BitSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TimeZone;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.usm.api.contenttypes.calendar.AppointmentContentType;
import com.openexchange.usm.api.contenttypes.calendar.CalendarConstants;
import com.openexchange.usm.api.contenttypes.common.CommonConstants;
import com.openexchange.usm.api.contenttypes.common.ContentTypeField;
import com.openexchange.usm.api.contenttypes.common.DefaultContentTypes;
import com.openexchange.usm.api.contenttypes.folder.OXFolderContent;
import com.openexchange.usm.api.datatypes.DataTypes;
import com.openexchange.usm.api.exceptions.InternalUSMException;
import com.openexchange.usm.api.exceptions.OXCommunicationException;
import com.openexchange.usm.api.exceptions.USMException;
import com.openexchange.usm.api.ox.json.JSONResult;
import com.openexchange.usm.api.ox.json.JSONResultType;
import com.openexchange.usm.api.ox.json.OXJSONAccess;
import com.openexchange.usm.api.session.DataObject;
import com.openexchange.usm.api.session.DataObjectFilter;
import com.openexchange.usm.api.session.Folder;
import com.openexchange.usm.api.session.Session;
import com.openexchange.usm.contenttypes.util.AbstractTransferHandler;
import com.openexchange.usm.contenttypes.util.USMContentTypesUtilErrorCodes;
import com.openexchange.usm.datatypes.tasks.calendar.CommonCalendarTasksFieldNames;
import com.openexchange.usm.datatypes.tasks.calendar.UserParticipantObject;
import com.openexchange.usm.session.dataobject.DataObjectUtil;

/**
 * Transfer Handler for the module "Calendar"
 * 
 * @author ldo
 */
public class AppointmentContentTypeTransferHandler extends AbstractTransferHandler {

    private static final long ONE_DAY_IN_MILLISECONDS = 86400000L;

    private static final long MAXIMUM_EXPECTED_EXCEPTION_MOVEMENT = ONE_DAY_IN_MILLISECONDS * 365L;

    private final BitSet _DELETE_EXCEPTION_FIELDS = new BitSet();

    /**
     * Initializes a new {@link AppointmentContentTypeTransferHandler}.
     * 
     * @param appointmentContentType
     * @param ajaxAccess
     */
    public AppointmentContentTypeTransferHandler(AppointmentContentType appointmentContentType, OXJSONAccess ajaxAccess) {
        super(appointmentContentType, ajaxAccess);
        addToFieldBitMask(
            _DELETE_EXCEPTION_FIELDS,
            appointmentContentType.getFields(),
            CommonConstants.FIELD_ID,
            CommonConstants.FOLDER_ID,
            CommonCalendarTasksFieldNames.START_DATE,
            CommonCalendarTasksFieldNames.RECURRENCE_POSITION,
            CommonCalendarTasksFieldNames.RECURRENCE_DATE_POSITION,
            CalendarConstants.RECURRENCE_ID,
            CalendarConstants.TIMEZONE);
    }

    private void addToFieldBitMask(BitSet mask, ContentTypeField[] fields, String... requiredFields) {
        for (String f : requiredFields)
            addToFieldBitMask(mask, f, fields);
    }

    private void addToFieldBitMask(BitSet mask, String f, ContentTypeField[] fields) {
        for (int i = 0; i < fields.length; i++) {
            if (f.equals(fields[i].getFieldName())) {
                mask.set(i);
                return;
            }
        }
        throw new IllegalStateException("Unknown field <" + f + "> in ContentType <" + _contentType + ">");
    }

    @Override
    public OXFolderContent readFolderContent(Folder folder, int limit, DataObjectFilter filter) throws USMException {
        OXFolderContent result = super.readFolderContent(folder, limit, filter);
        resetRecurrenceTypeBasedFields(result.getObjects());
        return result;
    }

    @Override
    public DataObject[] readFolderContent(Folder folder, BitSet requestedFields) throws USMException {
        if (LOG.isDebugEnabled())
            LOG.debug(folder.getSession() + " Read all " + _contentType.getID() + " in folder " + folder.getID());
        Map<String, String> parameters = new HashMap<String, String>();
        parameters.put(CommonConstants.FOLDER, folder.getID());
        parameters.put(CommonConstants.COLUMNS, createColumnsParameter(requestedFields));
        parameters.put(CommonConstants.RECURRENCE_MASTER, Boolean.TRUE.toString());
        Session session = folder.getSession();
        addTimeLimitParameters(parameters, session);
        addObjectsLimitParameters(folder.getSession(), parameters);
        addShowPrivateParameter(parameters, session);
        DataObject[] result = performActionAndCheckRights(CommonConstants.ACTION_ALL, folder, requestedFields, parameters);

        resetRecurrenceTypeBasedFields(result);
        return result;
    }
    
    @Override
    protected void addTimeLimitParameters(Map<String, String> parameters, Session session) {
        parameters.put(CommonConstants.START, String.valueOf(session.getCalendarStartDate()));
        parameters.put(CommonConstants.END, String.valueOf(session.getCalendarEndDate()));
    }

    @Override
    protected void addExtraTimestampListParameters(Map<String, String> parameters, Session session) {
        parameters.put(CommonConstants.RECURRENCE_MASTER, Boolean.TRUE.toString());
        addShowPrivateParameter(parameters, session);
    }
    
    @Override
    protected void addShowPrivateParameter(Map<String, String> parameters, Session session) {
        if("JSON".equals(session.getProtocol())) {
            parameters.put("showPrivate", Boolean.TRUE.toString());
        }
    }

    @Override
    protected void finishDataObjectFromOptimizedRead(DataObject o) {
        int recurrenceType = DataObjectUtil.getIntValue(o, CommonCalendarTasksFieldNames.RECURRENCE_TYPE);
        if (recurrenceType != 0) {
            int seriesId = DataObjectUtil.getIntValue(o, CalendarConstants.RECURRENCE_ID);
            if (String.valueOf(seriesId).equals(o.getID())) {
                o.setFieldContent(CommonCalendarTasksFieldNames.RECURRENCE_POSITION, null);
                o.setFieldContent(CommonCalendarTasksFieldNames.RECURRENCE_DATE_POSITION, null);
            }
        }
    }

    public DataObject[] getAllAppointments(Session session, BitSet requestedFields) throws USMException {
        if (LOG.isDebugEnabled())
            LOG.debug(session + " Getting new appointments for date range " + session.getCalendarStartDate() + " - " + session.getCalendarEndDate());
        Map<String, String> parameters = new HashMap<String, String>();

        parameters.put(CommonConstants.COLUMNS, createColumnsParameter(requestedFields));
        addTimeLimitParameters(parameters, session);
        addExtraTimestampListParameters(parameters, session);
        addObjectsLimitParameters(session, parameters);

        JSONResult jsonResult = _ajaxAccess.doGet(getOXAjaxAccessPath(), CommonConstants.ACTION_ALL, session, parameters);
        DataObject[] result = buildResultArray(session, requestedFields, jsonResult);
        return result;
    }

    public DataObject[] getAllRecurrencesAsAppointments(Session session, BitSet requestedFields, DataObject seriesObject) throws USMException {
        return getAllRecurrencesAsAppointments(session, requestedFields, seriesObject, session.getCalendarStartDate(), session.getCalendarEndDate());
    }

    public DataObject[] getAllRecurrencesAsAppointments(Session session, BitSet requestedFields, DataObject seriesObject, long startDate, long endDate) throws USMException {
        if (LOG.isDebugEnabled())
            LOG.debug(session + " Getting all appointments as recurrences");
        Map<String, String> parameters = new HashMap<String, String>();

        parameters.put(CommonConstants.COLUMNS, createColumnsParameter(requestedFields));

        String folder = seriesObject != null ? seriesObject.getParentFolderID() : null;
        if (folder != null)
            parameters.put(CommonConstants.FOLDER, folder);
        parameters.put(CommonConstants.RECURRENCE_MASTER, Boolean.FALSE.toString());
        addObjectsLimitParameters(session, parameters);
        addShowPrivateParameter(parameters, session);
        List<DataObject> resultList = new ArrayList<DataObject>();
        
        long nParts = 1L;
        long currentPart = 0L;
        
        while (currentPart < nParts) {
            if (nParts == 1L) {
                addTimeLimitParameters(parameters, session);
            }
            else {
                long width = ( session.getCalendarEndDate() - session.getCalendarStartDate() ) / nParts;
                parameters.put(CommonConstants.START, String.valueOf(session.getCalendarStartDate() + currentPart*width));
                parameters.put(CommonConstants.END, String.valueOf(session.getCalendarStartDate()+ (currentPart+1L)*width));
            }
    
            try {
                JSONResult jsonResult = _ajaxAccess.doGet(getOXAjaxAccessPath(), CommonConstants.ACTION_ALL, session, parameters);
                JSONArray jArray = extractArrayResult(jsonResult);
                for (int i = 0; i < jArray.length(); i++) {
                    JSONArray columnsArray;
                    DataObject destination = null;
                    try {
                        columnsArray = jArray.getJSONArray(i);
                        destination = _contentType.newDataObject(session);
                        updateDataObjectFromJSONArray(destination, requestedFields, columnsArray);
                        readOptionalTimestamp(jsonResult.getJSONObject(), destination);
                        if (seriesObject != null) {
                            if (destination.getID().equals(seriesObject.getID()) || seriesObject.getID().equals(
                                String.valueOf(destination.getFieldContent(CalendarConstants.RECURRENCE_ID))))
                                resultList.add(destination);
                        } else {
                            resultList.add(destination);
                        }
                    } catch (JSONException e) {
                        throw generateException(
                            USMContentTypesUtilErrorCodes.DATA_NOT_PRESENT_NUMBER1,
                            destination,
                            "get new appointmets:data not present",
                            e);
                    }
                }
                currentPart++;
            }
            catch (OXCommunicationException oxe) {
                try {
                    if (nParts < 100L && "CAL-5072".equals(oxe.getOxErrorForJSONResponse().get("code"))) {
                        nParts++;
                        currentPart = 0L;
                        resultList.clear();
                    }
                    else {
                        // giving up
                        throw(oxe);
                    }
                } catch (JSONException e) {
                    throw(oxe);
                }
            }
        }
        DataObject[] result = resultList.toArray(new DataObject[resultList.size()]);
        resetRecurrenceTypeBasedFields(result);
        return result;
    }

    public DataObject[] getAllChangeExceptions(Session session, BitSet requestedFields, DataObject seriesObject) throws USMException {
        if (LOG.isDebugEnabled())
            LOG.debug(session + " Getting all change exceptions for series object: " + seriesObject.getID());
        Map<String, String> parameters = new HashMap<String, String>();
        parameters.put(CommonConstants.COLUMNS, createColumnsParameter(requestedFields));
        parameters.put(CommonConstants.FOLDER, seriesObject.getParentFolderID());
        parameters.put(CommonConstants.ID, seriesObject.getID());
        addShowPrivateParameter(parameters, session);

        JSONResult jsonResult = _ajaxAccess.doGet(getOXAjaxAccessPath(), CalendarConstants.GET_CHANGE_EXCEPTIONS_ACTION, session, parameters);
        DataObject[] result = buildResultArray(session, requestedFields, jsonResult);
        return result;
    }

    private DataObject[] buildResultArray(Session session, BitSet requestedFields, JSONResult jsonResult) throws OXCommunicationException, InternalUSMException {
        JSONArray jArray = extractArrayResult(jsonResult);
        List<DataObject> resultList = new ArrayList<DataObject>();
        for (int i = 0; i < jArray.length(); i++) {
            JSONArray columnsArray;
            DataObject destination = null;
            try {
                columnsArray = jArray.getJSONArray(i);
                destination = _contentType.newDataObject(session);
                updateDataObjectFromJSONArray(destination, requestedFields, columnsArray);
                resultList.add(destination);
            } catch (JSONException e) {
                throw generateException(
                    USMContentTypesUtilErrorCodes.DATA_NOT_PRESENT_NUMBER1,
                    destination,
                    "get new appointments:data not present",
                    e);
            }
        }
        DataObject[] result = resultList.toArray(new DataObject[resultList.size()]);
        resetRecurrenceTypeBasedFields(result);
        return result;
    }

    @Override
    public void writeDeletedDataObject(DataObject object) throws USMException {
        if (Boolean.TRUE.equals(object.getSession().getCustomProperty(AppointmentContentType.DECLINE_INSTEAD_OF_DELETE))) {
            if (shouldDeclineInsteadOfDelete(object)) {
                // If yes, decline appointment instead and return
                confirm(object, 2);
                return;
            }
        }
        String recurrencePosition = String.valueOf(object.getFieldContent(CommonCalendarTasksFieldNames.RECURRENCE_POSITION));
        if (recurrencePosition == null || recurrencePosition.equals("0") || recurrencePosition.equals("null"))
            writeExtendedDeleteDataObject(object, CommonConstants.FOLDER, object.getParentFolderID());
        else {
            if (LOG.isTraceEnabled()) {
                LOG.trace("write deleted appointment to server: " + object.toString());
            }
           
            writeExtendedDeleteDataObject(
                object,
                CommonCalendarTasksFieldNames.RECURRENCE_POSITION,
                recurrencePosition,
                CommonConstants.FOLDER,
                object.getParentFolderID());
        }
    }

    private boolean shouldDeclineInsteadOfDelete(DataObject object) {
        if (object.getParentFolderOwnerID() == 0) {
            // If not in shared folder
            int userID = object.getSession().getUserIdentifier();
            Object o = object.getFieldContent(CommonConstants.CREATED_BY);
            int organizer = 0;
            if (o != null) {
                try {
                    organizer = Integer.parseInt(String.valueOf(o));
                } catch (NumberFormatException ignored) {
                    //ignored
                }
            }
            if (userID != organizer) {
                // And if not organizer
                o = object.getFieldContent(CommonCalendarTasksFieldNames.USERS);
                if (o instanceof Object[]) {
                    Object[] list = (Object[]) o;
                    for (Object u : list) {
                        if (u instanceof UserParticipantObject) {
                            UserParticipantObject participant = (UserParticipantObject) u;
                            if (participant.getId() == userID)
                                // And if participant -> decline instead of delete
                                return true;
                        }
                    }
                }
            }
        }
        return false;
    }

    @Override
    public void writeNewDataObject(DataObject object) throws USMException {
        // don't call the standard write utility method because conflicts array can be returned from server
        if (LOG.isDebugEnabled())
            LOG.debug(object.getSession() + " New " + DefaultContentTypes.CALENDAR_ID);
        checkTypeToWrite(object, DefaultContentTypes.CALENDAR_ID);
        Map<String, String> parameters = new HashMap<String, String>();
        JSONObject requestBody = createRequestBodyWithIgnoreConflicts(object, false);
        // printTimeValues("New", requestBody);
        int retryCount = 50;
        String previousInvalidChars = "";
        while (retryCount-- > 0) {
            JSONResult result = _ajaxAccess.doPut(
                CalendarConstants.CALENDAR_PATH,
                CommonConstants.ACTION_NEW,
                object.getSession(),
                parameters,
                requestBody);
            JSONArray errorParams = checkInvalidCharacters(result.getJSONObject());
            if (errorParams == null)
                checkResult(result, JSONResultType.JSONObject);

            JSONObject o = result.getJSONObject();
            readOptionalTimestamp(o, object);
            try {
                if (errorParams != null) {
                    if (retryCount == 0) {
                        LOG.warn(o.optString(CommonConstants.ERROR));
                    }
                    String contentKey = errorParams.getString(0);
                    String invalidChars = errorParams.getString(1);
                    if (invalidChars == null || invalidChars.equals(previousInvalidChars)) {
                        throw new OXCommunicationException(
                            USMContentTypesCalendarErrorCodes.OTHER_ERROR,
                            "Appointment could not be created because of invalid content",
                            o.toString());
                    }
                    previousInvalidChars = invalidChars;
                    
                    byte[] bytes = invalidChars.getBytes(Charset.forName("UTF-8"));
                    String HEX_DIGITS = "0123456789ABCDEF";
                    StringBuilder sb = new StringBuilder(bytes.length + 3);
                    for (byte bb: bytes) {
                        int b = bb & 0xFF;
                        sb.append(HEX_DIGITS.charAt(b >> 4)).append(HEX_DIGITS.charAt(b % 16)).append(' ');
                    }
                    LOG.debug("invalid characters as bytes: " + sb.toString());
                    
                    if (!requestBody.has(contentKey)) {
                        contentKey = contentKey.toLowerCase();
                    }
                    if (!requestBody.has(contentKey)) {
                        throw new OXCommunicationException(
                            USMContentTypesCalendarErrorCodes.OTHER_ERROR,
                            "Appointment could not be created because of invalid content",
                            o.toString());
                    }
                    String replacedContent = requestBody.getString(contentKey);

                    int offset = 0, strLen = invalidChars.length();
                    StringBuilder sb3 = new StringBuilder(strLen + 3);
                    while (offset < strLen) {
                      int curChar = invalidChars.codePointAt(offset);
                      int charLen = Character.charCount(curChar);
                      sb3.append(invalidChars.substring(offset, offset+charLen));
                      sb3.append(": ");
                      while (curChar > 0) {
                          int b = curChar & 0xFF;
                          sb3.append(HEX_DIGITS.charAt(b >> 4)).append(HEX_DIGITS.charAt(b % 16)).append(' ');
                          curChar = curChar >> 8;
                      }
                      sb3.append(";  ");
                      replacedContent = replacedContent.replaceAll(invalidChars.substring(offset, offset+charLen), "");
                      offset += charLen;
                    }
                    LOG.debug("codePoints: " +  sb3.toString());
                    requestBody.put(contentKey, replacedContent);
                    object.setFieldContent(contentKey, replacedContent);
                } else {
                    JSONObject resultObject = o.getJSONObject(CommonConstants.RESULT_DATA);
                    if (resultObject.has(CommonConstants.RESULT_ID)) {
                        object.setID(resultObject.getString(CommonConstants.RESULT_ID));
                        retryCount = 0;
                    } else if (resultObject.has(CalendarConstants.CONFLICTS)) {
                        JSONArray arr = resultObject.getJSONArray(CalendarConstants.CONFLICTS);
                        throw new OXCommunicationException(
                            USMContentTypesCalendarErrorCodes.CONFLICTS,
                            "Appointment could not be created because of conflicts",
                            arr.getJSONObject(0));
                    } else {
                        throw new OXCommunicationException(
                            USMContentTypesCalendarErrorCodes.OTHER_ERROR,
                            "Appointment could not be created because of ox communication error",
                            resultObject.toString());
                    }
                }
            } catch (JSONException e) {
                throw new OXCommunicationException(
                    USMContentTypesUtilErrorCodes.OX_RESULT_ERROR_NUMBER1,
                    "Invalid Result on new " + DefaultContentTypes.CALENDAR_ID,
                    e,
                    result.toString());
            }
        }
        writeDeletedExceptions(object);
    }

    private void writeDeletedExceptions(DataObject seriesObject) throws USMException {
        Object[] deleteExceptions = (Object[]) seriesObject.getFieldContent(CommonCalendarTasksFieldNames.DELETE_EXCEPTIONS);
        if (deleteExceptions == null || deleteExceptions.length == 0)
            // || !seriesObject.isFieldModified(CommonCalendarTasksFieldNames.DELETE_EXCEPTIONS))
            return;
        long intervalStart = Long.MAX_VALUE;
        long intervalEnd = 0L;
        Set<Number> deleteExceptionSet = new HashSet<Number>();
        for (Object o : deleteExceptions) {
            if (o instanceof Number) {
                Number n = (Number) o;
                deleteExceptionSet.add(n);
                long l = n.longValue();
                intervalStart = Math.min(intervalStart, l - MAXIMUM_EXPECTED_EXCEPTION_MOVEMENT);
                intervalEnd = Math.max(intervalEnd, l + MAXIMUM_EXPECTED_EXCEPTION_MOVEMENT);
            }
        }
        if (deleteExceptionSet.isEmpty())
            return;
        DataObject[] allOcurrences = getAllRecurrencesAsAppointments(
            seriesObject.getSession(),
            _DELETE_EXCEPTION_FIELDS,
            seriesObject,
            intervalStart,
            intervalEnd);
        long timestamp = 0L;
        for (int j = 0; j < allOcurrences.length; j++) {
            DataObject occ = allOcurrences[j];
            if (occ != null) {
                if (shouldDeleteOccurrence(occ, deleteExceptionSet)) {
                    if (timestamp != 0L)
                        occ.setTimestamp(timestamp);
                    writeDeletedDataObject(occ);
                    timestamp = occ.getTimestamp();
                }
            }
        }
        if (timestamp != 0L)
            seriesObject.setTimestamp(timestamp);
    }

    private boolean shouldDeleteOccurrence(DataObject occ, Set<Number> deleteExceptionSet) {
        Object o = occ.getFieldContent(CommonCalendarTasksFieldNames.RECURRENCE_DATE_POSITION);
        if (o != null) // the DataObject is a change exception, and therefore recurrence_date_position is set to the original date
            return deleteExceptionSet.contains(o);
        o = occ.getFieldContent(CommonCalendarTasksFieldNames.START_DATE);
        if (!(o instanceof Number))
            return false;
        long start = ((Number) o).longValue();
        long offset = 0L;
        // modify start_date so that it's 00:00 UTC
        o = occ.getFieldContent(CalendarConstants.TIMEZONE);
        if (o instanceof String)
            offset = TimeZone.getTimeZone((String) o).getOffset(start);
        start = (start + offset) / ONE_DAY_IN_MILLISECONDS;
        start *= ONE_DAY_IN_MILLISECONDS;
        return deleteExceptionSet.contains(start);
    }

    private JSONObject createRequestBodyWithIgnoreConflicts(DataObject object, boolean isUpdate) throws InternalUSMException {
        JSONObject requestBody = new JSONObject();
        try {
            requestBody.put(CalendarConstants.IGNORE_CONFLICTS, Boolean.TRUE);
        } catch (JSONException e) {
            throwInternalUSMException(
                USMContentTypesUtilErrorCodes.ERROR_WRITING_DATA_NUMBER7,
                "writing",
                object,
                CalendarConstants.IGNORE_CONFLICTS,
                DataTypes.BOOLEAN,
                e);
        }
        ContentTypeField[] fields = object.getContentType().getFields();
        boolean isException = isSeriesException(object);
        boolean isNewException = isException && isNewException(object);
        for (int i = 0; i < fields.length; i++) {
            ContentTypeField field = fields[i];
            if (isUpdate && field.getFieldName().equals(CommonCalendarTasksFieldNames.ALARM)) { // special workaround for "alarm", see bug
                                                                                                // #17327
                if (object.isFieldModified(CommonCalendarTasksFieldNames.ALARM) && object.getFieldContent(CommonCalendarTasksFieldNames.ALARM) == null) {
                    try {
                        requestBody.put(CommonCalendarTasksFieldNames.ALARM, -1);
                        continue;
                    } catch (JSONException e) {
                        throwInternalUSMException(
                            USMContentTypesUtilErrorCodes.ERROR_WRITING_DATA_NUMBER6,
                            "writing",
                            object,
                            CommonCalendarTasksFieldNames.ALARM,
                            DataTypes.NUMBER,
                            e);
                    }
                }
            }
            boolean isNoSpecialField = isNoSpecialSeriesField(field, !isNewException);
            if (isException) {
                if (isNoSpecialField)
                    addFieldToRequestBody(object, isUpdate, !isUpdate, requestBody, field, i);
            } else {
                addFieldToRequestBody(object, isUpdate && isNoSpecialField, !isUpdate, requestBody, field, i);
            }
        }
        requestBody.remove(CalendarConstants.RECURRENCE_ID);
        // XXX Workaround for Bug #14952 (start)
        if (requestBody.optBoolean(CalendarConstants.FULL_TIME) && !requestBody.has(CommonCalendarTasksFieldNames.START_DATE)) {
            for (int i = 0; i < fields.length; i++) {
                if (fields[i].getFieldName().equals(CommonCalendarTasksFieldNames.START_DATE)) {
                    addFieldToRequestBody(object, false, true, requestBody, fields[i], i);
                    break;
                }
            }
        }
        // XXX Workaround for Bug #14952 (end)
        if (!isException) {
            try {
                requestBody.remove(CommonCalendarTasksFieldNames.RECURRENCE_POSITION);
                requestBody.remove(CommonCalendarTasksFieldNames.RECURRENCE_DATE_POSITION);
                requestBody.remove(CommonCalendarTasksFieldNames.CHANGE_EXCEPTIONS);
                requestBody.remove(CommonCalendarTasksFieldNames.DELETE_EXCEPTIONS);
                Number recType = (Number) object.getFieldContent(CommonCalendarTasksFieldNames.RECURRENCE_TYPE);
                if (recType != null && recType.intValue() > 0) {
                    if (object.isFieldModified(CommonCalendarTasksFieldNames.UNTIL)) {
                        // this is a special case for changing the until of the series. (iPhone - for all future appointments)
                        // All relevant recurrence fields have to be sent
                        requestBody.put(CommonCalendarTasksFieldNames.RECURRENCE_TYPE, recType);
                        requestBody.put(
                            CommonCalendarTasksFieldNames.INTERVAL,
                            object.getFieldContent(CommonCalendarTasksFieldNames.INTERVAL));
                        requestBody.put(CommonCalendarTasksFieldNames.UNTIL, object.getFieldContent(CommonCalendarTasksFieldNames.UNTIL));
                        Object days = object.getFieldContent(CommonCalendarTasksFieldNames.DAYS);
                        if (days instanceof Number && ((Number) days).intValue() != 127)
                            requestBody.put(CommonCalendarTasksFieldNames.DAYS, days);
                        else
                            requestBody.remove(CommonCalendarTasksFieldNames.DAYS);
                    }
                } else {
                    if (object.isFieldModified(CommonCalendarTasksFieldNames.RECURRENCE_TYPE))
                        requestBody.put(CommonCalendarTasksFieldNames.RECURRENCE_TYPE, recType);
                    else
                        requestBody.remove(CommonCalendarTasksFieldNames.RECURRENCE_TYPE);
                    requestBody.remove(CommonCalendarTasksFieldNames.INTERVAL);
                    requestBody.remove(CommonCalendarTasksFieldNames.UNTIL);
                    requestBody.remove(CommonCalendarTasksFieldNames.DAYS);
                    requestBody.remove(CommonCalendarTasksFieldNames.OCCURENCES);
                }
            } catch (JSONException e) {
                throwInternalUSMException(
                    USMContentTypesUtilErrorCodes.ERROR_WRITING_DATA_NUMBER1,
                    "writing",
                    object,
                    CalendarConstants.IGNORE_CONFLICTS,
                    DataTypes.BOOLEAN,
                    e);
            }
        }
        return requestBody;
    }

    private boolean isNoSpecialSeriesField(ContentTypeField field, boolean isUpdate) {
        int id = field.getFieldID();
        return id < (isUpdate ? 206 : 209) || (id > 216 && id != 222);
    }

    private boolean isSeriesException(DataObject object) {
        Object o = object.getFieldContent(CommonCalendarTasksFieldNames.RECURRENCE_POSITION);
        return (o instanceof Number) && ((Number) o).intValue() != 0;
    }

    private boolean isNewException(DataObject object) {
        Object o = object.getFieldContent(CalendarConstants.RECURRENCE_ID);
        return String.valueOf(o).equals(object.getID());
    }

    @Override
    public void writeUpdatedDataObject(DataObject object, long timestamp) throws USMException {
        if (LOG.isDebugEnabled())
            LOG.debug(object.getSession() + " Updating " + _contentType.getID());
        if (LOG.isTraceEnabled())
            LOG.trace("Updating data object on server: " + object.toString());
        checkTypeToWrite(object, _contentType.getID());

        Map<String, String> parameters = new HashMap<String, String>();
        parameters.put(CommonConstants.ID, object.getID());
        parameters.put(CommonConstants.FOLDER, object.getOriginalParentFolderID());
        parameters.put(CommonConstants.TIMESTAMP, String.valueOf(timestamp));
        JSONObject requestBody = createRequestBodyWithIgnoreConflicts(object, true);
        try {
            if (requestBody.has(CalendarConstants.SEQUENCE))
                requestBody.put("ignoreOutdatedSequence", "true");
        } catch (JSONException e1) {
            throw new USMException(USMContentTypesUtilErrorCodes.OX_RESULT_ERROR_NUMBER14, "Error on creating request body", e1);
        }

        if (LOG.isTraceEnabled())
            try {
                LOG.trace("Updating data object (json data)  " + requestBody.toString(1, 4));
            } catch (JSONException e1) {
                // ignore
            }
        // printTimeValues("Change", requestBody);
        JSONResult result = _ajaxAccess.doPut(
            getOXAjaxAccessPath(),
            CommonConstants.ACTION_UPDATE,
            object.getSession(),
            parameters,
            requestBody);

        checkResult(result, JSONResultType.JSONObject);
        // XXX Shouldn't we use the timestamp returned from the OX server ?
        object.setTimestamp(timestamp);
        JSONObject o = result.getJSONObject();
        updateIdForNewException(object, result);
        try {
            JSONObject resultObject = o.getJSONObject(CommonConstants.RESULT_DATA);
            if (resultObject.has(CalendarConstants.CONFLICTS)) {
                JSONArray arr = resultObject.getJSONArray(CalendarConstants.CONFLICTS);
                throw new OXCommunicationException(
                    USMContentTypesCalendarErrorCodes.CONFLICTS,
                    "Appointment could not be changed because of conflicts",
                    arr.getJSONObject(0));
            }
        } catch (JSONException e) {
            throw new OXCommunicationException(
                USMContentTypesUtilErrorCodes.OX_RESULT_ERROR_NUMBER14,
                "Invalid Result on update " + DefaultContentTypes.CALENDAR_ID,
                e,
                result.toString());
        } finally {
            setAppOrTaskConfirmations(object);
            writeDeletedExceptions(object);
        }
    }

    @Override
    public DataObject[] readUpdatedFolderContent(Folder folder, BitSet requestedFields, long timestamp) throws USMException {
        Session session = folder.getSession();
        DataObject[] result = readStandardUpdatedFolderContent(
            folder,
            timestamp,
            requestedFields,
            CommonConstants.ACTION_UPDATES,
            CommonConstants.FOLDER,
            null,
            null,
            String.valueOf(session.getCalendarStartDate()),
            String.valueOf(session.getCalendarEndDate()));
        resetRecurrenceTypeBasedFields(result);
        return result;
    }

    @Override
    protected String getOXAjaxAccessPath() {
        return CalendarConstants.CALENDAR_PATH;
    }

    public String resolveUID(Session session, String uid) throws USMException {
        if (LOG.isDebugEnabled())
            LOG.debug(session + " Resolving uid " + uid);
        Map<String, String> parameters = new HashMap<String, String>();
        parameters.put(CommonConstants.UID, uid);

        JSONResult result = _ajaxAccess.doGet(getOXAjaxAccessPath(), CommonConstants.ACTION_RESOLVEUID, session, parameters);

        checkResult(result, JSONResultType.JSONObject);
        JSONObject o = result.getJSONObject();
        try {
            JSONObject resultObject = o.getJSONObject(CommonConstants.RESULT_DATA);
            if (resultObject.has(CommonConstants.ID)) {
                String id = resultObject.getString(CommonConstants.ID);
                return id;
            }
        } catch (JSONException e1) {
            throw new OXCommunicationException(
                USMContentTypesUtilErrorCodes.ERROR_READING_DATA_NUMBER7,
                "Invalid id",
                e1,
                result.toString());
        }
        return null;
    }
    
    private void updateIdForNewException(DataObject object, JSONResult result) {
        try {
            JSONObject resultObject = result.getJSONObject().getJSONObject(CommonConstants.RESULT_DATA);
            if (resultObject.has(CommonConstants.ID)) {
                String id = resultObject.getString(CommonConstants.ID);
                if (id.length()>0 && !object.getID().equals(id)) {
                    object.setID(id);
                }
            }
        } catch (JSONException e) {
            // ignore any error
        }
    }

}
