/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.api.datatypes;

import java.io.Serializable;
import java.util.Arrays;
import java.util.UUID;

/**
 * Stores all information on attachments to an OX PIM object, including the last modified timestamp. This class is serializable, only the
 * relevant sync information (timestamp, ox ids and uuids) will be serialized.
 * 
 * @author afe
 */
public class PIMAttachments implements Serializable {

    /**
	 * 
	 */
    private static final long serialVersionUID = 1L;

    private final long _timestamp;

    private final PIMAttachment[] _attachments;

    // Special flag that can be set to force this object to differ from other PIMAttachments (to enforce that a SyncState is written, since
    // normal equals only checks for the timestamp, not for the contained PIMAttachment objects)
    private transient boolean _forceModified;

    public PIMAttachments(long timestamp, PIMAttachment... attachments) {
        this(timestamp, false, attachments);
    }

    public PIMAttachments(long timestamp, boolean forceModified, PIMAttachment... attachments) {
        _timestamp = timestamp;
        _attachments = attachments;
        _forceModified = forceModified;
    }

    public long getTimestamp() {
        return _timestamp;
    }

    public int size() {
        return (_attachments == null) ? 0 : _attachments.length;
    }

    public PIMAttachment[] getAttachments() {
        return _attachments;
    }

    public boolean containsAttachment(UUID uuid) {
        if (uuid != null) {
            for (PIMAttachment a : _attachments) {
                if (uuid.equals(a.getUUID()))
                    return true;
            }
        }
        return false;
    }

    /**
     * @return true if this PIMAttachments is set to be forced as not equals to other PIMAttachments with the same timestamp (normal equals
     *         does not check the contained PIMAttachment objects, only the timestamp)
     */
    public boolean isForceModified() {
        return _forceModified;
    }

    public void setForceModified(boolean forceModified) {
        _forceModified = forceModified;
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + (int) (_timestamp ^ (_timestamp >>> 32));
        return result;
    }

    /**
     * Returns true if obj is a PIMAttachments object that has the same timestamp as this one and if neither of the 2 PIMAttachments objects
     * has the forceModified flag set.
     */
    @Override
    public boolean equals(Object obj) {
        if (this == obj)
            return true;
        if (obj == null)
            return false;
        if (!(obj instanceof PIMAttachments))
            return false;
        PIMAttachments other = (PIMAttachments) obj;
        if (_timestamp != other._timestamp)
            return false;
        if (_forceModified || other._forceModified)
            return false;
        return true;
    }

    @Override
    public String toString() {
        return "PIMAttachments(timestamp=" + _timestamp + ',' + _forceModified + ',' + Arrays.toString(_attachments) + ')';
    }
}
