/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.usm.ox_json.impl;

import static com.openexchange.usm.util.OXJSONUtil.DATA_FIELD;
import static com.openexchange.usm.util.OXJSONUtil.ERROR_FIELD;
import static com.openexchange.usm.util.OXJSONUtil.isOXError;
import java.io.BufferedReader;
import java.io.IOException;
import org.json.JSONException;
import org.json.JSONObject;
import org.json.JSONValue;
import com.openexchange.java.Streams;
import com.openexchange.usm.api.ox.json.JSONResult;
import com.openexchange.usm.api.ox.json.JSONResultType;
import com.openexchange.usm.util.OXJSONUtil;

public class OXJSONSupport {


    public static JSONResult parseResult(BufferedReader contentReader, JSONResultType expectedResultType) throws IOException {
        BufferedReader reader = contentReader;
        try {
            reader.mark(8192);
            try {
                JSONValue jValue = JSONObject.parse(reader);
                if (jValue.isArray()) {
                    JSONArrayResult result = new JSONArrayResult(jValue.toArray());
                    if (isValidResultType(expectedResultType, JSONResultType.JSONArray)) {
                        return result;
                    }
                    JSONObject errorDetails = new JSONObject(4);
                    try {
                        errorDetails.put(DATA_FIELD, result.getJSONArray());
                        errorDetails.put(ERROR_FIELD, "Invalid array response from server, expected " + expectedResultType);
                    } catch (JSONException ignored) {
                        //should not happen - object creation
                    }
                    return new JSONErrorResult(errorDetails);
                }

                // Otherwise a JSON object...
                JSONObject object = jValue.toObject();
                JSONResult jsonResult = generateJSONResultForJSONObject(object, expectedResultType);
                Streams.close(reader);
                reader = null;
                return jsonResult;
            } catch (JSONException e) {
                // Neither a JSON object or a JSON array
            }
            reader.reset();

            StringBuilder sb = new StringBuilder(2048);
            char[] cbuf = new char[8192];
            for (int read; (read = reader.read(cbuf, 0, cbuf.length)) > 0;) {
                sb.append(cbuf, 0, read);
            }
            Streams.close(reader);
            reader = null;

            if (isValidResultType(expectedResultType, JSONResultType.String)) {
                return new JSONStringResult(sb.toString());
            }

            JSONObject errorDetails = new JSONObject(4);
            try {
                errorDetails.put(DATA_FIELD, sb.toString());
                errorDetails.put(ERROR_FIELD, "Invalid string response from server, expected " + expectedResultType);
            } catch (JSONException ignored) {
              //Should not happen - object creation
            }
            return new JSONErrorResult(errorDetails);
        } finally {
            Streams.close(reader);
        }
    }

    /**
     * if JSONObject has a field 'error' and the category of this error is not "13" then return JSONErrorResult if a field 'data' is
     * available and is not empty then observe the content of this field; if starts with '[' or '(' return JSONArrayResult if starts with
     * '{' return JSONObjectResult
     *
     * @param object
     * @return
     */
    public static JSONResult generateJSONResultForJSONObject(JSONObject object, JSONResultType expectedResultType) {
        if (isOXError(object)) {
            return new JSONErrorResult(object);
        }
        if (isValidResultType(expectedResultType, JSONResultType.JSONObject)) {
            return new JSONObjectResult(object);
        }
        JSONObject errorDetails = new JSONObject();
        try {
            errorDetails.put(DATA_FIELD, object);
            errorDetails.put(ERROR_FIELD, "Invalid object response from server, expected " + expectedResultType);
        } catch (JSONException ignored) {
            // can not happen
        }
        return new JSONErrorResult(errorDetails);
    }



    private static boolean isValidResultType(JSONResultType expected, JSONResultType actual) {
        return expected == null || expected == actual;
    }


    public static boolean shouldPerformNewLogin(JSONResult result) {
        return result.getResultType() == JSONResultType.Error &&  OXJSONUtil.isReloginErrorCode(result.getJSONObject());
    }
}
