/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.tools.config;

import org.apache.commons.logging.Log;
import org.json.JSONObject;

import com.openexchange.config.ConfigurationService;
import com.openexchange.exception.OXException;
import com.openexchange.jslob.JSlob;
import com.openexchange.jslob.JSlobService;
import com.openexchange.office.tools.osgi.ServiceLookupRegistry;
import com.openexchange.server.ServiceLookup;
import com.openexchange.session.Session;


/**
 * A helper class to ease the use of properties settings value
 * retrieval.
 *
 * {@link ConfigurationHelper}
 *
 * @author <a href="mailto:carsten.driesner@open-xchange.com">Carsten Driesner</a>
 */
public class ConfigurationHelper {

    /**
     * Retrieves a office.properties boolean value using the cascade
     * configuration that supports to overwrite settings for a user.
     * A default value can be provided which is returned if the value
     * cannot be retrieved.
     *
     * @param serviceLookup
     *  The service lookup instance to be used for the retrieval.
     *
     * @param session
     *  The session of the user that defines the context for the cascade
     *  configuration value retrieval.
     *
     * @param settingPath
     *  The path including the setting name to be retrieved from the
     *  configuration, e.g //module
     *
     * @param defaultValue
     *  The default value to be used if the value cannot be retrieved.
     *
     * @return
     *  The value of the setting or defaultValue if the real value cannot
     *  be retrieved from the configuration.
     */
    static public Boolean getBooleanOfficeConfigurationValue(ServiceLookup serviceLookupDEPRECATED, Session session, String settingPath, Boolean defaultValue) {
        String strValue = null;

        strValue = ConfigurationHelper.getOfficeConfigurationValue(null, session, settingPath, null);
        if (null == strValue) {
            return defaultValue;
        } else {
            return Boolean.parseBoolean(strValue);
        }
    }

    /**
     * Retrieves a office.properties integer value using the cascade
     * configuration that supports to overwrite settings for a user.
     * A default value can be provided which is returned if the value
     * cannot be retrieved.
     *
     * @param serviceLookup
     *  The service lookup instance to be used for the retrieval.
     *
     * @param session
     *  The session of the user that defines the context for the cascade
     *  configuration value retrieval.
     *
     * @param settingPath
     *  The path including the setting name to be retrieved from the
     *  configuration, e.g //module
     *
     * @param defaultValue
     *  The default value to be used if the value cannot be retrieved.
     *
     * @return
     *  The value of the setting or defaultValue if the real value cannot
     *  be retrieved from the configuration.
     */
    static public Integer getIntegerOfficeConfigurationValue(ServiceLookup serviceLookupDEPRECATED, Session session, String settingPath, Integer defaultValue) {
        String strValue = null;

        strValue = ConfigurationHelper.getOfficeConfigurationValue(null, session, settingPath, null);
        if (null == strValue) {
            return defaultValue;
        }
        try {
            return Integer.parseInt(strValue);
    	}
    	catch(NumberFormatException e) {
    		return defaultValue;
        }
    }

    /**
     * Retrieves a office.properties string value using the cascade
     * configuration that supports to overwrite settings for a user.
     * A default value can be provided which is returned if the value
     * cannot be retrieved.
     *
     * @param serviceLookup
     *  The service lookup instance to be used for the retrieval.
     *
     * @param session
     *  The session of the user that defines the context for the cascade
     *  configuration value retrieval.
     *
     * @param settingPath
     *  The path including the setting name to be retrieved from the
     *  configuration, e.g //module
     *
     * @param defaultValue
     *  The default value to be used if the value cannot be retrieved.
     *
     * @return
     *  The string value of the setting or defaultValue if the real value
     *  cannot be retrieved from the configuration.
     */
    static public String getStringOfficeConfigurationValue(ServiceLookup serviceLookupDEPRECATED, Session session, String settingPath, String defaultValue) {
        return getOfficeConfigurationValue(null, session, settingPath, defaultValue);
    }

    /**
     * Retrieves a office.properties value using the cascade configuration
     * that supports to overwrite settings for a user. A default value can
     * be provided which is returned if the value cannot be retrieved.
     *
     * @param serviceLookup
     *  The service lookup instance to be used for the retrieval.
     *
     * @param session
     *  The session of the user that defines the context for the cascade
     *  configuration value retrieval.
     *
     * @param settingPath
     *  The path including the setting name to be retrieved from the
     *  configuration, e.g //module/<setting-name>
     *
     * @param defaultValue
     *  The default value to be used if the value cannot be retrieved.
     *
     * @return
     *  The value of the setting or defaultValue if the real value cannot
     *  be retrieved from the configuration.
     */
     static public String getOfficeConfigurationValue(ServiceLookup serviceLookupDEPRECATED, Session session, String settingPath, String defaultValue) {
         String result = defaultValue;

         result = ConfigurationHelper.getCascadeConfigurationValue((ServiceLookup)null, session, "io.ox/office", settingPath);
         if (null == result) {
             result = defaultValue;
         }

        return result;
     }

    /**
     * Retrieves a properties value using the cascade configuration
     * that supports to overwrite settings for a user.
     *
     * @param jslobService
     *  The jslob service lookup instance to be used for the retrieval.
     *
     * @param session
     *  The session of the user that defines the context for the cascade
     *  configuration value retrieval. If the session is null so the user
     *  is unknown, the method uses the configuration service as fallback.
     *
     * @param configuration
     *  The name of the configuration file to used for the retrieval,
     *  e.g. io.ox/office
     *
     * @param path
     *  The path within the configuration file to retrieve the setting
     *  value, e.g. //module/maxTableCells
     *
     * @return
     *  The value of the setting or null if the setting is not available.
     */
     static public String getCascadeConfigurationValue(ServiceLookup serviceLookupDEPRECATED, Session session, String configuration, String path) {
        String result = null;

        if (null != configuration && null != path) {

            final JSlobService jslobService = ServiceLookupRegistry.get().getService(JSlobService.class);
            if ((null != session) && (null != jslobService)) {
                result = getCascadeConfigurationValue(jslobService, session, configuration, path);
            } else {
                String configurationFullPath = configuration + path;
                final ConfigurationService configService = ServiceLookupRegistry.get().getService(com.openexchange.config.ConfigurationService.class);
                if (null != configService) {
                    result = configService.getProperty(configurationFullPath, (String)null);
                }
            }
        }

        return result;
    }

    /**
     * Retrieves a properties value using the cascade configuration
     * that supports to overwrite settings for a user.
     *
     * @param jslobService
     *  The jslob service lookup instance to be used for the retrieval.
     *
     * @param session
     *  The session of the user that defines the context for the cascade
     *  configuration value retrieval.
     *
     * @param configuration
     *  The name of the configuration file to used for the retrieval,
     *  e.g. io.ox/office
     *
     * @param path
     *  The path within the configuration file to retrieve the setting
     *  value, e.g. //module/maxTableCells
     *
     * @return
     *  The value of the setting or null if the setting is not available.
     */
    static private String getCascadeConfigurationValue(JSlobService jslobService, Session session, String configuration, String path) {
        String result = null;

        if ((null != session) && (null != jslobService)) {
            try {
                final JSlob officeJSlob = jslobService.get(configuration, session);
                JSONObject json = null;

                if (null != officeJSlob) {
                    json = officeJSlob.getJsonObject();
                }

                // cascade through the hierarchical structure to find the correct setting
                if (null != json) {
                    int i = 0;
                    String[] entries = path.split("/");
                    while (i < entries.length) {
                        String entry = entries[i++];
                        // ignore empty strings
                        if (entry.length() > 0) {
                            Object obj = json.opt(entry);
                            if (null == obj) {
                                return null;
                            } else if (obj instanceof JSONObject) {
                                // step down on JSONObject
                                json = (JSONObject)obj;
                            } else {
                                // use other object types as return value
                                result = String.valueOf(obj);
                                break;
                            }
                        }
                    }
                }
            } catch (OXException e) {
                LOG.error("Exception caught while retrieving cascade configuration value", e);
            }
        }

        return result;
    }

    private static final Log LOG = com.openexchange.log.Log.loggerFor(ConfigurationHelper.class);
}
