/*
*
*    OPEN-XCHANGE legal information
*
*    All intellectual property rights in the Software are protected by
*    international copyright laws.
*
*
*    In some countries OX, OX Open-Xchange, open xchange and OXtender
*    as well as the corresponding Logos OX Open-Xchange and OX are registered
*    trademarks.
*    The use of the Logos is not covered by the GNU General Public License.
*    Instead, you are allowed to use these Logos according to the terms and
*    conditions of the Creative Commons License, Version 2.5, Attribution,
*    Non-commercial, ShareAlike, and the interpretation of the term
*    Non-commercial applicable to the aforementioned license is published
*    on the web site http://www.open-xchange.com/EN/legal/index.html.
*
*    Please make sure that third-party modules and libraries are used
*    according to their respective licenses.
*
*    Any modifications to this package must retain all copyright notices
*    of the original copyright holder(s) for the original code used.
*
*    After any such modifications, the original and derivative code shall remain
*    under the copyright of the copyright holder(s) and/or original author(s)per
*    the Attribution and Assignment Agreement that can be located at
*    http://www.open-xchange.com/EN/developer/. The contributing author shall be
*    given Attribution for the derivative code and a license granting use.
*
*     Copyright (C) 2016 OX Software GmbH
*     Mail: info@open-xchange.com
*
*
*     This program is free software; you can redistribute it and/or modify it
*     under the terms of the GNU General Public License, Version 2 as published
*     by the Free Software Foundation.
*
*     This program is distributed in the hope that it will be useful, but
*     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
*     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
*     for more details.
*
*     You should have received a copy of the GNU General Public License along
*     with this program; if not, write to the Free Software Foundation, Inc., 59
*     Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
*/
package com.openexchange.office.realtime.impl.presentation;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.exception.ExceptionUtils;
import com.openexchange.exception.OXException;
import com.openexchange.office.document.ChunkableDocLoader;
import com.openexchange.office.document.DocContextLogger;
import com.openexchange.office.document.ImExportHelper;
import com.openexchange.office.document.OXDocument;
import com.openexchange.office.filter.api.FilterException;
import com.openexchange.office.filter.api.IImporter;
import com.openexchange.office.filter.core.FilterExceptionToErrorCode;
import com.openexchange.office.htmldoc.GenericHtmlDocumentBuilder;
import com.openexchange.office.message.MessageChunk;
import com.openexchange.office.message.MessageHelper;
import com.openexchange.office.message.MessagePropertyKey;
import com.openexchange.office.message.OperationHelper;
import com.openexchange.office.realtime.doc.DocumentEventHelper;
import com.openexchange.office.realtime.impl.ConnectionStatus;
import com.openexchange.office.realtime.impl.DocumentConnection;
import com.openexchange.office.realtime.tools.DebugHelper;
import com.openexchange.office.realtime.tools.MessageData;
import com.openexchange.office.tools.common.system.SystemInfoHelper;
import com.openexchange.office.tools.doc.OXDocumentException;
import com.openexchange.office.tools.error.ErrorCode;
import com.openexchange.office.tools.user.UserData;
import com.openexchange.realtime.packet.ID;
import com.openexchange.realtime.packet.Message;
import com.openexchange.realtime.packet.Stanza;
import com.openexchange.realtime.payload.PayloadElement;
import com.openexchange.realtime.payload.PayloadTree;
import com.openexchange.realtime.payload.PayloadTreeNode;
import com.openexchange.realtime.util.ActionHandler;
import com.openexchange.realtime.util.ElementPath;
import com.openexchange.server.ServiceLookup;
import com.openexchange.tools.session.ServerSession;

/**
 * Presentation-specific implementation of a DocumentConection. This class provides methods
 * to handle the text-specific requests, loading/saving text documents.
 *
 * @author Carsten Driesner
 * @since 7.8.2
 */
public class PresentationConnection extends DocumentConnection {
    @SuppressWarnings( "deprecation" )
    private final static org.apache.commons.logging.Log LOG = com.openexchange.log.LogFactory.getLog(PresentationConnection.class);
    private final static ActionHandler m_handler = new ActionHandler(PresentationConnection.class);

    public PresentationConnection(final ServiceLookup servicesDEPRECATED, final ID groupID, final String componentID) {
        super(null, groupID, m_handler, componentID);
    }

    /**
     * Provides a stanza for the first message sent to the client joining a document (chat-room). The OX Documents client expects to find
     * the document operation within the stanza provided by this method.
     *
     * @param onBehalfOf The ID of the client which wants to join the document.
     * @return The stanza sent to the client which wants to join the document. The stanza includes the document operations and operations
     *         which are currently pending for the next save. The stanza also has the update state including the editor ID.
     * @see com.openexchange.realtime.group.GroupDispatcher#getWelcomeMessage(com.openexchange.realtime.packet.ID)
     */
    @Override
    public Stanza getWelcomeMessage(final ID onBehalfOf) {
        Message welcomeMessage = null;
        final UserData userData = this.getUserData(onBehalfOf);

        if ((null != onBehalfOf) && (isMember(onBehalfOf))) {
            final ServerSession serverSession = MessageHelper.getServerSession(null, null, userData, onBehalfOf);

            LOG.debug("RT connection: [getWelcomeMessge] called for presentation document: " + DebugHelper.getDocumentFolderAndFileId(userData));

            // Check, if we can retrieve the session from the join request. Include using the session id
            // which cannot be processed correctly in some cases, if the session storage hazelcast bundle
            // is not started!!
            if ((null == serverSession) || (!isSessionIdAccessbile(userData.getSessionId()))) {
                LOG.error("RT Connection: Session information not accessible. " + ((null == serverSession) ? "ServerSession is null" : "IsSessionIdAccessible is false - check cluster setup"));
                return impl_handleError4Welcome(onBehalfOf, serverSession, ErrorCode.GENERAL_SERVER_COMPONENT_NOT_WORKING_ERROR, userData.getFolderId(), userData.getFileId());
            }

            // Check global save state - stop and provide error message, if global save is set!
            // This prevents the user to edit a document which is saved by background save which
            // lasts longer. The user wouldn't be able to save it on close
            if (this.getGlobalSaveState()) {
                return impl_handleError4Welcome(onBehalfOf, serverSession, ErrorCode.GENERAL_BACKGROUNDSAVE_IN_PROGRESS_ERROR, userData.getFolderId(), userData.getFileId());
            }

            final OXDocument oxDocument = new OXDocument(serverSession, getServices(), userData, getFileHelper(), isNewDocLoaded(), getResourceManager(), getStorageHelper(), isDebugOperations(), null);
            final int documentOSN = oxDocument.getUniqueDocumentId();
            final JSONObject operations = new JSONObject();

            // Initialize the error code retrieved by our OXDocument instance. The error code is
            // set by the ctor of the instance which tries to retrieve the document stream.
            ErrorCode errorCode = oxDocument.getLastError();

            // check error code after loading the document
            if (errorCode.isError()) {
                return impl_handleError4Welcome(onBehalfOf, serverSession, errorCode, userData.getFolderId(), userData.getFileId());
            }

            boolean canLoadWithLocalStorage = handleLocalStorageLoading(oxDocument, onBehalfOf, userData);

            // check consistency and create necessary hazelcast backup entries for save-as
            try {
                checkAndUpdateDocumentState(oxDocument);
            } catch (OXDocumentException e) {
            	 return impl_handleError4Welcome(onBehalfOf, serverSession, e.getErrorcode(), userData.getFolderId(), userData.getFileId());
            }

            // Initialize the error code retrieved by our OXDocument instance. The error code is
            // set by the ctor of the instance which tries to retrieve the document stream.
            String genericHtmlDocument = null;
            boolean fastLoad = false;
            final JSONObject importerProps = new JSONObject();

            // Set current meta data for the document
            synchronized(getSyncAccess()) {
                setLastKnownMetaData(oxDocument.getDocumentMetaData());
            }
            // must be called after meta data have been set - locked state must be set#
            final boolean writeAccess = !oxDocument.getDocumentMetaData().getWriteProtectedState();
            impl_handleEditRights(onBehalfOf, writeAccess);

            // In case the client joined us from a "fastempty" operation or wants to load via local
            // storage we can take a short-cut now and send a simple answer.
            if (userData.getFastEmptyState())
                return impl_provideSimpleWelcomeMessageForFastEmpty(onBehalfOf, serverSession, oxDocument);
            else if (canLoadWithLocalStorage)
                return impl_provideSimpleWelcomeMessageForLocalStorageLoading(onBehalfOf, serverSession, oxDocument);

            // check error code before continuing to retrieve operations
            if (errorCode.isNoError()) {
                // Initialize result object with operations from persistent document first.
                // Check for FilterExceptions that can occur due to problems with the file.
                try {
                    final IImporter importer = ImExportHelper.getImporter(getServices(), getFileHelper());
                    if (importer != null) {
	                    final JSONObject documentOperations = new JSONObject();
	                    final ChunkableDocLoader chunkDocLoader = new ChunkableDocLoader(oxDocument, importer, importerProps);
	                    chunkDocLoader.prepareLoad();

	                    // presentation loads the complete document - will be changed in the future
	                    OperationHelper.appendJSON(documentOperations, chunkDocLoader.getGlobalOperations());
	                    OperationHelper.appendJSON(documentOperations, chunkDocLoader.getActiveOperations());
	                    OperationHelper.appendJSON(documentOperations,  chunkDocLoader.getRemainingOperations());

	                    final String module = "presentation";
	                    if (GenericHtmlDocumentBuilder.isFastLoadActive(module, serverSession)) {
	                        fastLoad = true;
	                        genericHtmlDocument = GenericHtmlDocumentBuilder.buildHtmlDocument(module, documentOperations, oxDocument.getMetaData(), serverSession);
	                    }
	                    OperationHelper.appendJSON(operations, documentOperations);
                    } else {
                    	errorCode = ErrorCode.LOADDOCUMENT_NO_FILTER_FOR_DOCUMENT_ERROR;
                        final String docUserData = DocContextLogger.getContextStringWithDoc(isDebugOperations(), onBehalfOf, userData, oxDocument);
                        LOG.warn("RT connection: Coulnd't find filter for presentation document format. " + docUserData);
                    }
                } catch (final FilterException e) {
                    errorCode = getErrorCodeFromFilterExceptionAtImport(e, onBehalfOf, userData, oxDocument);
                } catch (final Throwable e) {
                    errorCode = ErrorCode.LOADDOCUMENT_FAILED_ERROR;
                    LOG.warn("RT connection: getWelcomeMessage, Exception caught", e);
                }
            }

            // check error code again as the filter and other parts can throw errors, too
            if (errorCode.isError()) {
                welcomeMessage = impl_handleError4Welcome(onBehalfOf, serverSession, errorCode, userData.getFolderId(), userData.getFileId());
                return welcomeMessage;
            }

            welcomeMessage = impl_createWelcomeMessage(onBehalfOf, operations, documentOSN, false, genericHtmlDocument, 0, oxDocument.getDocumentMetaData(), fastLoad);

            // update recent file list on open file now
            impl_addToRecentFileList(oxDocument.getDocumentMetaData(), serverSession);
        } else {
            // update statistics with load error
            DocumentEventHelper.addOpenErrorEvent(getFileHelper());
            LOG.debug("RT connection: Didn't send [welcome message] since the requester was no member: " + ((null != onBehalfOf) ? onBehalfOf.toString() : "null"));
        }

        return welcomeMessage;
    }

    /**
     * Handles the request to apply new actions to the document. The method controls that the sender has the edit rights and it makes some
     * basic operation checking to protected the document consistency.
     *
     * @notice Stanzas layout: ApplyActions { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666"
     *         element: "message", payloads: [{ element: "action", data: "applyactions" }, { namespace: m_componentID, element: "actions", data:
     *         "[{operations: [...]}, ...]" }] }
     * @param stanza A Stanza which contain the operations to be applied to the current document.
     * @return The message
     * @throws OXException
     */
    public Message handleApplyActions(Stanza stanza) throws OXException {
        final UserData userData = this.getUserData(stanza.getFrom());
        LOG.debug("RT connection: [applyActions] received for document: " + DebugHelper.getDocumentFolderAndFileId(userData));

        checkForDisposed();
        checkForThreadId("handleApplyActions");

        // Check that we have a ID for the sender stanza and that the sender
        // has the edit rights.
        final Message returnMessage = new Message();
        final ConnectionStatus connectionStatus = getConnectionStatus();
        final ID fromId = stanza.getFrom();
        if (this.isValidMemberAndEditor(fromId)) {
            final JSONObject jsonRequest = MessageHelper.getJSONRequest(stanza, new ElementPath(getComponentID(), MessagePropertyKey.KEY_ACTIONS), true);
            final MessageChunk actionChunk = new MessageChunk(jsonRequest, fromId);
            final JSONObject jsonResult = new JSONObject();
            boolean emptyActionChunk = false;

            // increase number of received actions
            LOG.debug("RT connection: [applyActions] received, applied from: " + fromId.toString());

            // Server must check the consistency of the operations sent by the current editor client.
            // If check fails the client must receive a 'hangup' message and be excluded from the
            // document group.
            final ConnectionStatus statusToCheck = getConnectionStatusClone();
            final ErrorCode checkOpErr = OperationHelper.hasValidOperations(actionChunk, statusToCheck.getOperationStateNumber());
            if (checkOpErr.isError()) {
                // send hangup message and ignore operations
                try {
                    sendHangUp(fromId, statusToCheck, checkOpErr, this.getRestoreData(), null, HangUpReceiver.HANGUP_ID,
                               "Sent hangup due to inconsistent operations in message chunk.");
                } catch (Exception e) {
                    LOG.error("RT connection: Exception caught while trying to send hangup message to client " + fromId.toString());
                }
                if (impl_resetEditor(fromId)) {
                    // also send a update message with the new/reseted editor to all clients
                    updateClients(new MessageData(null, getConnectionStatusClone()), fromId);
                }
                return returnMessage;
            }

            // set document state to modified
            this.setModified();

            final String actionsChunkUuid = actionChunk.getUuid();

            // handling operation state number for operations from clients
            int osn = -1;
            try {
                MessageChunk osnActionChunk = actionChunk;
                if (osnActionChunk.getOperations().hasAndNotNull(MessagePropertyKey.KEY_ACTIONS)) {
                    final JSONArray allActions = osnActionChunk.getOperations().getJSONArray(MessagePropertyKey.KEY_ACTIONS);
                    final JSONObject lastAction = allActions.getJSONObject(allActions.length() - 1);

                    if (lastAction.has("operations")) {
                        final JSONArray allOps = lastAction.getJSONArray("operations");
                        final JSONObject lastOp = allOps.getJSONObject(allOps.length() - 1);

                        // update statistics with incoming operations requests
                        DocumentEventHelper.addIncomingEvent(getFileHelper());

                        if (lastOp.has("osn") && lastOp.has("opl")) {
                            synchronized (connectionStatus) {
                                osn = lastOp.getInt("osn") + lastOp.getInt("opl");
                                connectionStatus.setOperationStateNumber(osn);
                            }
                            LOG.debug("RT connection: handleApplyActions, lastOp received = " + OperationHelper.operationToString(lastOp));
                            LOG.debug("RT connection: handleApplyActions, current server osn = " + osn);
                        }
                    }
                }
            } catch (JSONException e) {
                LOG.debug("RT connection: handleApplyActions: Error setting operation state number to operation", e);
            }

            // set document state to modified
            this.setModified();

            // update the shared state
            int msgChunkListSize = 0;
            synchronized (getSyncAccess()) {
                if (!emptyActionChunk) {
                    addMessageChunkToQueue(actionChunk);
                }
                msgChunkListSize = getMessageQueueCount();
            }
            LOG.debug("RT connection: handleApplyActions, message chunk list size = " + String.valueOf(msgChunkListSize));

            // add the operations to the DocumentDataManager to have a copy
            // of the operations for a possible later save-as
            this.pushOperationsForRestore(actionChunk, osn);

            final MessageData messageData = new MessageData(actionChunk, getConnectionStatusClone(),
                                                    MessageHelper.finalizeJSONResult(jsonResult, ErrorCode.NO_ERROR, jsonRequest, null));

            // Send the message to all participants (excluding the editor who sent it originally)
            returnMessage.setFrom(fromId);
            returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(messageData, MessageData.class.getName(), getComponentID(), "update")).build()));

            LOG.debug("RT connection: relaying [applyActions], applied from: " + fromId.toString());
            relayToAllExceptSender(returnMessage, stanza);

            // update statistics with distributed operations count
            int otherClientCount = 0;
            synchronized (connectionStatus) {
                otherClientCount = connectionStatus.getActiveClients() - 1;
            }

            DocumentEventHelper.addDistributedEvent(getFileHelper(), otherClientCount);

            // Send a special update to the editor, too. The editor can remove
            // pending waiting actions from its waiting queue. This ensures that
            // the client only removes actions which have been successfully processed
            // by the backend.
            final Message senderMessage = new Message();
            final ConnectionStatus updateForEditor = getConnectionStatusClone();

            updateForEditor.setAcks(new String[] { actionsChunkUuid });
            final MessageData senderMessageData = new MessageData(null, updateForEditor, null);
            senderMessage.setFrom(fromId);
            senderMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(senderMessageData, MessageData.class.getName(), getComponentID(), "update")).build()));
            if (stanza.getTracer() != null) {
                senderMessage.setTracer(stanza.getTracer());
            }
            relayToID(senderMessage, fromId);
        } else {
            String editor = null;
            synchronized(connectionStatus) { editor = connectionStatus.getCurrentEditingUserId(); }
            final ID editorId = (editor != null) && (editor.length() > 0) ? new ID(editor) : null;
            final boolean stanzaFromEditor = (null != editor) && (null != fromId) && (fromId.equals(editorId));

            if (null != fromId && !stanzaFromEditor) {
                final JSONObject jsonRequest = MessageHelper.getJSONRequest(stanza, new ElementPath(getComponentID(), MessagePropertyKey.KEY_ACTIONS), true);
                final MessageChunk actionChunk = new MessageChunk(jsonRequest, fromId);
                // The sender has no edit rights, therefore we send a hang up message
                // so the client will be notified and can reload the document.
                try {
                    sendHangUp(fromId, getConnectionStatusClone(), ErrorCode.HANGUP_NO_EDIT_RIGHTS_ERROR, this.getRestoreData(), actionChunk.getOperations(), HangUpReceiver.HANGUP_ID,
                               "Relaying [hangup] due to missing edit rights, applied from: " + fromId.toString());
                } catch (Exception e) {
                    LOG.error("RT connection: Exception caught while trying to send hangup message to client " + fromId.toString());
                }
            }

            if (LOG.isDebugEnabled()) {
                if (!isMember(fromId)) {
                    LOG.debug("RT connection: Didn't relay [applyActions] since the requester was no member: " + ((null != fromId) ? fromId.toString() : "null"));
                }
                if (!stanzaFromEditor) {
                    LOG.debug("RT connection: Didn't relay [applyActions] because the requester doesn't have the edit rights");
                }
            }
        }
        return returnMessage;
    }

    /**
     * Handles the "dispose" notification provided by the Realtime-Framework.
     * This method is called when the last client left the document and the
     * Realtime-Framework wants to dispose the GroupDispatcher instance.
     * Attention: If getSignOffMessage or onLeave is marked with the annotation
     * "Asynchronous" onDispose is also called from the same background thread.
     * Therefore this method must be thread-safe, too.
     *
     * @param id The ID of the last client that left the document (chat-room).
     * @see com.openexchange.realtime.group.GroupDispatcher#onDispose()
     */
    @Override
    protected void onDispose(ID id) throws OXException {
        LOG.debug("RT connection: onDispose called for presentation document: " + DebugHelper.getDocumentFolderAndFileId(null, this.getResourceID()) + ", identity: " + getIdentity());

        try {
            final UserData userData = this.getUserData(id);

            if (isSaveOnDispose()) {
                flushDocument(MessageHelper.getServerSession(null, null, userData, id), userData, true, true, false, false, false);
            }
            impl_resetConnectionStatusOnDispose();
        } catch (Throwable e) {
        	ExceptionUtils.handleThrowable(e);
            LOG.error("RT connection: Exception while cleanup connection instance", e);
        } finally {
            super.onDispose(id);
        }
    }
}
