/*
*
*    OPEN-XCHANGE legal information
*
*    All intellectual property rights in the Software are protected by
*    international copyright laws.
*
*
*    In some countries OX, OX Open-Xchange, open xchange and OXtender
*    as well as the corresponding Logos OX Open-Xchange and OX are registered
*    trademarks.
*    The use of the Logos is not covered by the GNU General Public License.
*    Instead, you are allowed to use these Logos according to the terms and
*    conditions of the Creative Commons License, Version 2.5, Attribution,
*    Non-commercial, ShareAlike, and the interpretation of the term
*    Non-commercial applicable to the aforementioned license is published
*    on the web site http://www.open-xchange.com/EN/legal/index.html.
*
*    Please make sure that third-party modules and libraries are used
*    according to their respective licenses.
*
*    Any modifications to this package must retain all copyright notices
*    of the original copyright holder(s) for the original code used.
*
*    After any such modifications, the original and derivative code shall remain
*    under the copyright of the copyright holder(s) and/or original author(s)per
*    the Attribution and Assignment Agreement that can be located at
*    http://www.open-xchange.com/EN/developer/. The contributing author shall be
*    given Attribution for the derivative code and a license granting use.
*
*     Copyright (C) 2016 OX Software GmbH
*     Mail: info@open-xchange.com
*
*
*     This program is free software; you can redistribute it and/or modify it
*     under the terms of the GNU General Public License, Version 2 as published
*     by the Free Software Foundation.
*
*     This program is distributed in the hope that it will be useful, but
*     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
*     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
*     for more details.
*
*     You should have received a copy of the GNU General Public License along
*     with this program; if not, write to the Free Software Foundation, Inc., 59
*     Temple Place, Suite 330, Boston, MA 02111-1307 USA
*
*/
package com.openexchange.office.realtime.impl.calc;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import com.openexchange.exception.ExceptionUtils;
import com.openexchange.exception.OXException;
import com.openexchange.office.message.MessageChunk;
import com.openexchange.office.message.MessageHelper;
import com.openexchange.office.message.MessagePropertyKey;
import com.openexchange.office.message.OperationHelper;
import com.openexchange.office.realtime.doc.DocumentEventHelper;
import com.openexchange.office.realtime.impl.ConnectionStatus;
import com.openexchange.office.realtime.impl.DocumentConnection;
import com.openexchange.office.realtime.tools.DebugHelper;
import com.openexchange.office.realtime.tools.MessageData;
import com.openexchange.office.document.ChunkableDocLoader;
import com.openexchange.office.document.DocContextLogger;
import com.openexchange.office.document.ImExportHelper;
import com.openexchange.office.document.OXDocument;
import com.openexchange.office.filter.api.FilterException;
import com.openexchange.office.filter.core.FilterExceptionToErrorCode;
import com.openexchange.office.filter.api.IImporter;
import com.openexchange.office.tools.common.system.SystemInfoHelper;
import com.openexchange.office.tools.doc.OXDocumentException;
import com.openexchange.office.tools.error.ErrorCode;
import com.openexchange.office.tools.error.ExceptionToErrorCode;
import com.openexchange.office.tools.user.UserData;
import com.openexchange.realtime.packet.ID;
import com.openexchange.realtime.packet.Message;
import com.openexchange.realtime.packet.Stanza;
import com.openexchange.realtime.payload.PayloadElement;
import com.openexchange.realtime.payload.PayloadTree;
import com.openexchange.realtime.payload.PayloadTreeNode;
import com.openexchange.realtime.util.ActionHandler;
import com.openexchange.realtime.util.ElementPath;
import com.openexchange.server.ServiceLookup;
import com.openexchange.tools.session.ServerSession;

/**
 * Spreadsheet-specific implementation of a DocumentConection. This class provides methods
 * to handle the spreadsheet-specific requests, loading/saving spreadsheet documents.
 *
 * @author Carsten Driesner
 * @since 7.8.0
 */
public class CalcConnection extends DocumentConnection implements IRequestProcessor {
    @SuppressWarnings( "deprecation" )
    private final static org.apache.commons.logging.Log LOG = com.openexchange.log.LogFactory.getLog(CalcConnection.class);
    private final static ActionHandler m_handler = new ActionHandler(CalcConnection.class);
    private final static boolean NO_FASTLOAD = false;
    private final AsyncLoadRequestQueue m_loadRequestQueue;

    public CalcConnection(final ServiceLookup servicesDEPRECATED, final ID groupID, final AsyncLoadRequestQueue requestQueue, final String componentID) {
        super(null, groupID, m_handler, componentID);

        m_loadRequestQueue = requestQueue;
    }

    /**
     * Provides a stanza for the first message sent to the client joining a document (chat-room). The OX Documents client expects to find
     * the document operation within the stanza provided by this method.
     *
     * @param onBehalfOf The ID of the client which wants to join the document.
     * @return The stanza sent to the client which wants to join the document. The stanza includes the document operations and operations
     *         which are currently pending for the next save. The stanza also has the update state including the editor ID.
     * @see com.openexchange.realtime.group.GroupDispatcher#getWelcomeMessage(com.openexchange.realtime.packet.ID)
     */
    @Override
    public Stanza getWelcomeMessage(final ID onBehalfOf) {
        Message welcomeMessage = null;
        final   UserData userData = this.getUserData(onBehalfOf);
        boolean singleSheet = false;

        if ((null != onBehalfOf) && (isMember(onBehalfOf))) {
            final ServerSession serverSession = MessageHelper.getServerSession(null, null, userData, onBehalfOf);

            LOG.debug("RT connection: [getWelcomeMessge] called for spreadsheet document: " + DebugHelper.getDocumentFolderAndFileId(userData));

            // Check, if we can retrieve the session from the join request. Include using the session id
            // which cannot be processed correctly in some cases, if the session storage hazelcast bundle
            // is not started!!
            if ((null == serverSession) || (!isSessionIdAccessbile(userData.getSessionId()))) {
                LOG.error("RT Connection: Session information not accessible. " + ((null == serverSession) ? "ServerSession is null" : "IsSessionIdAccessible is false - check cluster setup"));
                return impl_handleError4Welcome(onBehalfOf, serverSession, ErrorCode.GENERAL_SERVER_COMPONENT_NOT_WORKING_ERROR, userData.getFolderId(), userData.getFileId());
            }

            // Check global save state - stop and provide error message, if global save is set!
            // This prevents the user to edit a document which is saved by background save which
            // lasts longer. The user wouldn't be able to save it on close
            if (this.getGlobalSaveState()) {
                return impl_handleError4Welcome(onBehalfOf, serverSession, ErrorCode.GENERAL_BACKGROUNDSAVE_IN_PROGRESS_ERROR, userData.getFolderId(), userData.getFileId());
            }

            final OXDocument oxDocument = new OXDocument(serverSession, getServices(), userData, getFileHelper(), isNewDocLoaded(), getResourceManager(), getStorageHelper(), isDebugOperations(), null);
            final int documentOSN = oxDocument.getUniqueDocumentId();
            final JSONObject operations = new JSONObject();

            // Initialize the error code retrieved by our OXDocument instance. The error code is
            // set by the ctor of the instance which tries to retrieve the document stream.
            ErrorCode errorCode = oxDocument.getLastError();

            // check error code after loading the document
            if (errorCode.isError()) {
                return impl_handleError4Welcome(onBehalfOf, serverSession, errorCode, userData.getFolderId(), userData.getFileId());
            }

            // check consistency and create necessary hazelcast backup entries for save-as
            try {
                checkAndUpdateDocumentState(oxDocument);
            } catch (OXDocumentException e) {
           	    return impl_handleError4Welcome(onBehalfOf, serverSession, e.getErrorcode(), userData.getFolderId(), userData.getFileId());
            }

            // Initialize the error code retrieved by our OXDocument instance. The error code is
            // set by the ctor of the instance which tries to retrieve the document stream.
            final JSONObject importerProps = new JSONObject();
            JSONObject previewOperations = null;

            // Set current meta data for the document
            synchronized(getSyncAccess()) {
                setLastKnownMetaData(oxDocument.getDocumentMetaData());
            }
            // must be called after meta data have been set - locked state must be set#
            final boolean writeAccess = !oxDocument.getDocumentMetaData().getWriteProtectedState();
            impl_handleEditRights(onBehalfOf, writeAccess);

            ChunkableDocLoader chunkDocLoader = null;

            // check error code before continuing to retrieve operations
            if (errorCode.isNoError()) {
                // Initialize result object with operations from persistent document first.
                // Check for FilterExceptions that can occur due to problems with the file.
                try {
                    final IImporter importer = ImExportHelper.getImporter(getServices(), getFileHelper());
                    if (importer != null) {
	                    // create and initialize the chunkable document loader wrapper
	                    chunkDocLoader = new ChunkableDocLoader(oxDocument, importer, importerProps);
	                    chunkDocLoader.prepareLoad();

	                    // Retrieve the global and active part of the document for the first answer
	                    final JSONObject globalOps  = chunkDocLoader.getGlobalOperations();
	                    final JSONObject activeOps  = chunkDocLoader.getActiveOperations();
	                    final int        partsCount = chunkDocLoader.getPartsCount();

	                    if (partsCount > 1) {
	                        // we need to send the operations in two parts - possible rescue
	                        // operations will be sent via the second part
	                        singleSheet = false;
	                        previewOperations = globalOps;
	                        OperationHelper.appendJSON(previewOperations, activeOps);
	                    } else {
	                        // we can send the operations in one synchronous part
	                        singleSheet = true;
	                        OperationHelper.appendJSON(operations, globalOps);
	                        OperationHelper.appendJSON(operations, activeOps);
	                        // we always call the remaining part to receive all possible operations (including rescue ops)
	                        OperationHelper.appendJSON(operations, chunkDocLoader.getRemainingOperations());
	                    }
                    } else {
                    	errorCode = ErrorCode.LOADDOCUMENT_NO_FILTER_FOR_DOCUMENT_ERROR;
                        final String docUserData = DocContextLogger.getContextStringWithDoc(isDebugOperations(), onBehalfOf, userData, oxDocument);
                        LOG.warn("RT connection: Coulnd't find filter for spreadsheet document format. " + docUserData);
                    }
                } catch (final FilterException e) {
                    errorCode = getErrorCodeFromFilterExceptionAtImport(e, onBehalfOf, userData, oxDocument);
                } catch (final Exception e) {
                    errorCode = ErrorCode.LOADDOCUMENT_FAILED_ERROR;
                    LOG.warn("RT connection: getWelcomeMessage, Exception caught", e);
                }
            }

            // check error code again as the filter and other parts can throw errors, too
            if (errorCode.isError()) {
                welcomeMessage = impl_handleError4Welcome(onBehalfOf, serverSession, errorCode, userData.getFolderId(), userData.getFileId());
                return welcomeMessage;
            } else {

                try {
                    // create request and process load request in the calc load request queue asynchronously
                    if ((null != operations) && (operations.has(MessagePropertyKey.KEY_OPERATIONS))) {
                        // In case we don't have preview operations we send the
                        // complete operations array to the client
                        welcomeMessage = impl_createWelcomeMessage(onBehalfOf, operations, documentOSN, false, null, 0, oxDocument.getDocumentMetaData(), false);
                    } else if ((null != previewOperations) && (previewOperations.has(MessagePropertyKey.KEY_OPERATIONS))) {
                        if (singleSheet) {
                            // In case we just have one sheet and loaded the document synchronously
                            // we are done and return the operations.
                            welcomeMessage = impl_createWelcomeMessage(onBehalfOf, previewOperations, documentOSN, false, null, 0, oxDocument.getDocumentMetaData(), NO_FASTLOAD);
                        } else {
                            // In case we have preview operations we provide the OXDocument instance
                            // to complete the loading of the remaining sheets in a background thread.
                            final int       activeSheetIndex = chunkDocLoader.getActivePartIndex();
                            final JSONArray previewOpsArray = previewOperations.getJSONArray(MessagePropertyKey.KEY_OPERATIONS);
                            final Request   request = new Request(this.getResourceID(), onBehalfOf, chunkDocLoader, this, documentOSN, previewOpsArray.length(), userData);

                            welcomeMessage = impl_createAsyncLoadWelcomeMessage(getId(), onBehalfOf, previewOpsArray, activeSheetIndex);
                            m_loadRequestQueue.putRequest(request);
                        }
                    } else {
                        LOG.error("RT connection: Impossible case trying to process spreadsheet request");
                        welcomeMessage = impl_createErrorMessage4WelcomeMessage(onBehalfOf, ErrorCode.LOADDOCUMENT_FAILED_ERROR, getConnectionStatusClone());
                        return welcomeMessage;
                    }
                } catch (OXException e) {
                    errorCode = ExceptionToErrorCode.map(e, ErrorCode.LOADDOCUMENT_FAILED_ERROR, true);
                    welcomeMessage = impl_handleError4Welcome(onBehalfOf, serverSession, errorCode, userData.getFolderId(), userData.getFileId());
                    return welcomeMessage;
                } catch (Exception e) {
                    LOG.error("Exception while determining multi-sheet doc & async load preparation", e);
                    welcomeMessage = impl_handleError4Welcome(onBehalfOf, serverSession, ErrorCode.LOADDOCUMENT_FAILED_ERROR, userData.getFolderId(), userData.getFileId());
                    return welcomeMessage;
                }
            }

            // update recent file list on open file now
            impl_addToRecentFileList(oxDocument.getDocumentMetaData(), serverSession);
        } else {
            // update statistics with load error
            DocumentEventHelper.addOpenErrorEvent(getFileHelper());
            LOG.debug("RT connection: Didn't send [welcome message] since the requester was no member: " + ((null != onBehalfOf) ? onBehalfOf.toString() : "null"));
        }

        return welcomeMessage;
    }

    /**
     * Handles the request to apply new actions to the document. The method controls that the sender has the edit rights and it makes some
     * basic operation checking to protected the document consistency.
     *
     * @notice Stanzas layout: ApplyActions { to: "synthetic.office://operations/folderId.fileId session: "72306eae544b4ca6aabab1485ec8a666"
     *         element: "message", payloads: [{ element: "action", data: "applyactions" }, { namespace: m_componentID, element: "actions", data:
     *         "[{operations: [...]}, ...]" }] }
     * @param stanza A Stanza which contain the operations to be applied to the current document.
     * @return The message
     * @throws OXException
     */
    public Message handleApplyActions(Stanza stanza) throws OXException {
        final UserData userData = this.getUserData(stanza.getFrom());
        LOG.debug("RT connection: [applyActions] received for document: " + DebugHelper.getDocumentFolderAndFileId(userData));

        checkForDisposed();
        checkForThreadId("handleApplyActions");

        // Check that we have a ID for the sender stanza and that the sender
        // has the edit rights.
        final Message returnMessage = new Message();
        final ConnectionStatus connectionStatus = getConnectionStatus();
        final ID fromId = stanza.getFrom();
        if (this.isValidMemberAndEditor(fromId)) {
            final JSONObject jsonRequest = MessageHelper.getJSONRequest(stanza, new ElementPath(getComponentID(), MessagePropertyKey.KEY_ACTIONS), true);
            final MessageChunk actionChunk = new MessageChunk(jsonRequest, fromId);
            final JSONObject jsonResult = new JSONObject();
            boolean emptyActionChunk = false;

            // increase number of received actions
            LOG.debug("RT connection: [applyActions] received, applied from: " + fromId.toString());

            // Server must check the consistency of the operations sent by the current editor client.
            // If check fails the client must receive a 'hangup' message and be excluded from the
            // document group.
            final ConnectionStatus statusToCheck = getConnectionStatusClone();
            final ErrorCode checkOpErr = OperationHelper.hasValidOperations(actionChunk, statusToCheck.getOperationStateNumber());
            if (checkOpErr.isError()) {
                // send hangup message and ignore operations
                try {
                    sendHangUp(fromId, statusToCheck, checkOpErr, getRestoreData(), null, HangUpReceiver.HANGUP_ID,
                               "Sent hangup due to inconsistent operations in message chunk.");
                } catch (Exception e) {
                    LOG.error("RT connection: Exception caught while trying to send hangup message to client " + fromId.toString());
                }
                if (impl_resetEditor(fromId)) {
                    // also send a update message with the new/reseted editor to all clients
                    updateClients(new MessageData(null, getConnectionStatusClone()), fromId);
                }
                return returnMessage;
            }

            // set document state to modified
            this.setModified();

            final String actionsChunkUuid = actionChunk.getUuid();

            // handling operation state number for operations from clients
            int osn = -1;
            try {
                MessageChunk osnActionChunk = actionChunk;
                if (osnActionChunk.getOperations().hasAndNotNull(MessagePropertyKey.KEY_ACTIONS)) {
                    final JSONArray allActions = osnActionChunk.getOperations().getJSONArray(MessagePropertyKey.KEY_ACTIONS);
                    final JSONObject lastAction = allActions.getJSONObject(allActions.length() - 1);

                    if (lastAction.has(MessagePropertyKey.KEY_OPERATIONS)) {
                        final JSONArray allOps = lastAction.getJSONArray(MessagePropertyKey.KEY_OPERATIONS);
                        final JSONObject lastOp = allOps.getJSONObject(allOps.length() - 1);

                        // update statistics with incoming operations requests
                        DocumentEventHelper.addIncomingEvent(getFileHelper());

                        if (lastOp.has("osn") && lastOp.has("opl")) {
                            synchronized (connectionStatus) {
                                osn = lastOp.getInt("osn") + lastOp.getInt("opl");
                                connectionStatus.setOperationStateNumber(osn);
                            }
                            LOG.debug("RT connection: handleApplyActions, lastOp received = " + OperationHelper.operationToString(lastOp));
                            LOG.debug("RT connection: handleApplyActions, current server osn = " + osn);
                        }
                    }
                }
            } catch (JSONException e) {
                LOG.debug("RT connection: handleApplyActions: Error setting operation state number to operation", e);
            }

            // update the shared state
            int msgChunkListSize = 0;
            synchronized (getSyncAccess()) {
                if (!emptyActionChunk) {
                    addMessageChunkToQueue(actionChunk);
                }
                msgChunkListSize = getMessageQueueCount();
            }
            LOG.debug("RT connection: handleApplyActions, message chunk list size = " + String.valueOf(msgChunkListSize));

            // add the operations to the DocumentDataManager to have a copy
            // of the operations for a possible later save-as
            this.pushOperationsForRestore(actionChunk, osn);

            final MessageData messageData = new MessageData(actionChunk, getConnectionStatusClone(),
                                                    MessageHelper.finalizeJSONResult(jsonResult, ErrorCode.NO_ERROR, jsonRequest, null));

            // Send the message to all participants (excluding the editor who sent it originally)
            returnMessage.setFrom(fromId);
            returnMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(messageData, MessageData.class.getName(), getComponentID(), "update")).build()));

            LOG.debug("RT connection: relaying [applyActions], applied from: " + fromId.toString());
            relayToAllExceptSender(returnMessage, stanza);

            // update statistics with distributed operations count
            int otherClientCount = 0;
            synchronized (connectionStatus) {
                otherClientCount = connectionStatus.getActiveClients() - 1;
            }

            DocumentEventHelper.addDistributedEvent(getFileHelper(), otherClientCount);

            // Send a special update to the editor, too. The editor can remove
            // pending waiting actions from its waiting queue. This ensures that
            // the client only removes actions which have been successfully processed
            // by the backend.
            final Message senderMessage = new Message();
            final ConnectionStatus updateForEditor = getConnectionStatusClone();

            updateForEditor.setAcks(new String[] { actionsChunkUuid });
            final MessageData senderMessageData = new MessageData(null, updateForEditor, null);
            senderMessage.setFrom(fromId);
            senderMessage.addPayload(new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(senderMessageData, MessageData.class.getName(), getComponentID(), "update")).build()));
            if (stanza.getTracer() != null) {
                senderMessage.setTracer(stanza.getTracer());
            }
            relayToID(senderMessage, fromId);
        } else {
            String editor = null;
            synchronized(connectionStatus) { editor = connectionStatus.getCurrentEditingUserId(); }
            final ID editorId = (editor != null) && (editor.length() > 0) ? new ID(editor) : null;
            final boolean stanzaFromEditor = (null != editor) && (null != fromId) && (fromId.equals(editorId));

            if (null != fromId && !stanzaFromEditor) {
                final JSONObject jsonRequest = MessageHelper.getJSONRequest(stanza, new ElementPath(getComponentID(), MessagePropertyKey.KEY_ACTIONS), true);
                final MessageChunk actionChunk = new MessageChunk(jsonRequest, fromId);
                // The sender has no edit rights, therefore we send a hang up message
                // so the client will be notified and can reload the document.
                try {
                    sendHangUp(fromId, getConnectionStatusClone(), ErrorCode.HANGUP_NO_EDIT_RIGHTS_ERROR, getRestoreData(), actionChunk.getOperations(), HangUpReceiver.HANGUP_ID,
                               "Relaying [hangup] due to missing edit rights, applied from: " + fromId.toString());
                } catch (Exception e) {
                    LOG.error("RT connection: Exception caught while trying to send hangup message to client " + fromId.toString());
                }
            }

            if (LOG.isDebugEnabled()) {
                if (!isMember(fromId)) {
                    LOG.debug("RT connection: Didn't relay [applyActions] since the requester was no member: " + ((null != fromId) ? fromId.toString() : "null"));
                }
                if (!stanzaFromEditor) {
                    LOG.debug("RT connection: Didn't relay [applyActions] because the requester doesn't have the edit rights");
                }
            }
        }
        return returnMessage;
    }

    /**
     * Handles the "dispose" notification provided by the Realtime-Framework.
     * This method is called when the last client left the document and the
     * Realtime-Framework wants to dispose the GroupDispatcher instance.
     * Attention: If getSignOffMessage or onLeave is marked with the annotation
     * "Asynchronous" onDispose is also called from the same background thread.
     * Therefore this method must be thread-safe, too.
     *
     * @param id The ID of the last client that left the document (chat-room).
     * @see com.openexchange.realtime.group.GroupDispatcher#onDispose()
     */
    @Override
    protected void onDispose(ID id) throws OXException {
        LOG.debug("RT connection: onDispose called for spreadsheet document: " + DebugHelper.getDocumentFolderAndFileId(null, this.getResourceID()) + ", identity: " + getIdentity());

        try {
            final UserData userData = this.getUserData(id);

            if (isSaveOnDispose()) {
                flushDocument(MessageHelper.getServerSession(null, null, userData, id), userData, true, true, false, false, false);
            }
            impl_resetConnectionStatusOnDispose();
        } catch (Throwable e) {
        	ExceptionUtils.handleThrowable(e);
            LOG.error("RT connection: Exception while cleanup connection instance", e);
        } finally {
            super.onDispose(id);
        }
    }

    /**
     * Implements the request processing for the asynchronous loading of documents.
     *
     * @param request
     *  The current request which must be processed by the Connection object.
     */
    @Override
    public void processRequest(Request request) throws OXException {
        final ConnectionStatus connectionStatus = getConnectionStatus();

        Message loadMessage = null;
        ErrorCode errorCode = ErrorCode.NO_ERROR;

        try {
            LOG.trace("RT connection: loading operations in chunks for document: " + request.getDocumentId() + ", user: " + request.getUserData().getUserRTId());

            final ChunkableDocLoader docLoader = request.getDocumentLoader();

            JSONObject remainingOps = docLoader.getRemainingOperations();
            if (remainingOps == null) {
                final JSONArray opsArray = new JSONArray();
                remainingOps = new JSONObject();
                remainingOps.put(MessagePropertyKey.KEY_OPERATIONS, opsArray);
            }
            if (LOG.isTraceEnabled()) {
                final JSONArray opsArray = remainingOps.optJSONArray(MessagePropertyKey.KEY_OPERATIONS);
                if (opsArray != null) {
                    LOG.trace("RT connection: remaining ops info: size = " + opsArray.length());
                }
            }
            loadMessage = impl_createWelcomeMessage(request.getId(), remainingOps, request.getDocumentOSN(), true, null, request.getPreviewOpsCount(), request.getDocumentMetaData(), NO_FASTLOAD);
        } catch (FilterException e) {
            final String docUserData = ChunkableDocLoader.getContextStringWithDoc(isDebugOperations(), request.getId(), request.getUserData(), request.getDocumentLoader());
            FilterExceptionToErrorCode.LogLevel level = FilterExceptionToErrorCode.determineLogLevel(e);
            FilterExceptionToErrorCode.log(LOG, level, "RT connection: Exception while asynchronous spreadsheet loading. ", docUserData, e);

            errorCode = FilterExceptionToErrorCode.map(e, ErrorCode.LOADDOCUMENT_CANNOT_RETRIEVE_OPERATIONS_ERROR);
            if (errorCode.getCode() == ErrorCode.GENERAL_UNKNOWN_ERROR.getCode()) {
                // map an unknown error for a filter exception to a more specific one
                errorCode = ErrorCode.LOADDOCUMENT_CANNOT_RETRIEVE_OPERATIONS_ERROR;
            }
        } catch (Throwable e) {
        	ExceptionUtils.handleThrowable(e);
            LOG.error("RT connection: Error generating spreadsheet operations", e);
            errorCode = ErrorCode.LOADDOCUMENT_FAILED_ERROR;
        } finally {
            if (!isDisposed()) {
                if (errorCode.getCode() == ErrorCode.NO_ERROR.getCode()) {
                    // in case of no error we just sent the finalLoad update
                    try {
                        boolean isMember = false;
                        synchronized (connectionStatus) {
                            isMember = connectionStatus.hasActiveUser(request.getId().toString());
                        }
                        if (isMember) {
                            this.relayToID(loadMessage, request.getId());
                        }
                    } catch (Throwable t) {
        	        	ExceptionUtils.handleThrowable(t);
                        LOG.debug("RT connection: Not able to send message to the document clients, may be client already left", t);
                    }
                } else {
                    try {
                        // in case of an error we have to sent a finalLoad update which
                        // has 'hasErrors' set and 'errorCode' includes more information
                        // about the root cause.
                        final Message aErrorMessage = this.impl_createErrorMessage4AsyncWelcomeMessage(request.getId(), errorCode);
                        this.relayToID(aErrorMessage, request.getId());
                    } catch (Throwable t) {
        	        	ExceptionUtils.handleThrowable(t);
                        LOG.warn("RT connection: Not able to send message to the document clients, maybe client already left", t);
                    }
                }
            }
            LOG.trace("RT connection: request finished for document: " + request.getDocumentId());
            request.finished();
        }
    }

    /**
     * Creates a welcome message for the client where it's notified that an asynchronous load process is used on the backend side. That
     * means that the client must wait until the final load update is sent.
     *
     * @param fromId The id of this real-time connection instance.
     * @param onBehalfOf The id of the client that requested to join the connection instance.
     * @param previewOperations Optional operations for the preview mode.
     * @param previewActiveSheetIndex Optional preview active sheet index.
     * @return A welcome message prepared for asynchronous loading by the client.
     */
    protected Message impl_createAsyncLoadWelcomeMessage(ID fromId, ID onBehalfOf, final JSONArray previewOperations, final int previewActiveSheetIndex) {
        Message welcomeMessage = null;
        // Provide a welcome message where the syncLoad attribute is set to false
        // which notifies the client that it has to wait for a specific update
        // message which has 'finalLoad' attribute set to true.
        final ConnectionStatus connectionStatus = getConnectionStatusClone();
        final JSONObject actionData = new JSONObject();
        connectionStatus.setSyncLoad(false);

        LOG.debug("RT connection: Completed synchronous loading for document " + Request.buildDocumentId(this.getResourceID()) + ", asynchronous loading started.");

        // set preview operations
        JSONObject previewJSONData = null;
        try {
            if (null != previewOperations) {
                final JSONObject previewData = new JSONObject();
                previewJSONData = new JSONObject();
                previewData.put(MessagePropertyKey.KEY_OPERATIONS, previewOperations);
                previewData.put(MessagePropertyKey.KEY_ACTIVESHEET, previewActiveSheetIndex);
                previewJSONData.put(MessagePropertyKey.KEY_PREVIEW, previewData);
            }
        } catch (JSONException e) {
            LOG.warn("Exception while creating preview object for asynchronous loading", e);
        }

        final MessageData messageData = new MessageData(
            new MessageChunk(actionData, getId()), connectionStatus, MessageHelper.finalizeJSONResult(ErrorCode.NO_ERROR, null, previewJSONData));

        // set empty actions as payload for the first welcome message
        welcomeMessage = new Message();
        welcomeMessage.setFrom(fromId);
        welcomeMessage.setTo(onBehalfOf);
        welcomeMessage.addPayload(
            new PayloadTree(PayloadTreeNode.builder().withPayload(
                new PayloadElement(messageData, MessageData.class.getName(), getComponentID(), "getactions")).build()));

        LOG.debug("RT connection: Sending preview [welcome message] to: " + onBehalfOf.toString());

        return welcomeMessage;
    }

}
