/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

/**
 *
 * @author sven.jacobi@open-xchange.com
 */

package com.openexchange.office.filter.odt.dom.components;

import java.util.Map;
import org.json.JSONException;
import org.json.JSONObject;
import org.xml.sax.SAXException;
import com.openexchange.office.filter.api.OCKey;
import com.openexchange.office.filter.odf.DLList;
import com.openexchange.office.filter.odf.DLNode;
import com.openexchange.office.filter.odf.INodeAccessor;
import com.openexchange.office.filter.odf.OdfOperationDoc;
import com.openexchange.office.filter.odf.OpAttrs;
import com.openexchange.office.filter.odf.components.Component;
import com.openexchange.office.filter.odf.components.ComponentContext;
import com.openexchange.office.filter.odf.draw.DrawFrame;
import com.openexchange.office.filter.odf.draw.DrawingType;
import com.openexchange.office.filter.odf.draw.IDrawing;
import com.openexchange.office.filter.odf.draw.IDrawingType;
import com.openexchange.office.filter.odf.styles.StyleFamily;
import com.openexchange.office.filter.odf.styles.StyleManager;
import com.openexchange.office.filter.odf.styles.StyleParagraph;
import com.openexchange.office.filter.odf.table.Table;
import com.openexchange.office.filter.odt.dom.Paragraph;

public class DrawFrameComponent extends Component implements IDrawingType {

	final DrawFrame drawFrame;

	public DrawFrameComponent(ComponentContext parentContext, DLNode<Object> textFrameNode, int componentNumber) {
		super(parentContext, textFrameNode, componentNumber);
		drawFrame = (DrawFrame)getObject();
	}

	@Override
	public String simpleName() {
	    return "DrawFrame";
	}

	@Override
    public DrawingType getType() {
		return drawFrame.getType();
	}

	@Override
	public Component getNextChildComponent(ComponentContext previousChildContext, Component previousChildComponent) {
		final IDrawing iDrawing = drawFrame.getDrawing();
		if(iDrawing!=null) {
	        final int nextComponentNumber = previousChildComponent != null ? previousChildComponent.getNextComponentNumber() : 0;
			DLNode<Object> nextNode = previousChildContext!=null ? previousChildContext.getNode().next : ((INodeAccessor)iDrawing).getContent().getFirstNode();
			while(nextNode!=null) {
				if(nextNode.getObject() instanceof Paragraph) {
					return new ParagraphComponent(this, nextNode, nextComponentNumber);
				}
				else if(nextNode.getObject() instanceof Table) {
					return new TableComponent(this, nextNode, nextComponentNumber);
				}
				nextNode = nextNode.next;
			}
		}
		return null;
	}

    @Override
    public Component insertChildComponent(OdfOperationDoc operationDocument, ComponentContext parentContext, DLNode<Object> contextNode, int number, Component child, Type type, JSONObject attrs) {
		final IDrawing iDrawing = drawFrame.getDrawing();
		if(iDrawing!=null) {
		    final DLList<Object> indexedNodeList = ((INodeAccessor)iDrawing).getContent();
	        final DLNode<Object> referenceNode = child != null && child.getComponentNumber()== number ? child.getNode() : null;

	        switch(type) {
	            case PARAGRAPH : {
	            	final DLNode<Object> newParagraphNode = new DLNode<Object>(new Paragraph(null));
	                indexedNodeList.addNode(referenceNode, newParagraphNode, true);
	                return new ParagraphComponent(parentContext, newParagraphNode, number);
	            }
	            case TABLE : {
	            	final DLNode<Object> newTableNode = new DLNode<Object>(new Table("text"));
	                indexedNodeList.addNode(referenceNode, newTableNode, true);
	                return new TableComponent(parentContext, newTableNode, number);
	            }
	            default : {
	                throw new UnsupportedOperationException();
	            }
	        }
		}
		return null;
    }

    @Override
    public Component insertChildComponent(OdfOperationDoc operationDocument, int number, JSONObject attrs, Type type)
    	throws UnsupportedOperationException, JSONException, SAXException {

        final Component c = insertChildComponent(operationDocument, this, getNode(), number, getChildComponent(number), type, attrs);
        if(c.getComponentNumber()==0&&c instanceof ParagraphComponent) {
        	// inheriting paragraph attributes from FrameStyle
        	final StyleManager styleManager = operationDocument.getDocument().getStyleManager();
    		final StyleParagraph paragraphStyle = new StyleParagraph(styleManager.getUniqueStyleName(StyleFamily.PARAGRAPH, isContentAutoStyle()), true, isContentAutoStyle());

    		final OpAttrs opAttrs = new OpAttrs();
    		styleManager.collectAllParagraphPropertiesFromStyle(opAttrs, drawFrame.getStyleName(), StyleFamily.GRAPHIC, isContentAutoStyle());
    		styleManager.collectAllTextPropertiesFromStyle(opAttrs, drawFrame.getStyleName(), StyleFamily.GRAPHIC, isContentAutoStyle());

    		// taking care of frame margins, they have to be applied to the paragraph style :-(
    		final OpAttrs allDrawingAttrs = new OpAttrs();
    		styleManager.collectAllGraphicPropertiesFromStyle(allDrawingAttrs, drawFrame.getStyleName(), StyleFamily.GRAPHIC, isContentAutoStyle(), false);
    		final Map<String, Object> drawingAttrs = allDrawingAttrs.getMap(OCKey.DRAWING.value(), false);
    		if(drawingAttrs!=null) {
    			final Map<String, Object> paragraphAttrs = opAttrs.getMap(OCKey.PARAGRAPH.value(), true);
    			final Object marginLeft = drawingAttrs.get(OCKey.MARGIN_LEFT.value());
    			final Object marginTop = drawingAttrs.get(OCKey.MARGIN_TOP.value());
    			final Object marginRight = drawingAttrs.get(OCKey.MARGIN_RIGHT.value());
    			final Object marginBottom = drawingAttrs.get(OCKey.MARGIN_BOTTOM.value());
    			if(marginLeft!=null) {
    				paragraphAttrs.put(OCKey.MARGIN_LEFT.value(), marginLeft);
    			}
    			if(marginTop!=null) {
    				paragraphAttrs.put(OCKey.MARGIN_TOP.value(), marginTop);
    			}
    			if(marginRight!=null) {
    				paragraphAttrs.put(OCKey.MARGIN_RIGHT.value(), marginRight);
    			}
    			if(marginBottom!=null) {
    				paragraphAttrs.put(OCKey.MARGIN_BOTTOM.value(), marginBottom);
    			}
    		}
    		if(attrs!=null) {
				StyleManager.deepCopy(attrs.asMap(), opAttrs);
    		}
    		paragraphStyle.applyAttrs(styleManager, new JSONObject(opAttrs));
    		final String existingStyleId = styleManager.getExistingStyleIdForStyleBase(paragraphStyle);
    		if(existingStyleId!=null) {
    			((Paragraph)((ParagraphComponent)c).getObject()).setStyleName(existingStyleId);
    		}
    		else {
    			styleManager.addStyle(paragraphStyle);
        		((Paragraph)((ParagraphComponent)c).getObject()).setStyleName(paragraphStyle.getName());
    		}
        }
        else if(attrs!=null) {
            c.applyAttrsFromJSON(operationDocument, attrs);
        }
        return c;
    }

	@Override
	public void applyAttrsFromJSON(OdfOperationDoc operationDocument, JSONObject attrs)
			throws JSONException {

		drawFrame.applyAttrsFromJSON(operationDocument, attrs, isContentAutoStyle());
	}

	@Override
	public void createAttrs(OdfOperationDoc operationDocument, OpAttrs attrs) {

		drawFrame.createAttrs(operationDocument, attrs, isContentAutoStyle());
	}
}
