/*
 *
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package org.docx4j.openpackaging.parts;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import javax.xml.bind.JAXBException;
import javax.xml.bind.annotation.XmlAccessType;
import javax.xml.bind.annotation.XmlAccessorType;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import javax.xml.stream.XMLStreamWriter;

@XmlAccessorType(XmlAccessType.NONE)
public abstract class DocumentSerialization implements IDocumentSerialization {

    final protected Map<String, String> prefixToUri;
    final protected Map<String, String> uriToPrefix;
    protected String version = "1.0";
    private Boolean isStandalone = Boolean.TRUE;
    private String characterEncodingScheme = null;
    protected String ignorable = null;

    public DocumentSerialization() {
        prefixToUri = new HashMap<String, String>();
        uriToPrefix = new HashMap<String, String>();
    }

    /* reads the root element of the document. Used prefixes and uris are saved and
     * will be written 1:1 when calling writeObject, this solves the problem we
     * had with prefixes used in the ignorable attribute and the corresponding uris. */

    @Override
    public void readObject(XMLStreamReader reader, DocumentPart<?> documentPart)
        throws JAXBException, XMLStreamException {

        final String v = reader.getVersion();
        if(v!=null) {
            version = v;
        }
        if(reader.standaloneSet()) {
            isStandalone = Boolean.valueOf(reader.isStandalone());
        }
        characterEncodingScheme = reader.getCharacterEncodingScheme();

        int tag = reader.nextTag();
        int namespaceCount = reader.getNamespaceCount();
        for(int i=0; i<namespaceCount; i++) {
            String prefix = reader.getNamespacePrefix(i);
            if(prefix==null) {
                prefix = "xmlns";
            }
            prefixToUri.put(prefix, reader.getNamespaceURI(i));
            uriToPrefix.put(reader.getNamespaceURI(i), prefix);
        }
        ignorable = reader.getAttributeValue("http://schemas.openxmlformats.org/markup-compatibility/2006", "Ignorable");
    }

    static final private String[] ns = {
        "r", "http://schemas.openxmlformats.org/officeDocument/2006/relationships",
        "mc", "http://schemas.openxmlformats.org/markup-compatibility/2006",
        "v", "urn:schemas-microsoft-com:vml",
        "w10", "urn:schemas-microsoft-com:office:word",
        "w", "http://schemas.openxmlformats.org/wordprocessingml/2006/main",
        "sl", "http://schemas.openxmlformats.org/schemaLibrary/2006/main",
        "w15", "http://schemas.microsoft.com/office/word/2012/wordml",
        "w16se", "http://schemas.microsoft.com/office/word/2015/wordml/symex",
        "w14", "http://schemas.microsoft.com/office/word/2010/wordml",
        "m", "http://schemas.openxmlformats.org/officeDocument/2006/math",
        "o", "urn:schemas-microsoft-com:office:office",
        "xsi", "http://www.w3.org/2001/XMLSchema-instance",
        "dsp", "http://schemas.microsoft.com/office/drawing/2008/diagram",
        "pic", "http://schemas.openxmlformats.org/drawingml/2006/picture",
        "dgm", "http://schemas.openxmlformats.org/drawingml/2006/diagram",
        "xdr", "http://schemas.openxmlformats.org/drawingml/2006/spreadsheetDrawing",
        "c", "http://schemas.openxmlformats.org/drawingml/2006/chart",
        "a", "http://schemas.openxmlformats.org/drawingml/2006/main",
        "wne", "http://schemas.microsoft.com/office/word/2006/wordml",
        "wp", "http://schemas.openxmlformats.org/drawingml/2006/wordprocessingDrawing",
        "wps", "http://schemas.microsoft.com/office/word/2010/wordprocessingShape",
        "cx1", "http://schemas.microsoft.com/office/drawing/2014/chartex",
        "wpc", "http://schemas.microsoft.com/office/word/2010/wordprocessingCanvas",
        "wpg", "http://schemas.microsoft.com/office/word/2010/wordprocessingGroup",
        "", "http://schemas.openxmlformats.org/drawingml/2006/lockedCanvas",
        "", "http://schemas.openxmlformats.org/drawingml/2006/compatibility",
        "", "http://schemas.openxmlformats.org/officeDocument/2006/bibliography",
        "", "http://schemas.microsoft.com/office/2006/coverPageProps",
        "", "http://schemas.microsoft.com/office/drawing/2012/chartStyle",
        "", "http://schemas.openxmlformats.org/drawingml/2006/chartDrawing",
        "", "http://schemas.microsoft.com/office/drawing/2010/main",
        "", "urn:schemas-microsoft-com:office:excel",
        "", "urn:schemas-microsoft-com:office:powerpoint"
    };

    @Override
    public void writeObject(XMLStreamWriter writer, DocumentPart<?> documentPart)
        throws XMLStreamException, JAXBException {

        final Iterator<Entry<String, String>> namespaceIter = prefixToUri.entrySet().iterator();
        while(namespaceIter.hasNext()) {
            final Entry<String, String> namespace = namespaceIter.next();
            final String u = namespace.getValue();
            final String p = namespace.getKey();
            if(p.equals("xmlns")) {
                writer.writeDefaultNamespace(u);
            }
            else {
                writer.writeNamespace(p, u);
            }
        }
        if(!prefixToUri.containsKey("xmlns")) {
            writer.writeDefaultNamespace("");
        }

        // ensure to use namespaces which are defined in package infos,
        // so the fragment writer will not write namespaces each time
        int nsPrefixNumber = 1;
        for(int i=0;i<ns.length;i++) {
            final String prefix = ns[i++];
            final String namespace = ns[i];
            if(!uriToPrefix.containsKey(namespace)) {
                if(!prefix.isEmpty()&&!prefixToUri.containsKey(prefix)) {
                    writer.writeNamespace(prefix, namespace);
                }
                else {
                    while(true) {
                        final String nsPrefix = "ns" + Integer.valueOf(nsPrefixNumber++).toString();
                        if(!prefixToUri.containsKey(nsPrefix)) {
                            writer.writeNamespace(nsPrefix, namespace);
                            break;
                        }
                    }
                }
            }
        }
        if(ignorable!=null) {
            writer.writeAttribute("http://schemas.openxmlformats.org/markup-compatibility/2006", "Ignorable", ignorable);
        }
    }
}
