/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package com.openexchange.office.rest;

import static com.openexchange.java.Strings.isEmpty;
import java.io.IOException;
import java.io.InputStream;
import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import com.openexchange.ajax.container.FileHolder;
import com.openexchange.ajax.requesthandler.AJAXRequestData;
import com.openexchange.ajax.requesthandler.AJAXRequestResult;
import com.openexchange.ajax.requesthandler.DispatcherNotes;
import com.openexchange.conversion.DataProperties;
import com.openexchange.conversion.SimpleData;
import com.openexchange.documentconverter.IDocumentConverter;
import com.openexchange.exception.OXException;
import com.openexchange.java.Streams;
import com.openexchange.office.document.LocalFileMappingManager;
import com.openexchange.office.tools.doc.ExtensionHelper;
import com.openexchange.office.tools.files.FileHelper;
import com.openexchange.office.tools.osgi.ServiceLookupRegistry;
import com.openexchange.preview.PreviewDocument;
import com.openexchange.preview.PreviewExceptionCodes;
import com.openexchange.preview.PreviewOutput;
import com.openexchange.preview.PreviewService;
import com.openexchange.server.ServiceLookup;
import com.openexchange.tools.servlet.AjaxExceptionCodes;
import com.openexchange.tools.session.ServerSession;


/**
 * {@link TemplatePreviewAction}
 *
 * @author <a href="mailto:carsten.driesner@open-xchange.com">Carsten Driesner</a>
 */
/*
@Action(method = RequestMethod.GET, name = "templatepreview", description = "Request to receive a list of possible template files.", parameters = {
    @Parameter(name = "session", description = "A session ID previously obtained from the login module."),
    @Parameter(name = "id", description = "The id of the file to get a preview from."),
    @Parameter(name = "folder", description = "The id of the folder where the file is located."),
    @Parameter(name = "mimeType", optional=true, description = "The mime type of the file referenced by id and folder if available."),
    @Parameter(name = "format", description = "Describes what format should be used for the preview."),
    @Parameter(name = "width", description = "Describes the requested width of the preview image."),
    @Parameter(name = "height", description = "Describes the requested height of the preview image."),
    @Parameter(name = "delivery", description = "Specifies for what purpose the preview is requested."),
    @Parameter(name = "scaleType", optional = true, description = "Specifies the scaling to be used for the preview generation."),
    @Parameter(name = "previewType", optional = true, description = "Specifies what type of preview should by rendered.")
}, responseDescription = "Response with timestamp: A JSON object containing the list of templates or a certain type.")
*/
@DispatcherNotes(defaultFormat = "file", allowPublicSession = true)
public class TemplatePreviewAction extends DocumentRESTAction {

    private static final Log LOG = com.openexchange.log.Log.loggerFor(TemplatePreviewAction.class);
    private static final long TEMPLATEPREVIEW_CACHE_EXPIRES = 10 * 60 * 60 * 1000; // 10 hours

    private LocalFileMappingManager m_localFileMapper = null;

    /**
     * Initializes a new {@link TemplatePreviewAction}.
     *
     * @param services
     *  The service lookup instance to be used by the TemplatePreviewAction.
     *
     * @param localFileMapper
     *  The local file mapping manager to be used by the TemplatePreviewAction
     *  instance to support previews for local files.
     */
    public TemplatePreviewAction(ServiceLookup servicesDEPRECATED, LocalFileMappingManager localFileMapper) {
        super(null);

        m_localFileMapper = localFileMapper;
    }

    /* (non-Javadoc)
     * @see com.openexchange.ajax.requesthandler.AJAXActionService#perform(com.openexchange.ajax.requesthandler.AJAXRequestData, com.openexchange.tools.session.ServerSession)
     */
    @Override
    public AJAXRequestResult perform(AJAXRequestData requestData, ServerSession session) throws OXException {
        AJAXRequestResult requestResult = null;
        final String id = requestData.getParameter("id");
        final String folder = requestData.getParameter("folder");

        if ((null != session) && (null != id) && (null != folder)) {

            // we have to check if we received a ID referencing a local file or
            // a file stored in OX Drive.
            if (id.startsWith(LocalFileMappingManager.prefixID)) {
                // Local template file
                final String fullFilePath = m_localFileMapper.getPathFromId(id);

                if (null != fullFilePath) {
                    final InputStream stream = m_localFileMapper.getInputStreamFromID(id);

                    if (null != stream) {
                        final PreviewService previewService = ServiceLookupRegistry.get().getService(PreviewService.class);

                        if (null != previewService) {
                            final String mimeType = requestData.getParameter("mimeType");
                            final String fileName = FileHelper.getFileName(fullFilePath);
                            final DataProperties dataProperties = new DataProperties(7);
                            InputStream thumbnail = null;

                            try {
                                String delivery = requestData.getParameter("delivery");
                                String scaleType = requestData.getParameter("scaleType");
                                String previewType = requestData.getParameter("previewType");
                                String contentType = getContentType(fileName, mimeType);
                                String fileNameStr = (null != fileName) ? fileName : "\"null\"";

                                LOG.debug("TemplatePreview: fullFilePath=" + ((null != fullFilePath) ? fullFilePath : "null") + ", fileName=" + fileNameStr + ", contentType=" + ((null != contentType) ? contentType : "null"));
                                dataProperties.put(DataProperties.PROPERTY_CONTENT_TYPE, contentType);
                                dataProperties.put(DataProperties.PROPERTY_NAME, fileName);
                                dataProperties.put("PreviewType", (previewType == null) ? "Thumbnail" : previewType);
                                dataProperties.put("PreviewWidth", requestData.getParameter("width"));
                                dataProperties.put("PreviewHeight", requestData.getParameter("height"));
                                dataProperties.put("PreviewDelivery", (delivery == null) ? "view" : delivery);
                                dataProperties.put("PreviewScaleType", (scaleType == null) ? "contain" : scaleType);

                                PreviewDocument previewDocument = null;

                                try {
                                    previewDocument = previewService.getPreviewFor(new SimpleData<>(stream, dataProperties), PreviewOutput.IMAGE, session, 1);
                                } catch (OXException e) {
                                    if (null == ServiceLookupRegistry.get().getService(IDocumentConverter.class)) {
                                        throw AjaxExceptionCodes.UNEXPECTED_ERROR.create("DocumentConverterService is not available");
                                    }

                                    throw e;
                                }

                                thumbnail = previewDocument.getThumbnail();
                                if (null == thumbnail) {
                                    // No thumbnail available
                                    throw PreviewExceptionCodes.THUMBNAIL_NOT_AVAILABLE.create("Preview image has not been generated in time");
                                }

                                // Prepare response
                                if(previewDocument.getClass().getName().equals("com.openexchange.documentpreview.OfficePreviewDocument")) {
                                    requestData.putParameter("transformationNeeded", "false");
                                }

                                final byte[] bytes = Streams.stream2bytes(thumbnail);
                                final FileHolder responseFileHolder = new FileHolder(Streams.newByteArrayInputStream(bytes), bytes.length, "image/jpeg", "image.jpg");
                                requestResult = new AJAXRequestResult();
                                requestResult.setResultObject(responseFileHolder, "preview_image");
                                requestResult.setFormat("preview_image");
                                requestResult.setExpires(TEMPLATEPREVIEW_CACHE_EXPIRES);
                                requestResult.setHeader("ETag", id);
                            } catch (IOException e) {
                                throw AjaxExceptionCodes.IO_ERROR.create(e, e.getMessage());
                            } catch (final RuntimeException e) {
                                throw AjaxExceptionCodes.UNEXPECTED_ERROR.create(e, e.getMessage());
                            } finally {
                                IOUtils.closeQuietly(thumbnail);
                            }
                        } else {
                            throw AjaxExceptionCodes.UNEXPECTED_ERROR.create("PreviewService is not available.");
                        }
                    } else {
                        throw AjaxExceptionCodes.UNEXPECTED_ERROR.create("Template file could not be read.");
                    }
                } else {
                    throw AjaxExceptionCodes.INVALID_PARAMETER_VALUE.create("id", "Unkown template file id");
                }
            } else {
                throw AjaxExceptionCodes.INVALID_PARAMETER_VALUE.create("id", "Template preview should only be called for global template files.");
            }
        } else {
            throw AjaxExceptionCodes.MISSING_PARAMETER.create("Session, id or folder parameters are missing or have invalid values.");
        }

        if (null == requestResult) {
            requestData.setFormat("json");
            requestResult = new AJAXRequestResult();
        }

        return requestResult;
    }

    /**
     * Provide the content type (mime type) based on the provided contentType
     * or fileName.
     *
     * @param fileName
     *  The file name of the template file including the extension.
     *
     * @param contentType
     *  A mime type of the template file or null if unknown.
     *
     * @return
     *  The content type (mime type) of the template file based on the provided
     *  content type of file name. Can be null if the content type was not
     *  recognized.
     */
    private static String getContentType(final String fileName, final String contentType) {
        final String fallbackContentType = "application/octet-stream";

        String mimeType = contentType;
        if (isEmpty(contentType)) {
            // Determine Content-Type by file name
            String ext = FileHelper.getExtension(fileName);
            mimeType = ExtensionHelper.getMimeTypeFromTemplateExtension(ext);
        }
        return (null == mimeType) ? fallbackContentType : mimeType;
    }
}
