/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.rest;

import java.io.InputStream;
import java.util.Map;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import com.openexchange.ajax.requesthandler.AJAXRequestData;
import com.openexchange.ajax.requesthandler.AJAXRequestResult;
import com.openexchange.config.ConfigurationService;
import com.openexchange.exception.OXException;
import com.openexchange.file.storage.File;
import com.openexchange.file.storage.FileStorageFileAccess;
import com.openexchange.file.storage.composition.IDBasedFileAccess;
import com.openexchange.office.document.ChunkableDocLoader;
import com.openexchange.office.document.DocFileHelper;
import com.openexchange.office.document.ImExportHelper;
import com.openexchange.office.document.OXDocument;
import com.openexchange.office.document.OXDocumentHelper;
import com.openexchange.office.filter.api.FilterException;
import com.openexchange.office.filter.api.IImporter;
import com.openexchange.office.filter.core.FilterExceptionToErrorCode;
import com.openexchange.office.htmldoc.GenericHtmlDocumentBuilder;
import com.openexchange.office.imagemgr.ResourceManager;
import com.openexchange.office.message.MessagePropertyKey;
import com.openexchange.office.message.OperationHelper;
import com.openexchange.office.rest.job.AsyncJobWorker;
import com.openexchange.office.rest.operations.GetOperationsJob;
import com.openexchange.office.rest.operations.OperationData;
import com.openexchange.office.rest.operations.OperationDataAccess;
import com.openexchange.office.rest.operations.OperationDataCache;
import com.openexchange.office.rest.operations.OperationTools;
import com.openexchange.office.rest.operations.OperationsResult;
import com.openexchange.office.rest.tools.ConversionHelper;
import com.openexchange.office.rest.tools.EncryptionInfo;
import com.openexchange.office.rest.tools.ConversionHelper.ConversionData;
import com.openexchange.office.rest.tools.HttpStatusCode;
import com.openexchange.office.rest.tools.ParamValidator;
import com.openexchange.office.tools.common.memory.MemoryObserver;
import com.openexchange.office.tools.common.system.SystemInfoHelper;
import com.openexchange.office.tools.common.system.SystemInfoHelper.MemoryInfo;
import com.openexchange.office.tools.doc.ApplicationType;
import com.openexchange.office.tools.doc.DocumentFormat;
import com.openexchange.office.tools.doc.DocumentFormatHelper;
import com.openexchange.office.tools.doc.ExtensionHelper;
import com.openexchange.office.tools.doc.StreamInfo;
import com.openexchange.office.tools.doc.Sync;
import com.openexchange.office.tools.error.ErrorCode;
import com.openexchange.office.tools.error.ExceptionToErrorCode;
import com.openexchange.office.tools.files.FileHelper;
import com.openexchange.office.tools.files.StorageHelper;
import com.openexchange.office.tools.osgi.ServiceLookupRegistry;
import com.openexchange.office.tools.user.AuthorizationCache;
import com.openexchange.server.ServiceLookup;
import com.openexchange.tools.session.ServerSession;

public class GetOperationsAction extends DocumentRESTAction {

    // ---------------------------------------------------------------
    @SuppressWarnings("deprecation")
    private static final Log LOG = com.openexchange.log.Log.loggerFor(GetOperationsAction.class);

    // ---------------------------------------------------------------
    private final static String[] aMandatoryParams       = { ParameterDefinitions.PARAM_FILE_ID,
                                                             ParameterDefinitions.PARAM_FOLDER_ID };

    private static final int      MAX_DOCUMENT_FILE_SIZE = 100; // 100 MB

    // ---------------------------------------------------------------
    private AsyncJobWorker<GetOperationsJob> aGetOpsAsyncWorker = new AsyncJobWorker<>(1);

    // ---------------------------------------------------------------
    public GetOperationsAction(ServiceLookup servicesDEPRECATED) {
        super(servicesDEPRECATED);
    }

    // ---------------------------------------------------------------
    @Override
    public AJAXRequestResult perform(AJAXRequestData requestData, ServerSession session) throws OXException {
        AJAXRequestResult aResult = null;

        if (!ParamValidator.areAllParamsNonEmpty(requestData, aMandatoryParams))
            return ParamValidator.getResultFor(HttpStatusCode.BAD_REQUEST.getStatusCode());

        final String folderId = requestData.getParameter(ParameterDefinitions.PARAM_FOLDER_ID);
        final String fileId   = requestData.getParameter(ParameterDefinitions.PARAM_FILE_ID);
        final String authCode = requestData.getParameter(ParameterDefinitions.PARAM_AUTH_CODE);

        if (!ParamValidator.canReadFile(session, folderId, fileId))
            return ParamValidator.getResultFor(HttpStatusCode.FORBIDDEN.getStatusCode());

        String subAction = requestData.getParameter(ParameterDefinitions.PARAM_SUB_ACTION);
        subAction = (null == subAction) ? ParameterDefinitions.PARAM_SUBACTION_OPEN : subAction;

        String encryptionInfo = null;
        String cryptoAction   = "";

        if (StringUtils.isNotEmpty(authCode)) {
            cryptoAction = "Decrypt";
            encryptionInfo = EncryptionInfo.createEncryptionInfo(session, authCode);
        }

        final IDBasedFileAccess fileAccess = getFileAccess(session, requestData, cryptoAction);
        if (null == fileAccess)
            return new AJAXRequestResult(ErrorCode.GENERAL_ARGUMENTS_ERROR.getAsJSONResultObject());

        JSONObject jsonResult = new JSONObject();
        ErrorCode  errorCode  = ErrorCode.NO_ERROR;

        try {
            if (subAction.equalsIgnoreCase(ParameterDefinitions.PARAM_SUBACTION_OPEN)) {
                final String        version     = (null == requestData.getParameter(ParameterDefinitions.PARAM_VERSION)) ? FileStorageFileAccess.CURRENT_VERSION : requestData.getParameter(ParameterDefinitions.PARAM_VERSION);
                      OperationData aCachedData = OperationDataCache.getInstance().get(fileId, version, session.getContextId());

                // check up-to-date state of the cached data - if not set to null and ensure to load latest document data
                if (null != aCachedData)
                    aCachedData = getHashForDocument(fileAccess, fileId, version).equals(aCachedData.getHash()) ? aCachedData : null;

                if (null == aCachedData) {
                    // cache miss or cache entry dirty
                    final OperationsResult aDocOpsResult = getOperationsFromDocument(fileAccess, session, folderId, fileId, version, encryptionInfo);

                    errorCode = aDocOpsResult.getErrorCode();
                    if (errorCode.isNoError()) {
                        jsonResult = getJsonResult(aDocOpsResult, folderId, fileId);

                        final String        key  = OperationDataCache.getKeyFromProps(fileId, version, session.getContextId());
                        final OperationData data = new OperationData(key,
                                                                     aDocOpsResult.getHash(),
                                                                     aDocOpsResult.getHtmlDoc(),
                                                                     aDocOpsResult.getPreviewData(),
                                                                     aDocOpsResult.getOperations());
                        OperationDataCache.getInstance().put(key, data);
                    }

                    LOG.info("GetOperationsAction: Document " + fileId + " in context " + session.getContextId() + " was not found in cache");
                } else {
                    // cache hit
                    jsonResult = getJsonResult(aCachedData, folderId, fileId);

                    LOG.info("GetOperationsAction: Document " + fileId + " in context " + session.getContextId() + " found in cache");
                }

                jsonResult.put(MessagePropertyKey.KEY_ERROR_DATA, errorCode.getAsJSON());

                if (StringUtils.isNotEmpty(encryptionInfo)) {
                    AuthorizationCache.addKey(String.valueOf(session.getContextId()), String.valueOf(session.getUserId()), fileId, authCode);
                }

                aResult = new AJAXRequestResult(jsonResult);
            } else if (subAction.equalsIgnoreCase(ParameterDefinitions.PARAM_SUBACTION_CLOSE)) {
                AuthorizationCache.removeKey(String.valueOf(session.getContextId()), String.valueOf(session.getUserId()), fileId);
            } else if (subAction.equalsIgnoreCase(ParameterDefinitions.PARAM_SUBACTION_PREFETCH)) {
                if (StringUtils.isEmpty(authCode)) {
                    // no pre-fetch for encrypted documents
                    final String version = (null == requestData.getParameter(ParameterDefinitions.PARAM_VERSION)) ? FileStorageFileAccess.CURRENT_VERSION : requestData.getParameter(ParameterDefinitions.PARAM_VERSION);

                    if (!OperationDataCache.getInstance().hasEntryAndTouch(fileId, version, session.getContextId()))
                        aGetOpsAsyncWorker.addJob(new GetOperationsJob(session, folderId, fileId, version));
                }
                aResult = ParamValidator.getResultFor(HttpStatusCode.OK.getStatusCode());
            } else {
                return ParamValidator.getResultFor(HttpStatusCode.BAD_REQUEST.getStatusCode());
            }
        }
        catch (final OXException e) {
            errorCode = ExceptionToErrorCode.map(e, ErrorCode.GENERAL_UNKNOWN_ERROR, false);
            try {jsonResult.put(MessagePropertyKey.KEY_ERROR_DATA, errorCode.getAsJSON()); } catch (final JSONException je) { /* ignore it */ }
            aResult = new AJAXRequestResult(jsonResult);

            LOG.error("GetOperationsAction: Exception caught trying to generate operations for provided document", e);
        }
        catch (final Exception e) {
            errorCode = ErrorCode.GENERAL_UNKNOWN_ERROR;
            try {jsonResult.put(MessagePropertyKey.KEY_ERROR_DATA, errorCode.getAsJSON()); } catch (final JSONException je) { /* ignore it */ }
            aResult = new AJAXRequestResult(jsonResult);

            LOG.error("GetOperationsAction: Exception caught trying to generate operations for provided document", e);
        }

        return aResult;
    }

    // ---------------------------------------------------------------
    private static JSONObject getJsonResult(final OperationDataAccess aOpsData, String folderId, String fileId) throws JSONException {
        final JSONObject jsonResult = new JSONObject();

        jsonResult.put(MessagePropertyKey.KEY_OPERATIONS, aOpsData.getOperations());
        if (null != aOpsData.getPreviewData())
            jsonResult.put(MessagePropertyKey.KEY_PREVIEW, aOpsData.getPreviewData());
        if (null != aOpsData.getHtmlDoc())
            jsonResult.put(MessagePropertyKey.KEY_HTMLDOCUMENT, aOpsData.getHtmlDoc());

        // Add sync info to result to be compatible with normal join response
        // as we don't support local storage load always provide version 1 and
        // document-osn = -1 (set by default).
        final Sync aSyncInfo = new Sync(LOG, folderId, fileId);
        aSyncInfo.updateFileVersion(Sync.SYNC_FIRST_VERSION);
        jsonResult.put(Sync.SYNC_INFO, aSyncInfo.toJSON());

        return jsonResult;
    }

    // ---------------------------------------------------------------
    public void shutdown() {
        aGetOpsAsyncWorker.shutdown();
    }

    // ---------------------------------------------------------------
    public static OperationsResult getOperationsFromDocument(final IDBasedFileAccess fileAccess, final ServerSession session, String folderId, String fileId, String version, String encryptionInfo) throws Exception {
        final File   metaData = fileAccess.getFileMetadata(fileId, version);
        final String fileName = metaData.getFileName();
        final String mimeType = metaData.getFileMIMEType();
        final long   fileSize = metaData.getFileSize();

        InputStream     documentStream = null;
        String          extension      = FileHelper.getExtension(fileName, true);
        DocumentFormat  docFormat      = DocumentFormat.NONE;
        ErrorCode       errorCode      = ErrorCode.NO_ERROR;

        if (fileSize < maxDocumentFileSize() && isMemConsumptionForDocumentOk(fileSize, encryptionInfo)) {
            final Map<String,String> conversionFormat = DocumentFormatHelper.getConversionFormatInfo(mimeType, extension);

            if (null != conversionFormat) {
                final ConversionData aConversionData = ConversionHelper.convertDocument(session, fileId, folderId, version, fileName, conversionFormat, encryptionInfo);

                errorCode      = aConversionData.getErrorCode();
                documentStream = aConversionData.getInputStream();
                extension      = aConversionData.getExtension();
                docFormat      = ExtensionHelper.getDocumentFormatFromExtension(extension);
            }
            else {
                final StorageHelper storeHelper = new StorageHelper(null, session, folderId);
                final StreamInfo    streamInfo  = DocFileHelper.getDocumentStream(null, session, folderId, fileId, version, storeHelper, encryptionInfo);

                errorCode      = streamInfo.getErrorCode();
                documentStream = streamInfo.getDocumentStream();
                docFormat      = streamInfo.getDocumentFormat();
            }

            if (errorCode.isNoError() && (null != documentStream)) {
                return loadDocumentFromStream(session, documentStream, docFormat, metaData);
            }

            errorCode = errorCode.isNoError() ? ErrorCode.GENERAL_UNKNOWN_ERROR : errorCode;
        } else {
            errorCode = ErrorCode.LOADDOCUMENT_COMPLEXITY_TOO_HIGH_ERROR;
        }

        return new OperationsResult(errorCode);
    }

    // ---------------------------------------------------------------
    public static OperationsResult loadDocumentFromStream(final ServerSession session, final InputStream documentStream, DocumentFormat docFormat, final File metaData ) throws Exception {
        final ApplicationType appType = ApplicationType.documentFormatToApplicationType(docFormat);

        switch (appType) {
            case APP_PRESENTATION:
            case APP_TEXT:         return loadDocument(session, documentStream, docFormat, appType, metaData);
            case APP_SPREADSHEET:  return loadSpreadsheetDocument(session, documentStream, docFormat, metaData);
            default:               throw new IllegalArgumentException("Unknown application type provided to ");
        }
    }

    // ---------------------------------------------------------------
    public static boolean isMemConsumptionForDocumentOk(long nFileSize, String encryptionInfo) {
        final MemoryObserver aMemObserver = MemoryObserver.getMemoryObserver();

        boolean bMemExceeded     = aMemObserver.isUsageThresholdExceeded();
        final MemoryInfo memInfo = SystemInfoHelper.getMemoryInfo();
        final float fCurrentFileSizeFactor = StringUtils.isEmpty(encryptionInfo) ? 1.0f : OXDocument.PGP_FILE_SIZE_FACTOR;
        return (!bMemExceeded && (((memInfo.maxHeapSize - memInfo.usedHeapSize) - Math.round(nFileSize * fCurrentFileSizeFactor)) > SystemInfoHelper.MINIMAL_FREE_BACKEND_HEAPSPACE));
    }

    // ---------------------------------------------------------------
    public static String getHashForDocument(final IDBasedFileAccess fileAccess, final String fileId, final String version) throws Exception {
        return fileAccess.getFileMetadata(fileId, version).getFileMD5Sum();
    }

    // ---------------------------------------------------------------
    private static OperationsResult loadDocument(final ServerSession session, final InputStream documentStream, DocumentFormat docFormat, ApplicationType appType, final File metaData) throws Exception {
        ErrorCode errorCode = ErrorCode.NO_ERROR;

        try {
            final IImporter        importer      = ImExportHelper.getImporterService(null, docFormat);
            final String           module        = ApplicationType.enumToString(appType);
            final OXDocumentHelper oxDocHelper   = new OXDocumentHelper(null, session, documentStream, new ResourceManager());
            final JSONObject       docOperations = oxDocHelper.getOperations(session, importer);
                  String           htmlDoc       = null;

            if (GenericHtmlDocumentBuilder.isFastLoadActive(module, session))
                htmlDoc = GenericHtmlDocumentBuilder.buildHtmlDocument(module, docOperations, metaData, session);

            OperationTools.setOperationStateNumberToOperations(oxDocHelper.getUniqueDocumentId(), docOperations);
            return new OperationsResult(errorCode, htmlDoc, null, getOperationsArray(docOperations), metaData.getFileMD5Sum());
        } catch (final FilterException e) {
            errorCode = handleFilterException(metaData.getId(), e);
        } finally {
            IOUtils.closeQuietly(documentStream);
        }

        return new OperationsResult(errorCode);
    }

    // ---------------------------------------------------------------
    private static OperationsResult loadSpreadsheetDocument(final ServerSession session, final InputStream documentStream, DocumentFormat docFormat, final File metaData) throws Exception {
        final OXDocument oxDocument = new OXDocument(session, null, documentStream, metaData.getFileName(), metaData.getFileMIMEType(), new ResourceManager(), null);
        final JSONObject operations = new JSONObject();

        // Initialize the error code retrieved by our OXDocument instance. The error code is
        // set by the ctor of the instance which tries to retrieve the document stream.
        ErrorCode errorCode = oxDocument.getLastError();

        // check error code after loading the document
        if (errorCode.isNoError()) {
            try {
                final IImporter importer = ImExportHelper.getImporterService(null, docFormat);
                if (importer != null) {
                    JSONObject               previewData    = null;
                    final ChunkableDocLoader chunkDocLoader = new ChunkableDocLoader(oxDocument, importer, new JSONObject());

                    chunkDocLoader.prepareLoad();

                    final JSONObject globalOps  = chunkDocLoader.getGlobalOperations();
                    final JSONObject activeOps  = chunkDocLoader.getActiveOperations();
                    final int        partsCount = chunkDocLoader.getPartsCount();

                    if (partsCount > 1) {
                        // we need to send the operations in two parts - possible rescue
                        // operations will be sent via the second part
                        previewData = globalOps;
                        previewData.put(MessagePropertyKey.KEY_ACTIVESHEET, chunkDocLoader.getActivePartIndex());
                        OperationHelper.appendJSON(previewData, activeOps);
                    } else {
                        OperationHelper.appendJSON(operations, globalOps);
                        OperationHelper.appendJSON(operations, activeOps);
                    }
                    OperationHelper.appendJSON(operations, chunkDocLoader.getRemainingOperations());
                    OperationTools.setOperationStateNumberToOperations(oxDocument.getUniqueDocumentId(), operations);

                    return new OperationsResult(errorCode, null, previewData, getOperationsArray(operations), metaData.getFileMD5Sum());
                }

                errorCode = ErrorCode.LOADDOCUMENT_NO_FILTER_FOR_DOCUMENT_ERROR;
            } catch (final FilterException e) {
                errorCode = handleFilterException(metaData.getId(), e);
            } catch (final Exception e) {
                errorCode = ErrorCode.LOADDOCUMENT_FAILED_ERROR;
                LOG.warn("GetOperationsActions: Exception caught", e);
            } finally {
                IOUtils.closeQuietly(documentStream);
            }
        }

        return new OperationsResult(errorCode);
    }

    // ---------------------------------------------------------------
    private static ErrorCode handleFilterException(final String fileId, FilterException e) {
        ErrorCode errorCode = ErrorCode.LOADDOCUMENT_CANNOT_RETRIEVE_OPERATIONS_ERROR;

        final FilterExceptionToErrorCode.LogLevel level = FilterExceptionToErrorCode.determineLogLevel(e);
        FilterExceptionToErrorCode.log(LOG, level, "GetOperationsActions caught filter exception. ", "document: " + fileId, e);

        if ((e.getErrorcode() == FilterException.ErrorCode.MEMORY_USAGE_TOO_HIGH) || (e.getErrorcode() == FilterException.ErrorCode.MEMORY_USAGE_MIN_FREE_HEAP_SPACE_REACHED)) {
            // try to figure out what happened with the memory usage of the filter as
            // we want to provide two different error codes.
            errorCode = ErrorCode.GENERAL_MEMORY_TOO_LOW_ERROR;
            if (e.getErrorcode() == FilterException.ErrorCode.MEMORY_USAGE_MIN_FREE_HEAP_SPACE_REACHED) {
                final SystemInfoHelper.MemoryInfo memInfo = SystemInfoHelper.getMemoryInfo();
                final String freeMemString = (memInfo == null) ? "unknown" : ((Long) ((memInfo.maxHeapSize - memInfo.usedHeapSize) / SystemInfoHelper.MBYTES)).toString();
                LOG.info("Document could not be loaded, because current available heap space is not sufficient. Current free heap space: " + freeMemString + " MB");
            }
        } else {
            // all other exceptions are mapped by general aspects
            errorCode = FilterExceptionToErrorCode.map(e, errorCode);
        }

        return errorCode;
    }

    // ---------------------------------------------------------------
    private static long maxDocumentFileSize() {
        final ConfigurationService configService = ServiceLookupRegistry.get().getService(ConfigurationService.class);
        long nMaxDocumentFileSize = MAX_DOCUMENT_FILE_SIZE * 1024L * 1024L;

        if (null != configService) {
            nMaxDocumentFileSize = configService.getIntProperty("com.openexchange.office.maxDocumentFileSize", MAX_DOCUMENT_FILE_SIZE) * 1024L * 1024L;
        }

        return nMaxDocumentFileSize;
    }

    // ---------------------------------------------------------------
    private static JSONArray getOperationsArray(final JSONObject opsObject) {
        return ((null == opsObject) || (null == opsObject.optJSONArray("operations"))) ? null : opsObject.optJSONArray("operations");
    }
}
