/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.rest;

import static org.apache.commons.lang3.StringUtils.isEmpty;
import static org.apache.commons.lang3.StringUtils.isNotEmpty;
import java.awt.Dimension;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.IOUtils;
import com.openexchange.ajax.requesthandler.AJAXRequestData;
import com.openexchange.ajax.requesthandler.AJAXRequestResult;
import com.openexchange.ajax.requesthandler.DispatcherNotes;
import com.openexchange.documentconverter.IDocumentConverter;
import com.openexchange.documentconverter.Properties;
import com.openexchange.exception.ExceptionUtils;
import com.openexchange.office.document.DocFileHelper;
import com.openexchange.office.imagemgr.Resource;
import com.openexchange.office.imagemgr.ResourceManager;
import com.openexchange.office.rest.tools.EncryptionInfo;
import com.openexchange.office.tools.doc.StreamInfo;
import com.openexchange.office.tools.files.FileHelper;
import com.openexchange.office.tools.files.StorageHelper;
import com.openexchange.office.tools.osgi.ServiceLookupRegistry;
import com.openexchange.office.tools.session.SessionUtils;
import com.openexchange.office.tools.user.AuthorizationCache;
import com.openexchange.server.ServiceLookup;
import com.openexchange.tools.session.ServerSession;

/**
 * {@link GetFileAction}
 *
 * @author <a href="mailto:firstname.lasname@open-xchange.com">Firstname Lastname</a>
 */
/*
 * MH/KA compile fix for buildsystem
 *
 * @Action(method = RequestMethod.GET, name = "getfile", description =
 * "Get the whole document as PDF file or get the substream of an infostore file in its naive format.", parameters = {
 *
 * @Parameter(name = "session", description = "A session ID previously obtained from the login module."),
 *
 * @Parameter(name = "id", description = "Object ID of the requested infoitem."),
 *
 * @Parameter(name = "folder_id", description = "Folder ID of the requested infoitem."),
 *
 * @Parameter(name = "uid", description = "The unique id of the client application."),
 *
 * @Parameter(name = "version", optional=true, description =
 * "If present, the infoitem data describes the given version. Otherwise the current version is returned."),
 *
 * @Parameter(name = "filter_format", optional=true, description =
 * "If this value is set to 'pdf', the whole document is as converted PDF file, in all other cases, the content of the document's substream, described by the parameter 'fragment, is returned as unmodified file."
 * ),
 *
 * @Parameter(name = "fragment", optional=true, description =
 * "If this value is set and the filter_format is not set or not set to 'pdf', this parameter describes the substream name of the document internal file to be returned."
 * ),
 *
 * @Parameter(name = "filename", optional=true, description =
 * "If present, this parameter contains the name of the infostore item to be used as initial part of the filename for the returned file, other the filename is set to 'file'."
 * ), }, responseDescription = "Response with timestamp: The content of the documents substream in its native format, or converted to PNG in
 * the case of EMF, WMF and SVM graphic file formats)
 */
@DispatcherNotes(defaultFormat = "file", allowPublicSession = true)
public class GetFileAction extends DocumentRESTAction {

    /**
     * Initializes a new {@link GetFileAction}.
     *
     * @param services
     * @param oqm
     */
    public GetFileAction(@SuppressWarnings("unused") ServiceLookup servicesDEPRECATED, ResourceManager resMgr) {
        super(null);
        resourceManager = resMgr;
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.ajax.requesthandler.AJAXActionService#perform(com.openexchange.ajax.requesthandler.AJAXRequestData,
     * com.openexchange.tools.session.ServerSession)
     */
    @Override
    public AJAXRequestResult perform(AJAXRequestData request, ServerSession session) {
        AJAXRequestResult requestResult = null;
        String resourceName = request.getParameter("get_filename");
        String getFormat = request.getParameter("get_format");
        String filename = "unknown file";
        final String fileId = request.getParameter("id");
        final String folderId = request.getParameter("folder_id");
        int userId = Integer.MIN_VALUE;

        // check fileId/folderId
        if (isEmpty(fileId) || isEmpty(folderId)) {
            requestResult = new AJAXRequestResult();
            requestResult.setHttpStatusCode(400);

            LOG.error(new StringBuilder("AjaxRequest GetFileAction: fileId/folderId not valid: ").
                append(fileId).append('/').append(folderId).toString());

            return requestResult;
        }

        // retrieve user id and check access rights for input file
        try {
            userId = SessionUtils.getUserId(session);
        } catch(Exception e) {
            LOG.error("AjaxRequest GetFileAction exception caught while trying to get user id", e);
        }

        if ((Integer.MIN_VALUE == userId) || !FileHelper.canReadFile(session, folderId, fileId, userId)) {
            final AJAXRequestResult result = new AJAXRequestResult();
            result.setHttpStatusCode(403);

            LOG.error("AjaxRequest GetFileAction error while checking file access rights");

            return result;
        }

        // check and sanitize resourceName
        if (isEmpty(resourceName)) {
            final AJAXRequestResult result = new AJAXRequestResult();
            result.setHttpStatusCode(400);

            LOG.error("AjaxRequest GetFileAction error while checking resource name");

            return result;
        } else  if (resourceName.startsWith("./")) {
            resourceName = resourceName.substring(2);
        }

        filename = FileHelper.getBaseName(resourceName);

        LOG.debug("AjaxRequest GetFileAction request for " + filename);

        if (!"json".equals(getFormat)) {
            getFormat = "file";
        }

        // try to get the graphic from the resource manager(s) first
        if (filename.startsWith("uid")) {
            // 1. try: resource is already contained within this nodes resource manager
            final long uid = Resource.getUIDFromResourceName(filename);
            final Resource nodeResource = resourceManager.getResource(uid);
            byte[] resdata = ((nodeResource != null) ? nodeResource.getBuffer() : null);

            LOG.trace("AjaxRequest GetFileAction resource data " + filename + " not found in resourceManager.");

            // 2. try: create a managed resource with the given uid
            if (null == resdata) {
                try {
                    final String managedId = Resource.getManagedIdFromUID(uid);

                    if ((null != managedId) && (managedId.length() > 0)) {
                        final Resource managedResource = Resource.createFrom(null, managedId);

                        // add a new managed resource to this nodes resource manager for later speedup
                        if (null != (resdata = ((null != managedResource) ? managedResource.getBuffer() : null))) {
                            resourceManager.addResource(managedResource);
                        }
                    }
                } catch (Exception e) {
                    // This is just an information: a cleanly loaded document contains the images
                    // and does not provide it via managed file. The 3rd access should provide the
                    // image. Therefore change this LOG into info.
                    LOG.debug("AjaxRequest GetFileAction exception caught while trying to create managed resource.",  e);
                }
            }

            // if we've got the resource data, set it as request result
            if (null != resdata) {
                requestResult = createFileRequestResult(request, resdata, getFormat, DocFileHelper.getMimeType(resourceName), resourceName);
            }
        }

        // 3. try getting the graphic from the document's zip archive
        if (null == requestResult) {
            LOG.trace("AjaxRequest GetFileAction image must be retrieved from document stream");

            final StorageHelper storageHelper = new StorageHelper(null, session, folderId);
            final String authCode = AuthorizationCache.getKey(""+session.getContextId(), ""+session.getUserId(), fileId);
            String encryptionInfo = "";

            // always use the current version of the document
            request.putParameter("version", "0");

            if (null != authCode) {
                encryptionInfo = EncryptionInfo.createEncryptionInfo(session, authCode);
            }

            try (final StreamInfo streamInfo = DocFileHelper.getDocumentStream(null, session, request, storageHelper, encryptionInfo);
                 final InputStream documentStm = streamInfo.getDocumentStream()) {

                if (null != documentStm) {
                    LOG.trace("AjaxRequest GetFileAction document stream successfully retrieved - try to get substream for resource data " + filename);
                    final String[] mimeType = { "" };

                    try (final InputStream resultStm = getZipEntryStream(documentStm, session, resourceName, mimeType, getImageExtents(request))) {
                        if (null != resultStm) {
                            byte[] buffer = IOUtils.toByteArray(resultStm);

                            if (null != buffer) {
                                requestResult = createFileRequestResult(request, buffer, getFormat, mimeType[0], filename);

                                LOG.trace("AjaxRequest GetFileAction resource data " + filename + " successfully retrieved.");
                            } else {
                                LOG.warn("AjaxRequest GetFileAction resource data " + filename + " not found.");
                            }
                        }
                    }
                }
            } catch (IOException e) {
                LOG.error("AjaxRequest GetFileAction exception caught while reading data from document stream.", e);
            }
        }

        if (null == requestResult) {
            LOG.warn("AjaxRequest: GetFileAction request not successfully processed for resource: " + ((null == resourceName) ? "unknown resource name" : resourceName) + " in: " + filename);

            request.setFormat("json");
            requestResult = new AJAXRequestResult();
        }

        return requestResult;
    }

    /**
     * @param documentStm
     * @param resourceName
     * @param mimeType
     * @param closeDocumentStream
     * @return
     */
    final private InputStream getZipEntryStream(InputStream documentStm, ServerSession session, String resourceName, String[] mimeType, Dimension extents) {
        InputStream resultStm = null;

        if (null != documentStm) {
            final String filename = FileHelper.getBaseName(resourceName);
            String uidResourceName = null;

            // check for uid Resources within all entries
            if (filename.startsWith("uid") && (filename.length() > 3)) {
                uidResourceName = filename;
            }

            try (final ZipInputStream zipInputStm = new ZipInputStream(documentStm)) {

                for (ZipEntry zipEntry = null; (null == resultStm) && ((zipEntry = zipInputStm.getNextEntry()) != null);) {
                    try {
                        final String curEntryName = zipEntry.getName();

                        if (curEntryName.equals(DocFileHelper.OX_RESCUEDOCUMENT_DOCUMENTSTREAM_NAME)) {
                            // check for resources contained within the original rescue document
                            byte[] entryBuffer = IOUtils.toByteArray(zipInputStm);

                            if (null != entryBuffer) {
                                try (final InputStream entryInputStream = new ByteArrayInputStream(entryBuffer)) {
                                    resultStm = getZipEntryStream(entryInputStream, session, resourceName, mimeType, extents);
                                }
                            }
                        } else if (curEntryName.equals(resourceName) || ((null != uidResourceName && curEntryName.endsWith(uidResourceName)))) {
                            // check for resources equal the the given name and for resources matching the uid resource manager naming schema
                            byte[] entryBuffer = null;

                            try {
                                entryBuffer = IOUtils.toByteArray(zipInputStm);
                            } catch (IOException e) {
                                LOG.error("AjaxRequest GetFileAction.getZipEntryStream exception caught while trying to read data from document stream.", e);
                            }

                            if (null != entryBuffer) {
                                final String fileExtension = FilenameUtils.getExtension(resourceName).trim().toLowerCase();

                                // check if we create a temporary png file that can be displayed within the browser,
                                // we will do this for the formats "wmf and emf... TODO: the resulting graphic should
                                // somehow be part of the resourcemanager to prevent unnecessary graphic conversions
                                if (fileExtension.equals("emf") || fileExtension.equals("wmf") || (resourceName.indexOf("ObjectReplacements/") > -1)) {
                                    final IDocumentConverter documentConverter = ServiceLookupRegistry.get().getService(IDocumentConverter.class);

                                    if (null != documentConverter) {
                                        final HashMap<String, Object> jobProperties = new HashMap<>(8);
                                        final HashMap<String, Object> resultProperties = new HashMap<>(8);

                                        try (final InputStream inputStm = new ByteArrayInputStream(entryBuffer)) {
                                            jobProperties.put(Properties.PROP_INPUT_STREAM, inputStm);
                                            jobProperties.put(Properties.PROP_INPUT_TYPE, fileExtension);
                                            jobProperties.put(Properties.PROP_MIME_TYPE, "image/png");
                                            jobProperties.put(Properties.PROP_INFO_FILENAME, resourceName);

                                            if (null != extents) {
                                                jobProperties.put(Properties.PROP_PIXEL_WIDTH, Integer.valueOf(extents.width));
                                                jobProperties.put(Properties.PROP_PIXEL_HEIGHT, Integer.valueOf(extents.height));
                                            }

                                            // this is a user request in every case
                                            jobProperties.put(Properties.PROP_USER_REQUEST, Boolean.TRUE);

                                            resultStm = documentConverter.convert("graphic", jobProperties, resultProperties);
                                            mimeType[0] = (String) resultProperties.get(Properties.PROP_RESULT_MIME_TYPE);
                                        }
                                    }
                                } else {
                                    // TODO (KA): consider providing the following params for IC request (2017-11-02)
                                    // final int maxImageSize = ConfigurationHelper.getIntegerOfficeConfigurationValue(null, session, "//module/documentImageSize", 1280);
                                    // final int jpegCompression = ConfigurationHelper.getIntegerOfficeConfigurationValue(null, session, "//module/documentImageQuality", 75);
                                }

                                if (null == resultStm) {
                                    resultStm = new ByteArrayInputStream(entryBuffer);
                                    mimeType[0] = DocFileHelper.getMimeType(resourceName);
                                }
                            }
                        }
                    } finally {
                        zipInputStm.closeEntry();
                    }
                }
            } catch (Throwable e) {
	        	ExceptionUtils.handleThrowable(e);
                LOG.error("AjaxRequest GetFileAction.getZipEntryStream exception caught while trying to read data from document stream.", e);
            }
        }

        // !!! Possible resource leak warning has been checked:
        // returned resultStm is closed by caller of this method
        return resultStm;
    }

    /**
     * @param request
     * @return
     */
    final private static Dimension getImageExtents(final AJAXRequestData request) {
        // get optional extents (width/height)
        try {
            final String widthParam = request.getParameter("width");
            final String heightParam = request.getParameter("height");
            int width = -1;
            int height = -1;

            if (isNotEmpty(widthParam)) {
                width = Integer.parseInt(widthParam);
            }

            if (isNotEmpty(heightParam)) {
                height = Integer.parseInt(heightParam);
            }

            if ((width != 0) || (height != 0)) {
                return new Dimension((width > 0) ? width : -1, (height > 0) ? height : -1);
            }
        } catch (NumberFormatException e) {
            LOG.error("AjaxRequest GetFileAction exception caught while parsing width/height parameters", e);
        }

        return null;

    }

    // - Members ---------------------------------------------------------------

    static protected final org.apache.commons.logging.Log LOG = com.openexchange.log.LogFactory.getLog(GetFileAction.class);

    private ResourceManager resourceManager = null;
}
